"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.unionOf = exports.diffKeyedEntities = exports.deepEqual = void 0;
/**
 * Compares two objects for equality, deeply. The function handles arguments that are
 * +null+, +undefined+, arrays and objects. For objects, the function will not take the
 * object prototype into account for the purpose of the comparison, only the values of
 * properties reported by +Object.keys+.
 *
 * If both operands can be parsed to equivalent numbers, will return true.
 * This makes diff consistent with CloudFormation, where a numeric 10 and a literal "10"
 * are considered equivalent.
 *
 * @param lvalue the left operand of the equality comparison.
 * @param rvalue the right operand of the equality comparison.
 *
 * @returns +true+ if both +lvalue+ and +rvalue+ are equivalent to each other.
 */
function deepEqual(lvalue, rvalue) {
    if (lvalue === rvalue) {
        return true;
    }
    // CloudFormation allows passing strings into boolean-typed fields
    if (((typeof lvalue === 'string' && typeof rvalue === 'boolean') ||
        (typeof lvalue === 'boolean' && typeof rvalue === 'string')) &&
        lvalue.toString() === rvalue.toString()) {
        return true;
    }
    // allows a numeric 10 and a literal "10" to be equivalent;
    // this is consistent with CloudFormation.
    if ((typeof lvalue === 'string' || typeof rvalue === 'string') &&
        safeParseFloat(lvalue) === safeParseFloat(rvalue)) {
        return true;
    }
    if (typeof lvalue !== typeof rvalue) {
        return false;
    }
    if (Array.isArray(lvalue) !== Array.isArray(rvalue)) {
        return false;
    }
    if (Array.isArray(lvalue) /* && Array.isArray(rvalue) */) {
        if (lvalue.length !== rvalue.length) {
            return false;
        }
        for (let i = 0; i < lvalue.length; i++) {
            if (!deepEqual(lvalue[i], rvalue[i])) {
                return false;
            }
        }
        return true;
    }
    if (typeof lvalue === 'object' /* && typeof rvalue === 'object' */) {
        if (lvalue === null || rvalue === null) {
            // If both were null, they'd have been ===
            return false;
        }
        const keys = Object.keys(lvalue);
        if (keys.length !== Object.keys(rvalue).length) {
            return false;
        }
        for (const key of keys) {
            if (!rvalue.hasOwnProperty(key)) {
                return false;
            }
            if (key === 'DependsOn') {
                if (!dependsOnEqual(lvalue[key], rvalue[key])) {
                    return false;
                }
                ;
                // check differences other than `DependsOn`
                continue;
            }
            if (!deepEqual(lvalue[key], rvalue[key])) {
                return false;
            }
        }
        return true;
    }
    // Neither object, nor array: I deduce this is primitive type
    // Primitive type and not ===, so I deduce not deepEqual
    return false;
}
exports.deepEqual = deepEqual;
/**
 * Compares two arguments to DependsOn for equality.
 *
 * @param lvalue the left operand of the equality comparison.
 * @param rvalue the right operand of the equality comparison.
 *
 * @returns +true+ if both +lvalue+ and +rvalue+ are equivalent to each other.
 */
function dependsOnEqual(lvalue, rvalue) {
    // allows ['Value'] and 'Value' to be equal
    if (Array.isArray(lvalue) !== Array.isArray(rvalue)) {
        const array = Array.isArray(lvalue) ? lvalue : rvalue;
        const nonArray = Array.isArray(lvalue) ? rvalue : lvalue;
        if (array.length === 1 && deepEqual(array[0], nonArray)) {
            return true;
        }
        return false;
    }
    // allows arrays passed to DependsOn to be equivalent irrespective of element order
    if (Array.isArray(lvalue) && Array.isArray(rvalue)) {
        if (lvalue.length !== rvalue.length) {
            return false;
        }
        for (let i = 0; i < lvalue.length; i++) {
            for (let j = 0; j < lvalue.length; j++) {
                if ((!deepEqual(lvalue[i], rvalue[j])) && (j === lvalue.length - 1)) {
                    return false;
                }
                break;
            }
        }
        return true;
    }
    return false;
}
/**
 * Produce the differences between two maps, as a map, using a specified diff function.
 *
 * @param oldValue  the old map.
 * @param newValue  the new map.
 * @param elementDiff the diff function.
 *
 * @returns a map representing the differences between +oldValue+ and +newValue+.
 */
function diffKeyedEntities(oldValue, newValue, elementDiff) {
    const result = {};
    for (const logicalId of unionOf(Object.keys(oldValue || {}), Object.keys(newValue || {}))) {
        const oldElement = oldValue && oldValue[logicalId];
        const newElement = newValue && newValue[logicalId];
        result[logicalId] = elementDiff(oldElement, newElement, logicalId);
    }
    return result;
}
exports.diffKeyedEntities = diffKeyedEntities;
/**
 * Computes the union of two sets of strings.
 *
 * @param lv the left set of strings.
 * @param rv the right set of strings.
 *
 * @returns a new array containing all elemebts from +lv+ and +rv+, with no duplicates.
 */
function unionOf(lv, rv) {
    const result = new Set(lv);
    for (const v of rv) {
        result.add(v);
    }
    return new Array(...result);
}
exports.unionOf = unionOf;
/**
 * A parseFloat implementation that does the right thing for
 * strings like '0.0.0'
 * (for which JavaScript's parseFloat() returns 0).
 */
function safeParseFloat(str) {
    const ret = parseFloat(str);
    const nonNumericRegex = /\d*\.\d+\./;
    if (ret === 0) {
        // if the str is exactly '0', that's OK;
        // but parseFloat() also returns 0 for things like '0.0';
        // in this case, return NaN, so we'll fall back to string comparison
        return str === '0' ? ret : NaN;
    }
    else if (nonNumericRegex.test(str)) {
        // if the str contains non-numeric characters,
        // return NaN, so we'll fall back to string comparison
        return NaN;
    }
    else {
        return ret;
    }
}
//# sourceMappingURL=data:application/json;base64,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