#!/usr/bin/env node
"use strict";
/**
 * automatically creates a module for any CloudFormation namespaces that do not
 * have an AWS construct library.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const fs = require("fs-extra");
const cfnspec = require("../lib");
// don't be a prude:
/* eslint-disable no-console */
/* eslint-disable quote-props */
async function main() {
    const root = path.join(__dirname, '..', '..');
    if (path.basename(root) !== '@aws-cdk') {
        throw new Error(`Something went wrong. We expected ${root} to be the "packages/@aws-cdk" directory. Did you move me?`);
    }
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const version = require('../package.json').version;
    // iterate over all cloudformation namespaces
    for (const namespace of cfnspec.namespaces()) {
        const module = cfnspec.createModuleDefinitionFromCfnNamespace(namespace);
        const lowcaseModuleName = module.moduleName.toLocaleLowerCase();
        const packagePath = path.join(root, module.moduleName);
        // we already have a module for this namesapce, move on.
        if (await fs.pathExists(packagePath)) {
            const packageJsonPath = path.join(packagePath, 'package.json');
            // eslint-disable-next-line @typescript-eslint/no-require-imports
            const packageJson = require(packageJsonPath);
            let scopes = packageJson['cdk-build'].cloudformation;
            if (typeof scopes === 'string') {
                scopes = [scopes];
            }
            if (scopes.indexOf(namespace) !== -1) {
                // V2-style module is already modeled in the root package, nothing to be done!
                continue;
            }
            else if (await fs.pathExists(path.join(root, `${module.moduleFamily}-${module.moduleBaseName}`.toLocaleLowerCase()))) {
                // V2-style package already has it's own package (legacy behavior), nothing to be done!
                continue;
            }
            else {
                // V2-style package needs to be added to it's "V1" package... Get down to business!
                console.error(`Adding ${namespace} to ${module.packageName}`);
                scopes.push(namespace);
                packageJson['cdk-build'].cloudformation = scopes;
                await fs.writeJson(packageJsonPath, packageJson, { encoding: 'utf-8', spaces: 2 });
                const indexTsPath = path.join(packagePath, 'lib', 'index.ts');
                const indexTs = [
                    (await fs.readFile(indexTsPath, { encoding: 'utf8' })).trimRight(),
                    `// ${namespace} CloudFormation Resources:`,
                    `export * from './${lowcaseModuleName}.generated';`,
                ].join('\n');
                await fs.writeFile(indexTsPath, indexTs, { encoding: 'utf8' });
                continue;
            }
        }
        async function write(relativePath, contents) {
            const fullPath = path.join(packagePath, relativePath);
            const dir = path.dirname(fullPath);
            await fs.mkdirp(dir);
            let data;
            if (typeof contents === 'string') {
                data = contents.trimLeft(); // trim first newline
            }
            else if (Array.isArray(contents)) {
                data = contents.join('\n');
            }
            else if (typeof contents === 'object') {
                data = JSON.stringify(contents, undefined, 2);
            }
            else {
                throw new Error('Invalid type of contents: ' + contents);
            }
            await fs.writeFile(fullPath, data + '\n');
        }
        console.log(`generating module for ${module.packageName}...`);
        await write('package.json', {
            name: module.packageName,
            version,
            description: `The CDK Construct Library for ${namespace}`,
            main: 'lib/index.js',
            types: 'lib/index.d.ts',
            jsii: {
                outdir: 'dist',
                projectReferences: true,
                targets: {
                    dotnet: {
                        namespace: module.dotnetPackage,
                        packageId: module.dotnetPackage,
                        signAssembly: true,
                        assemblyOriginatorKeyFile: '../../key.snk',
                        iconUrl: 'https://raw.githubusercontent.com/aws/aws-cdk/master/logo/default-256-dark.png',
                    },
                    java: {
                        package: `${module.javaGroupId}.${module.javaPackage}`,
                        maven: {
                            groupId: module.javaGroupId,
                            artifactId: module.javaArtifactId,
                        },
                    },
                    python: {
                        classifiers: [
                            'Framework :: AWS CDK',
                            'Framework :: AWS CDK :: 1',
                        ],
                        distName: module.pythonDistName,
                        module: module.pythonModuleName,
                    },
                },
            },
            repository: {
                type: 'git',
                url: 'https://github.com/aws/aws-cdk.git',
                directory: `packages/${module.packageName}`,
            },
            homepage: 'https://github.com/aws/aws-cdk',
            scripts: {
                build: 'cdk-build',
                watch: 'cdk-watch',
                lint: 'cdk-lint',
                test: 'cdk-test',
                integ: 'cdk-integ',
                pkglint: 'pkglint -f',
                package: 'cdk-package',
                awslint: 'cdk-awslint',
                cfn2ts: 'cfn2ts',
                'build+test': 'yarn build && yarn test',
                'build+test+package': 'yarn build+test && yarn package',
                compat: 'cdk-compat',
                gen: 'cfn2ts',
                'rosetta:extract': 'yarn --silent jsii-rosetta extract',
                'build+extract': 'yarn build && yarn rosetta:extract',
                'build+test+extract': 'yarn build+test && yarn rosetta:extract',
            },
            'cdk-build': {
                cloudformation: namespace,
                jest: true,
                env: {
                    AWSLINT_BASE_CONSTRUCT: 'true',
                },
            },
            keywords: [
                'aws',
                'cdk',
                'constructs',
                namespace,
                module.moduleName,
            ],
            author: {
                name: 'Amazon Web Services',
                url: 'https://aws.amazon.com',
                organization: true,
            },
            license: 'Apache-2.0',
            devDependencies: {
                '@aws-cdk/assertions': version,
                '@aws-cdk/cdk-build-tools': version,
                '@aws-cdk/cfn2ts': version,
                '@aws-cdk/pkglint': version,
                '@types/jest': '^26.0.22',
            },
            dependencies: {
                '@aws-cdk/core': version,
            },
            peerDependencies: {
                '@aws-cdk/core': version,
            },
            engines: {
                node: '>= 10.13.0 <13 || >=13.7.0',
            },
            stability: 'experimental',
            maturity: 'cfn-only',
            awscdkio: {
                announce: false,
            },
            publishConfig: {
                tag: 'latest',
            },
        });
        await write('.gitignore', [
            '*.js',
            '*.js.map',
            '*.d.ts',
            'tsconfig.json',
            'node_modules',
            '*.generated.ts',
            'dist',
            '.jsii',
            '',
            '.LAST_BUILD',
            '.nyc_output',
            'coverage',
            '.nycrc',
            '.LAST_PACKAGE',
            '*.snk',
            'nyc.config.js',
            '!.eslintrc.js',
            '!jest.config.js',
            'junit.xml',
        ]);
        await write('.npmignore', [
            '# Don\'t include original .ts files when doing `npm pack`',
            '*.ts',
            '!*.d.ts',
            'coverage',
            '.nyc_output',
            '*.tgz',
            '',
            'dist',
            '.LAST_PACKAGE',
            '.LAST_BUILD',
            '!*.js',
            '',
            '# Include .jsii',
            '!.jsii',
            '',
            '*.snk',
            '',
            '*.tsbuildinfo',
            '',
            'tsconfig.json',
            '',
            '.eslintrc.js',
            'jest.config.js',
            '',
            '# exclude cdk artifacts',
            '**/cdk.out',
            'junit.xml',
            'test/',
            '!*.lit.ts',
        ]);
        await write('lib/index.ts', [
            `// ${namespace} CloudFormation Resources:`,
            `export * from './${lowcaseModuleName}.generated';`,
        ]);
        await write(`test/${lowcaseModuleName}.test.ts`, [
            "import '@aws-cdk/assertions';",
            "import {} from '../lib';",
            '',
            "test('No tests are specified for this package', () => {",
            '  expect(true).toBe(true);',
            '});',
        ]);
        await cfnspec.createLibraryReadme(namespace, path.join(packagePath, 'README.md'));
        await write('.eslintrc.js', [
            "const baseConfig = require('@aws-cdk/cdk-build-tools/config/eslintrc');",
            "baseConfig.parserOptions.project = __dirname + '/tsconfig.json';",
            'module.exports = baseConfig;',
        ]);
        await write('jest.config.js', [
            "const baseConfig = require('@aws-cdk/cdk-build-tools/config/jest.config');",
            'module.exports = baseConfig;',
        ]);
        const templateDir = path.join(__dirname, 'template');
        for (const file of await fs.readdir(templateDir)) {
            await fs.copy(path.join(templateDir, file), path.join(packagePath, file));
        }
        await addDependencyToMegaPackage(path.join('@aws-cdk', 'cloudformation-include'), module.packageName, version, ['dependencies', 'peerDependencies']);
        await addDependencyToMegaPackage('aws-cdk-lib', module.packageName, version, ['devDependencies']);
        await addDependencyToMegaPackage('monocdk', module.packageName, version, ['devDependencies']);
        await addDependencyToMegaPackage('decdk', module.packageName, version, ['dependencies']);
    }
}
/**
 * A few of our packages (e.g., decdk, aws-cdk-lib) require a dependency on every service package.
 * This automates adding the dependency (and peer dependency) to the package.json.
 */
async function addDependencyToMegaPackage(megaPackageName, packageName, version, dependencyTypes) {
    const packageJsonPath = path.join(__dirname, '..', '..', '..', megaPackageName, 'package.json');
    const packageJson = JSON.parse(await fs.readFile(packageJsonPath, 'utf8'));
    dependencyTypes.forEach(dependencyType => {
        const unorderedDeps = {
            ...packageJson[dependencyType],
            [packageName]: version,
        };
        packageJson[dependencyType] = {};
        Object.keys(unorderedDeps).sort().forEach(k => {
            packageJson[dependencyType][k] = unorderedDeps[k];
        });
    });
    await fs.writeFile(packageJsonPath, JSON.stringify(packageJson, null, 2) + '\n');
}
main().catch(e => {
    console.error(e);
    process.exit(1);
});
//# sourceMappingURL=data:application/json;base64,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