"use strict";
/*
 * Invoked as part of the "build" script of this package,
 * this script takes all specification fragments in the
 * `spec-source` folder and generates a unified specification
 * document at `spec/specification.json`.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const fs = require("fs-extra");
const md5 = require("md5");
const massage_spec_1 = require("./massage-spec");
async function main() {
    const inputDir = path.join(process.cwd(), 'spec-source');
    const outDir = path.join(process.cwd(), 'spec');
    await generateResourceSpecification(inputDir, path.join(outDir, 'specification.json'));
    await mergeSpecificationFromDirs(path.join(inputDir, 'cfn-lint'), path.join(outDir, 'cfn-lint.json'));
}
/**
 * Generate CloudFormation resource specification from sources and patches
 */
async function generateResourceSpecification(inputDir, outFile) {
    const spec = { PropertyTypes: {}, ResourceTypes: {}, Fingerprint: '' };
    const files = await fs.readdir(inputDir);
    for (const file of files.filter(n => n.endsWith('.json')).sort()) {
        const data = await fs.readJson(path.join(inputDir, file));
        if (file.indexOf('patch') === -1) {
            massage_spec_1.decorateResourceTypes(data);
            massage_spec_1.forEachSection(spec, data, massage_spec_1.merge);
        }
        else {
            massage_spec_1.forEachSection(spec, data, massage_spec_1.patch);
        }
    }
    massage_spec_1.massageSpec(spec);
    spec.Fingerprint = md5(JSON.stringify(massage_spec_1.normalize(spec)));
    await fs.mkdirp(path.dirname(outFile));
    await fs.writeJson(outFile, spec, { spaces: 2 });
}
/**
 * Generate Cfnlint spec annotations from sources and patches
 */
async function mergeSpecificationFromDirs(inputDir, outFile) {
    const spec = {};
    for (const child of await fs.readdir(inputDir)) {
        const fullPath = path.join(inputDir, child);
        if (!(await fs.stat(fullPath)).isDirectory()) {
            continue;
        }
        const subspec = await loadMergedSpec(fullPath);
        spec[child] = subspec;
    }
    await fs.mkdirp(path.dirname(outFile));
    await fs.writeJson(outFile, spec, { spaces: 2 });
}
/**
 * Load all files in the given directory, merge them and apply patches in the order found
 *
 * The base structure is always an empty object
 */
async function loadMergedSpec(inputDir) {
    const structure = {};
    const files = await fs.readdir(inputDir);
    for (const file of files.filter(n => n.endsWith('.json')).sort()) {
        const data = await fs.readJson(path.join(inputDir, file));
        if (file.indexOf('patch') === -1) {
            // Copy properties from current object into structure, adding/overwriting whatever is found
            Object.assign(structure, data);
        }
        else {
            // Apply the loaded file as a patch onto the current structure
            massage_spec_1.patch(structure, data);
        }
    }
    return structure;
}
main()
    .catch(e => {
    // eslint-disable-next-line no-console
    console.error(e.stack);
    process.exit(-1);
});
//# sourceMappingURL=data:application/json;base64,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