"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SynthUtils = void 0;
const fs = require("fs");
const path = require("path");
const core = require("monocdk");
class SynthUtils {
    /**
     * Returns the cloud assembly template artifact for a stack.
     */
    static synthesize(stack, options = {}) {
        // always synthesize against the root (be it an App or whatever) so all artifacts will be included
        const assembly = synthesizeApp(stack, options);
        return assembly.getStackArtifact(stack.artifactId);
    }
    /**
     * Synthesizes the stack and returns the resulting CloudFormation template.
     */
    static toCloudFormation(stack, options = {}) {
        const synth = this._synthesizeWithNested(stack, options);
        if (isStackArtifact(synth)) {
            return synth.template;
        }
        else {
            return synth;
        }
    }
    /**
     * @returns Returns a subset of the synthesized CloudFormation template (only specific resource types).
     */
    static subset(stack, options) {
        const template = this.toCloudFormation(stack);
        if (template.Resources) {
            for (const [key, resource] of Object.entries(template.Resources)) {
                if (options.resourceTypes && !options.resourceTypes.includes(resource.Type)) {
                    delete template.Resources[key];
                }
            }
        }
        return template;
    }
    /**
     * Synthesizes the stack and returns a `CloudFormationStackArtifact` which can be inspected.
     * Supports nested stacks as well as normal stacks.
     *
     * @return CloudFormationStackArtifact for normal stacks or the actual template for nested stacks
     * @internal
     */
    static _synthesizeWithNested(stack, options = {}) {
        // always synthesize against the root (be it an App or whatever) so all artifacts will be included
        const assembly = synthesizeApp(stack, options);
        // if this is a nested stack (it has a parent), then just read the template as a string
        if (stack.nestedStackParent) {
            return JSON.parse(fs.readFileSync(path.join(assembly.directory, stack.templateFile)).toString('utf-8'));
        }
        return assembly.getStackArtifact(stack.artifactId);
    }
}
exports.SynthUtils = SynthUtils;
/**
 * Synthesizes the app in which a stack resides and returns the cloud assembly object.
 */
function synthesizeApp(stack, options) {
    const root = stack.node.root;
    if (!core.Stage.isStage(root)) {
        throw new Error('unexpected: all stacks must be part of a Stage or an App');
    }
    // to support incremental assertions (i.e. "expect(stack).toNotContainSomething(); doSomething(); expect(stack).toContainSomthing()")
    const force = true;
    return root.synth({
        force,
        ...options,
    });
}
function isStackArtifact(x) {
    return 'template' in x;
}
//# sourceMappingURL=data:application/json;base64,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