"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StackPathInspector = exports.StackInspector = exports.Inspector = void 0;
const cxschema = require("monocdk/cloud-assembly-schema");
const api = require("monocdk/cx-api");
const assertion_1 = require("./assertion");
const match_template_1 = require("./assertions/match-template");
class Inspector {
    constructor() {
        this.aroundAssert = undefined;
    }
    to(assertion) {
        return this.aroundAssert ? this.aroundAssert(() => this._to(assertion))
            : this._to(assertion);
    }
    notTo(assertion) {
        return this.to(assertion_1.not(assertion));
    }
    _to(assertion) {
        assertion.assertOrThrow(this);
    }
}
exports.Inspector = Inspector;
class StackInspector extends Inspector {
    constructor(stack) {
        super();
        this.stack = stack;
        this.template = stack instanceof api.CloudFormationStackArtifact ? stack.template : stack;
    }
    at(path) {
        if (!(this.stack instanceof api.CloudFormationStackArtifact)) {
            throw new Error('Cannot use "expect(stack).at(path)" for a raw template, only CloudFormationStackArtifact');
        }
        const strPath = typeof path === 'string' ? path : path.join('/');
        return new StackPathInspector(this.stack, strPath);
    }
    toMatch(template, matchStyle = match_template_1.MatchStyle.EXACT) {
        return this.to(match_template_1.matchTemplate(template, matchStyle));
    }
    get value() {
        return this.template;
    }
}
exports.StackInspector = StackInspector;
class StackPathInspector extends Inspector {
    constructor(stack, path) {
        super();
        this.stack = stack;
        this.path = path;
    }
    get value() {
        // The names of paths in metadata in tests are very ill-defined. Try with the full path first,
        // then try with the stack name preprended for backwards compat with most tests that happen to give
        // their stack an ID that's the same as the stack name.
        const metadata = this.stack.manifest.metadata || {};
        const md = metadata[this.path] || metadata[`/${this.stack.id}${this.path}`];
        if (md === undefined) {
            return undefined;
        }
        const resourceMd = md.find(entry => entry.type === cxschema.ArtifactMetadataEntryType.LOGICAL_ID);
        if (resourceMd === undefined) {
            return undefined;
        }
        const logicalId = resourceMd.data;
        return this.stack.template.Resources[logicalId];
    }
}
exports.StackPathInspector = StackPathInspector;
//# sourceMappingURL=data:application/json;base64,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