# This script is typically executed via Snakefile > Rscript
# It is mainly just a call to rmarkdown::render with 2 accomodations:
#   1. Import of knitmeta from exe=>md step prior to rendering
#   2. Adjustment of homepage link to direct to the project index.html
# Inputs: 
# 	Markdown file location 
#	Output directory
#	Location of projects index.html
# Output: HTML file
# Notes:
# Markdown file should have access to all referenced content (i.e. figures)
# A _site.yml file will be used by rmarkdown::render to generate
# the navbar of the website.

render_minimal <- function(input,outfile,index_html){
    outdir = dirname(outfile)    

    # Read the knit_meta() output generated by execute_code.R
    # which is a must for interactive plots (js libraries)
    knitmeta_filename = sub(input, pattern = "_tmp.md$",
        replacement = ".knitmeta.RDS")
    knitmeta = list()
    if(file.exists(knitmeta_filename)){
        knitmeta = readRDS(knitmeta_filename)
    }
    
    # Use of no extension allows for arbitrary outputs
    outfile_base = tools::file_path_sans_ext(outfile)
    # generate the html file
    rmarkdown::render(input,
        output_dir = outdir,
        output_file = outfile_base, 
        knit_meta = knitmeta
        )

    # Modify new html directly to point to top level out_html/index.html file
    # Alternative to this would be to include a copy of index.html in each
    # subdirectory
    if(file.exists(outfile)){
        html_text = readLines(outfile)
        out = gsub(pattern="(navbar-brand.*)index.html",
            replacement=paste0("\\1", index_html),
                x=html_text)
        writeLines(text=out, con=outfile)
    }else{
        warning(paste0(outfile," does not exist"))   
    }
}

# Allows this file to be used with source() or Rscript
if(!interactive()){
    # loading snakefile variables
    args = commandArgs(trailingOnly=TRUE)
    input = args[1]
    outfile = args[2]
    index_html = args[3]
    render_minimal(input,outfile,index_html)
}
