# This script is typically executed via the Snakefile
# Inputs: 
# 	Markdown file location 
#	Output directory
#	Location of internal scikick templates
#	Location of projects index.html
# Output: HTML file
# Notes:
# Markdown file should have access to all refenreced content 
# A _site.yml file will be used by rmarkdown::render to generate
# the navbar of the website.

# loading snakefile variables
args = commandArgs(trailingOnly=TRUE)

input = args[1]
outdir = dirname(args[2])
templatedir = args[3]
index_html = args[4]

# Read the knit_meta() output generated by knit.R
# which is a must for interactive plots (js libraries)
knitmeta_filename = sub(input, pattern = ".md$",
    replacement = ".knitmeta.RDS")
knitmeta = list()
if(file.exists(knitmeta_filename)){
    knitmeta = readRDS(knitmeta_filename)
}

# Setting default scikick options
render_options <- bookdown::html_document2(
    code_folding = "hide",
    highlight = "zenburn",
    theme = "journal",
    template = "default",
    toc = TRUE,
    number_sections = FALSE,
    css = "template/custom.css",
    # This would need some modifications in pandoc args to resolve the path
    # includes = rmarkdown::includes(before_body = "template/my.html"),
    toc_depth = 5,
    self_contained = TRUE
)

# expand css path
cssind  <- which(render_options$pandoc$args=="--css")+1
csspath <- render_options$pandoc$args[cssind]
# remove the redundant 'template/' prefix
csspath = sub(x = csspath, pattern = '^template/', replacement = '')
csspath <- file.path(templatedir, csspath)
render_options$pandoc$args[cssind] = csspath
# add path for regular markdown figures
render_options$pandoc$args =
	c(render_options$pandoc$args,
		paste0("--resource-path=", getwd(), ":",
			file.path(getwd(), dirname(input))))
# custom title for each html doc
render_options$pandoc$args =
	c(render_options$pandoc$args,
		paste0("--metadata=pagetitle:",
        	sub("\\.md$", "", basename(input))))
# generate the html file
suppressWarnings(suppressMessages({
rmarkdown::render(input,
	output_format = render_options,
	output_dir = outdir,
	output_file = file.path(outdir,
		sub(x=basename(input), pattern=".md$", replacement=".html")),
	quiet = TRUE, knit_meta = knitmeta)
}))

# Modify html
html_outfile = sub(pattern = ".md$", replacement = ".html",
	file.path(outdir, basename(input)))
html_text = readLines(html_outfile)

writeLines(text=gsub(pattern="(navbar-brand.*)index.html",
    replacement=paste0("\\1", index_html),
        x=html_text), con=html_outfile)
