# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pybet', 'pybet.staking']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'pybet',
    'version': '0.4.0',
    'description': 'A library of betting utilities to assist with calculation of bets, stakes and markets',
    'long_description': '# pybet\n\npybet is a library of betting utilities to assist with calculation of bets, stakes and markets\n\n## Installation\n\n`pip install pybet`\n\n## Usage\n\nThis initial release contains an Odds class which enables conversion from and to the following types of odds or odds-equivalent:\n\n- Decimal odds, e.g. 2.5\n- Fractional odds, e.g. 6/4\n- American moneyline odds, e.g. +150\n- Implied percentage, e.g. 40\n- Implied probability, e.g. 0.4\n\nand a Market class which allows for the creation of markets based on those odds and some limited calculations on those markets\n\nFor a basic guide to odds, please see [https://www.investopedia.com/articles/investing/042115/betting-basics-fractional-decimal-american-moneyline-odds.asp]\n\n### Odds\n\nInternally, the value of an Odds instance is stored as a Decimal, making decimal odds the effective default.\nOdds can be instantiated directly as decimals or via a class method, specifying the type of odds being instantiated from.\nAny Odds instance can then be output to any type with a to\\_{type} method, e.g.\n\n```\nfrom pybet import Odds\n\no = Odds(2.5)\no.to_moneyline()    # +150\no.to_percentage()   # 40\n\no = Odds.fractional(6, 4)\no.to_probability()  # 0.4\n```\n\nThe to_fractional() method requires a set of fractional odds to work from. It will then select the closest matching value\nfrom that set. For convenience, a set of standard odds, representing those most typically found in the UK, has been provided.\n\n```\no.to_fractional(\'5/4\', \'6/4\', \'7/4\', \'2/1\') # 6/4\no.to_fractional(Odds.STANDARD_FRACTIONALS) # 6/4\n```\n\nComparisons can be made between Odds instances. It is possible to check if one Odds instance is shorter (<)\nor longer (>) than another, e.g.\n\n```\nOdds(2.5) < Odds.fractional(11, 4)        # True\nOdds.probability(50) > Odds(3)            # False\n```\n\nThere are properties to compare Odds instances to evens, and a convenience method for evens itself, e.g.\n\n```\no.is_odds_against   # True\no.is_odds_on        # True\nOdds.evens() == 2   # True\n```\n\nIt is possible to invert odds, to turn odds against into the equivalent odds on value and vice versa, e.g.\n\n```\nOdds.inverted(2.5) == Odds.fractional(4, 6)  # True\n```\n\nIt is also possible to get the odds "to one" i.e. the numerator of fractional odds, but with a decimal numerator if that is applicable,\nor, put another way, the fractional odds with the stake removed.\n\n```\nOdds.fractional(5, 1).to_one() == 5     # True\nOdds.fractional(9, 4).to_one() == 2.25  # True\nOdds(3.25).to_one() == 2.25             # True\n```\n\nThere are also operators to perform calculations with Odds instances, though this is more useful in some cases than others.\nFor example, it is possible to calculate the combined odds of two 3/1 shots as follows:\n\n```\nOdds.fractional(3, 1) + Odds.fractional(3, 1) == Odds.evens  # True\n```\n\n### Market\n\nA Market is a dictionary of "runners" (which can be of any type) and Odds. A market also has a places attribute. The\ndefault for this is 1 (i.e. a win market), but it can be set to any value.\n\nA Market can be instantiated any way a python dictionary can. Given a list of runners and odds a market can be created like this:\n\n```\nrunners = [\'Frankel\', \'Sea The Stars\', \'Brigadier Gerard\', \'Dancing Brave\', \'Quixall Crossett\']\nodds = [Odds(x) for x in [2, 4, 5, 10, 1000]]\nmarket = Market(zip(runners, odds))\n```\n\nAlternatively, the market could be created runner by runner...\n\n```\nmarket = Market()\nmarket[\'Frankel\'] = Odds(2)\n```\n\nYou may also wish to create an "empty" market, to assign odds later:\n\n```\nmarket = Market.fromkeys(runners)\n```\n\nMarkets have a number of properties:\n\n- `favourites` - a list of the shortest price runners in the market (NB: It will always be a list, even if there is only one)\n- `percentage` - the sum of every runner\'s implied percentage chance\n- `overround_per_runner` - the above, divided by the number of runners\n- `is_overbroke` - true if the market is in the punter\'s favour, i.e. < 100% book, false otherwise\n- `is_overround` - true if the market is in the bookie\'s favour, i.e. > 100% book, false otherwise\n- `is_fair` - only true if the book is at exactly 100%\n\nThey also have a number of methods. The following market is used in the explanation of them:\n\n```\nmarket = Market({\'Frankel\': 2, \'Sea the Stars\': 3, \'Brigadier Gerard\': 6})\n```\n\n#### `apply_margin`\n\nAllows the user to manipulate the overround on a market. For example, in the \'fair\' market given above, applying a margin of 20% as follows:\n\n```\nmarket.apply_margin(20)\n```\n\nwill change the odds in the following way:\n\n```\nmarket.get(\'Frankel\')           # 1.667 (to 3 dp)\nmarket.get(\'Sea The Stars\')     # 2.5\nmarket.get(\'Brigadier Gerard\')  # 5\nmarket.percentage               # 120\n```\n\nNote that the method applies the margin in proportion to each runner\'s current odds.\n\n#### `equalise`\n\nResets the market to a fair market where all runners have the same odds.\n\n```\nmarket.equalise()\nmarket.get(\'Frankel\')           # 3\nmarket.get(\'Sea The Stars\')     # 3\nmarket.get(\'Brigadier Gerard\')  # 3\nmarket.percentage               # 100\n```\n\n#### `fill`\n\nFills out any missing odds in the market to the specified margin.\n\n```\nmarket[\'Frankel\'] = None\nmarket.fill(10)\nmarket.get(\'Frankel\')           # 1.667 (to 3 dp)\n```\n\nThat is, the odds of Sea The Stars (3) and Brigadier Gerard (6) represent a 50% market. To fill out the entire market to a 10% margin requires Frankel\'s odds to be 60% or 1.667. If there were three unpriced runners, they\'d all be set to 20% or 5.\n\nWhere no margin is specified, a 100% market is assumed.\n\n```\nmarket[\'Frankel\'] = None\nmarket.fill()\nmarket.get(\'Frankel\')           # 2\n```\n\n#### `wipe`\n\nClears the market, setting all odds to none.\n\n```\nmarket.wipe()\nmarket.get(\'Frankel\')           # None\n```\n\n#### `without`\n\nAllows the user to extract runners from markets. In its current state, it is of little practical use, as it just\nextracts the runners, normally leaving an overbroke market. In future releases, this will be enhanced to automatically recalculate.\n\n```\nmarket = market.without([\'Frankel\'])\nmarket.favourites == [\'Sea The Stars\']  # True\n```\n\n### staking\n\nThe `staking` module contains methods for calculating stakes for a given set of odds and bank size.\n\n#### `kelly`\n\nThis method calculates the correct stake according to the [Kelly Criterion](https://www.investopedia.com/articles/investing/042115/betting-basics-fractional-decimal-american-moneyline-odds.asp) for a given bank size. If the odds are in the bettor\'s favour, this will be positive.\nIf they aren\'t the method will return zero.\n\n```\nkelly(Odds(4), Odds(5), 100)   # 6.25\nkelly(Odds(5), Odds(4), 100)   # 0\n```\n',
    'author': 'Robert Peacock',
    'author_email': 'robertjamespeacock@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/peaky76/pybet',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.9.0,<4.0.0',
}


setup(**setup_kwargs)
