# -*- coding: utf-8 -*-
# Copyright 2022 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
import abc
from typing import Awaitable, Callable, Dict, Optional, Sequence, Union

import google.api_core
from google.api_core import exceptions as core_exceptions
from google.api_core import gapic_v1, operations_v1
from google.api_core import retry as retries
import google.auth  # type: ignore
from google.auth import credentials as ga_credentials  # type: ignore
from google.longrunning import operations_pb2  # type: ignore
from google.oauth2 import service_account  # type: ignore
import pkg_resources

from google.cloud.tpu_v1.types import cloud_tpu

try:
    DEFAULT_CLIENT_INFO = gapic_v1.client_info.ClientInfo(
        gapic_version=pkg_resources.get_distribution(
            "google-cloud-tpu",
        ).version,
    )
except pkg_resources.DistributionNotFound:
    DEFAULT_CLIENT_INFO = gapic_v1.client_info.ClientInfo()


class TpuTransport(abc.ABC):
    """Abstract transport class for Tpu."""

    AUTH_SCOPES = ("https://www.googleapis.com/auth/cloud-platform",)

    DEFAULT_HOST: str = "tpu.googleapis.com"

    def __init__(
        self,
        *,
        host: str = DEFAULT_HOST,
        credentials: ga_credentials.Credentials = None,
        credentials_file: Optional[str] = None,
        scopes: Optional[Sequence[str]] = None,
        quota_project_id: Optional[str] = None,
        client_info: gapic_v1.client_info.ClientInfo = DEFAULT_CLIENT_INFO,
        always_use_jwt_access: Optional[bool] = False,
        api_audience: Optional[str] = None,
        **kwargs,
    ) -> None:
        """Instantiate the transport.

        Args:
            host (Optional[str]):
                 The hostname to connect to.
            credentials (Optional[google.auth.credentials.Credentials]): The
                authorization credentials to attach to requests. These
                credentials identify the application to the service; if none
                are specified, the client will attempt to ascertain the
                credentials from the environment.
            credentials_file (Optional[str]): A file with credentials that can
                be loaded with :func:`google.auth.load_credentials_from_file`.
                This argument is mutually exclusive with credentials.
            scopes (Optional[Sequence[str]]): A list of scopes.
            quota_project_id (Optional[str]): An optional project to use for billing
                and quota.
            client_info (google.api_core.gapic_v1.client_info.ClientInfo):
                The client info used to send a user-agent string along with
                API requests. If ``None``, then default info will be used.
                Generally, you only need to set this if you're developing
                your own client library.
            always_use_jwt_access (Optional[bool]): Whether self signed JWT should
                be used for service account credentials.
        """

        scopes_kwargs = {"scopes": scopes, "default_scopes": self.AUTH_SCOPES}

        # Save the scopes.
        self._scopes = scopes

        # If no credentials are provided, then determine the appropriate
        # defaults.
        if credentials and credentials_file:
            raise core_exceptions.DuplicateCredentialArgs(
                "'credentials_file' and 'credentials' are mutually exclusive"
            )

        if credentials_file is not None:
            credentials, _ = google.auth.load_credentials_from_file(
                credentials_file, **scopes_kwargs, quota_project_id=quota_project_id
            )
        elif credentials is None:
            credentials, _ = google.auth.default(
                **scopes_kwargs, quota_project_id=quota_project_id
            )
            # Don't apply audience if the credentials file passed from user.
            if hasattr(credentials, "with_gdch_audience"):
                credentials = credentials.with_gdch_audience(
                    api_audience if api_audience else host
                )

        # If the credentials are service account credentials, then always try to use self signed JWT.
        if (
            always_use_jwt_access
            and isinstance(credentials, service_account.Credentials)
            and hasattr(service_account.Credentials, "with_always_use_jwt_access")
        ):
            credentials = credentials.with_always_use_jwt_access(True)

        # Save the credentials.
        self._credentials = credentials

        # Save the hostname. Default to port 443 (HTTPS) if none is specified.
        if ":" not in host:
            host += ":443"
        self._host = host

    def _prep_wrapped_messages(self, client_info):
        # Precompute the wrapped methods.
        self._wrapped_methods = {
            self.list_nodes: gapic_v1.method.wrap_method(
                self.list_nodes,
                default_timeout=None,
                client_info=client_info,
            ),
            self.get_node: gapic_v1.method.wrap_method(
                self.get_node,
                default_timeout=None,
                client_info=client_info,
            ),
            self.create_node: gapic_v1.method.wrap_method(
                self.create_node,
                default_timeout=None,
                client_info=client_info,
            ),
            self.delete_node: gapic_v1.method.wrap_method(
                self.delete_node,
                default_timeout=None,
                client_info=client_info,
            ),
            self.reimage_node: gapic_v1.method.wrap_method(
                self.reimage_node,
                default_timeout=None,
                client_info=client_info,
            ),
            self.stop_node: gapic_v1.method.wrap_method(
                self.stop_node,
                default_timeout=None,
                client_info=client_info,
            ),
            self.start_node: gapic_v1.method.wrap_method(
                self.start_node,
                default_timeout=None,
                client_info=client_info,
            ),
            self.list_tensor_flow_versions: gapic_v1.method.wrap_method(
                self.list_tensor_flow_versions,
                default_timeout=None,
                client_info=client_info,
            ),
            self.get_tensor_flow_version: gapic_v1.method.wrap_method(
                self.get_tensor_flow_version,
                default_timeout=None,
                client_info=client_info,
            ),
            self.list_accelerator_types: gapic_v1.method.wrap_method(
                self.list_accelerator_types,
                default_timeout=None,
                client_info=client_info,
            ),
            self.get_accelerator_type: gapic_v1.method.wrap_method(
                self.get_accelerator_type,
                default_timeout=None,
                client_info=client_info,
            ),
        }

    def close(self):
        """Closes resources associated with the transport.

        .. warning::
             Only call this method if the transport is NOT shared
             with other clients - this may cause errors in other clients!
        """
        raise NotImplementedError()

    @property
    def operations_client(self):
        """Return the client designed to process long-running operations."""
        raise NotImplementedError()

    @property
    def list_nodes(
        self,
    ) -> Callable[
        [cloud_tpu.ListNodesRequest],
        Union[cloud_tpu.ListNodesResponse, Awaitable[cloud_tpu.ListNodesResponse]],
    ]:
        raise NotImplementedError()

    @property
    def get_node(
        self,
    ) -> Callable[
        [cloud_tpu.GetNodeRequest], Union[cloud_tpu.Node, Awaitable[cloud_tpu.Node]]
    ]:
        raise NotImplementedError()

    @property
    def create_node(
        self,
    ) -> Callable[
        [cloud_tpu.CreateNodeRequest],
        Union[operations_pb2.Operation, Awaitable[operations_pb2.Operation]],
    ]:
        raise NotImplementedError()

    @property
    def delete_node(
        self,
    ) -> Callable[
        [cloud_tpu.DeleteNodeRequest],
        Union[operations_pb2.Operation, Awaitable[operations_pb2.Operation]],
    ]:
        raise NotImplementedError()

    @property
    def reimage_node(
        self,
    ) -> Callable[
        [cloud_tpu.ReimageNodeRequest],
        Union[operations_pb2.Operation, Awaitable[operations_pb2.Operation]],
    ]:
        raise NotImplementedError()

    @property
    def stop_node(
        self,
    ) -> Callable[
        [cloud_tpu.StopNodeRequest],
        Union[operations_pb2.Operation, Awaitable[operations_pb2.Operation]],
    ]:
        raise NotImplementedError()

    @property
    def start_node(
        self,
    ) -> Callable[
        [cloud_tpu.StartNodeRequest],
        Union[operations_pb2.Operation, Awaitable[operations_pb2.Operation]],
    ]:
        raise NotImplementedError()

    @property
    def list_tensor_flow_versions(
        self,
    ) -> Callable[
        [cloud_tpu.ListTensorFlowVersionsRequest],
        Union[
            cloud_tpu.ListTensorFlowVersionsResponse,
            Awaitable[cloud_tpu.ListTensorFlowVersionsResponse],
        ],
    ]:
        raise NotImplementedError()

    @property
    def get_tensor_flow_version(
        self,
    ) -> Callable[
        [cloud_tpu.GetTensorFlowVersionRequest],
        Union[cloud_tpu.TensorFlowVersion, Awaitable[cloud_tpu.TensorFlowVersion]],
    ]:
        raise NotImplementedError()

    @property
    def list_accelerator_types(
        self,
    ) -> Callable[
        [cloud_tpu.ListAcceleratorTypesRequest],
        Union[
            cloud_tpu.ListAcceleratorTypesResponse,
            Awaitable[cloud_tpu.ListAcceleratorTypesResponse],
        ],
    ]:
        raise NotImplementedError()

    @property
    def get_accelerator_type(
        self,
    ) -> Callable[
        [cloud_tpu.GetAcceleratorTypeRequest],
        Union[cloud_tpu.AcceleratorType, Awaitable[cloud_tpu.AcceleratorType]],
    ]:
        raise NotImplementedError()

    @property
    def kind(self) -> str:
        raise NotImplementedError()


__all__ = ("TpuTransport",)
