import os
import sys
import distutils

import subprocess 
import shutil

#try:
#    import cmake
#except:
#    print("import cmake failed")
#    import cmake

import pkg_resources
# rather than deal with --no-build-isolation flags and other various flags to get a decent CMake build,
# we do not use build and install requirements in pyproject.toml

min_wheel_version = ['0','33','1']
min_numpy_version = ['1','14','0']
#min_cmake_version = ['3','0','0']

min_wheel_version_str = '.'.join(min_wheel_version)
min_numpy_version_str = '.'.join(min_numpy_version)
#min_cmake_version_str = '.'.join(min_cmake_version)

is_error = False
error_msg = ""

try:
    import pathlib2
except ImportError:
    is_error = True
    error_msg = error_msg + """You need the Python 'pathlib2' package!
        install it by running 'pip install pathlib2'""" + "\n"

try:
    import wheel
    wheel_version = pkg_resources.get_distribution("wheel").version.split('.')
    assert wheel_version >= min_wheel_version
except ImportError:
    is_error = True
    error_msg = error_msg + """You need the Python 'wheel' package >= %s!
        Install it by running 'pip install wheel'"""%(min_wheel_version_str,) + "\n"
except AssertionError:
    is_error = True
    import wheel
    wheel_version = pkg_resources.get_distribution("wheel").version.split('.')
    wheel_version_str = '.'.join(wheel_version)
    error_msg = error_msg + """Python 'wheel' package is too old. [%s < %s]!
        Upgrade it by running 'pip install wheel --upgrade'"""%(wheel_version_str,min_wheel_version_str,) + "\n"

try:
    import numpy
    numpy_version = pkg_resources.get_distribution("numpy").version.split('.')
    assert numpy_version >= min_numpy_version
except ImportError:
    is_error = True
    error_msg = error_msg + """You need the Python 'numpy' package >= %s!
        Install it by running 'pip install numpy'"""%(min_numpy_version_str,) + "\n"
except AssertionError:
    is_error = True
    import numpy
    numpy_version = pkg_resources.get_distribution("numpy").version.split('.')
    numpy_version_str = '.'.join(numpy_version)
    error_msg = error_msg + """Python 'numpy' package is too old. [%s < %s]!
        Upgrade it by running 'pip install numpy --upgrade'"""%(numpy_version_str,min_numpy_version_str,) + "\n"

#try:
#    import cmake
#    cmake_version = pkg_resources.get_distribution("cmake").version.split('.')
#    assert cmake_version >= min_cmake_version
#except ImportError:
#    is_error = True
#    error_msg = error_msg + """You need the Python 'cmake' package >= %s!
#        install it by running 'pip install cmake'"""%(min_cmake_version_str,) + "\n"
#except AssertionError:
#    is_error = True
#    import cmake
#    cmake_version = pkg_resources.get_distribution("cmake").version.split('.')
#    cmake_version_str = '.'.join(cmake_version)
#    error_msg = error_msg + """Python 'cmake' package is too old. [%s < %s]!
#        Upgrade it by running 'pip install cmake --upgrade'"""%(cmake_version_str,min_cmake_version_str,) + "\n"

if is_error:
    sys.exit(error_msg)

from setuptools import setup, Extension
from setuptools.command.build_ext import build_ext as build_ext_orig
from setuptools.command.install import install

#inspired by pieces from
#https://stackoverflow.com/questions/42585210/extending-setuptools-extension-to-use-cmake-in-setup-py
#https://martinopilia.com/posts/2018/09/15/building-python-extension.html
#https://stackoverflow.com/questions/20288711/post-install-script-with-python-setuptools
#https://stackoverflow.com/questions/18725137/how-to-obtain-arguments-passed-to-setup-py-from-pip-with-install-option

def is_osx():
    name = distutils.util.get_platform()
    if sys.platform != 'darwin':
        return False
    elif name.startswith('macosx-10'):
        minor_version = int(name.split('-')[1].split('.')[1])
        if minor_version >= 7:
            return True
        else:
            return False
    else:
        return False

class CMakeExtension(Extension):
    def __init__(self, name, cmake_lists_dir='.', **kwa):
        Extension.__init__(self, name, sources=[], **kwa)
        self.cmake_lists_dir = os.path.abspath(cmake_lists_dir)

class InstallCommand(install):
    user_options = install.user_options + [
        ('cmake-file=', None, 'File declaring CMake variables, separated by new lines.'),
    ]

    def initialize_options(self):
        install.initialize_options(self)
        self.cmake_file = None

    def finalize_options(self):
        print("--cmake-file:", self.cmake_file)
        install.finalize_options(self)

    def run(self):

        global cmake_file

        cmake_file = self.cmake_file

        install.run(self)

class build_ext(build_ext_orig):

    def run(self):
        for ext in self.extensions:
            self.build_cmake(ext)
        build_ext_orig.run(self)


    def build_cmake(self, ext):

        # Accepts user provided compile, link, and python linker specific flags
        COMPILE_OPTIONS = []
        LINK_OPTIONS = []
        PYTHON_SPECIFIC_LINK_OPTIONS = []
        
        if is_osx():
            COMPILE_OPTIONS.append("-stdlib=libc++")
            LINK_OPTIONS.append("-lc++")
            PYTHON_SPECIFIC_LINK_OPTIONS.append("-undefined dynamic_lookup -nodefaultlibs")


        # Get location for where GMLS_Module will be installed
        cwd = pathlib2.Path().absolute()
        build_temp = pathlib2.Path(self.build_temp)
        try:
            build_temp.mkdir(parents=True)
        except:
            pass # already exists
        extdir = pathlib2.Path(self.get_ext_fullpath(ext.name))
        try:
            extdir.mkdir(parents=True)
        except:
            pass # already exists

        # Parse string for kokkos architecture
        cmake_file_string = ""
        try:
            if cmake_file != None:
                cmake_file_string = str(cmake_file)
                print("Custom cmake args file set to: %s"%(cmake_file_string,))
            else:
                print("Custom cmake args file not set.")
        except:
            print("Custom cmake args file not set.")

        # Configure CMake
        config = 'Debug' if self.debug else 'Release'
        cmake_args = [
            '-DGMLS_Module_DEST=' + str(extdir.parent.absolute()),
            '-DCMAKE_INSTALL_PREFIX=' + str(extdir.parent.absolute()),
            '-DCMAKE_CXX_FLAGS=' + " ".join(COMPILE_OPTIONS),
            '-DCMAKE_SHARED_LINKER_FLAGS=' + " ".join(LINK_OPTIONS),
            '-DCMAKE_PYTHON_SHARED_LINKER_FLAGS=' + " ".join(PYTHON_SPECIFIC_LINK_OPTIONS),
            '-DPYTHON_EXECUTABLE=' + str(sys.executable),
            '-DCMAKE_OSX_DEPLOYMENT_TARGET=10.11',
            '-DCompadre_USE_PYTHON:BOOL=ON',
            '-DCompadre_USE_MATLAB:BOOL=ON',
            '-DCompadre_EXAMPLES:BOOL=OFF',
            '-DPYTHON_CALLING_BUILD:BOOL=ON',
        ]

        cmake_file_list = list()
        if (cmake_file_string != ""):
            cmake_arg_list = [line.rstrip('\n') for line in open(cmake_file_string)]
            for arg in cmake_arg_list:
                cmake_args.append('-D'+arg)
            print('Custom CMake Args: ', cmake_arg_list)

        build_args = [
            '--config', config,
            '--', '-j4'
        ]
        os.chdir(str(build_temp))
        self.spawn(['cmake', str(cwd)] + cmake_args)
        if not self.dry_run:
            self.spawn(['cmake', '--build', '.'] + build_args)

        # copy files from this build directory
        python_temp = "./python"
        dest_temp = "../" + extdir.relative_to(*extdir.parts[:1]).as_posix() + "/../compadre"
        libs = [os.path.join(python_temp, _lib) for _lib in 
                os.listdir(python_temp) if 
                os.path.isfile(os.path.join(python_temp, _lib)) and 
                os.path.splitext(_lib)[1] in [".dll", ".so", ".dylib",".py"]
                and not (_lib.startswith("python"))]
        for lib in libs:
            shutil.move(lib, os.path.join(dest_temp,
                                          os.path.basename(lib)))

        # copy files from this build directory
        python_temps = ["./src","./kokkos/core/src","./kokkos/containers/src","./kokkos-kernels/src"]
        dest_temp = "../" + extdir.relative_to(*extdir.parts[:1]).as_posix() + "/../compadre"
        for python_temp in python_temps:

            libs_not_sym = [os.path.join(python_temp, _lib) for _lib in 
                    os.listdir(python_temp) if 
                    (not os.path.islink(os.path.join(python_temp, _lib))) and os.path.isfile(os.path.join(python_temp, _lib)) and 
                    ((os.path.splitext(_lib)[1] in [".dll", ".so", ".dylib"]) or (len(os.path.basename(_lib).split("."))>1 and (os.path.basename(_lib).split(".")[1] in ["dll", "so", "dylib"])))
                    and not (_lib.startswith("python"))]

            libs_are_sym = [os.path.join(python_temp, _lib) for _lib in 
                    os.listdir(python_temp) if 
                    os.path.islink(os.path.join(python_temp, _lib)) and os.path.isfile(os.path.join(python_temp, _lib)) and 
                    ((os.path.splitext(_lib)[1] in [".dll", ".so", ".dylib"]) or (len(os.path.basename(_lib).split("."))>1 and (os.path.basename(_lib).split(".")[1] in ["dll", "so", "dylib"])))
                    and not (_lib.startswith("python"))]

            for lib in libs_not_sym:
                shutil.move(lib, os.path.join(dest_temp,
                                              os.path.basename(lib)))

            for lib in libs_are_sym:
                os.symlink(os.path.basename(os.readlink(lib)), os.path.basename(lib))

        os.chdir(str(cwd))

with open("README.md", "r") as fh:
    long_description = fh.read()

setup(
    name="compadre",
    version="1.0.35", # generated by create_package.sh argument
    python_requires='>3.4.0',
    setup_requires=[
          "pathlib2",
          "wheel>=%s"%(min_wheel_version_str,),
          "numpy>=%s"%(min_numpy_version_str,),
          #"cmake>=%s"%(min_cmake_version_str,),
    ],
    install_requires=[
          "numpy>=%s"%(min_numpy_version_str,),
    ],
    author="Paul Kuberry",
    author_email="pkuberry@gmail.com",
    description="Compatible Particle Discretization and Remap",
    long_description=long_description,
    long_description_content_type="text/markdown",
    url="https://github.com/SNLComputation/compadre",
    packages=['compadre'],
    package_dir={'compadre': 'python/compadre'},
    classifiers=[
        "Programming Language :: Python :: 3",
        "License :: OSI Approved :: BSD License",
        "Operating System :: Unix",
    ],
    ext_modules=[CMakeExtension('compadre')],
    cmdclass={
        'build_ext': build_ext,
        'install': InstallCommand
    },
    include_package_data=False
)
