"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RetryWithJitterTask = void 0;
const cdk = require("@aws-cdk/core");
const lambda = require("@aws-cdk/aws-lambda");
const sfn = require("@aws-cdk/aws-stepfunctions");
const ResilientLambdaTask_1 = require("./ResilientLambdaTask");
class RetryWithJitterTask extends sfn.Parallel {
    constructor(scope, id, props) {
        const parallelProps = {
            comment: props.comment,
            inputPath: props.inputPath,
            outputPath: "$[0]",
            resultPath: props.resultPath,
            parameters: {
                "RetryCount.$": "$$.State.RetryCount",
                "Input.$": "$"
            }
        };
        super(scope, id, parallelProps);
        const calculateJitterLambda = new lambda.Function(this, 'CalculateJitterLambda', {
            runtime: lambda.Runtime.PYTHON_3_8,
            code: lambda.Code.fromAsset(__dirname + '/../../lambda/jitter'),
            handler: "main.lambda_handler",
        });
        const calculateJitterTask = new ResilientLambdaTask_1.ResilientLambdaTask(this, this.createStateName("CalculateJitter"), {
            lambdaFunction: calculateJitterLambda,
            payload: sfn.TaskInput.fromObject({
                "RetryCount.$": "$.RetryCount",
                "Backoff": 2
            }),
            resultPath: "$.WaitSeconds",
        });
        const waitTask = new sfn.Wait(this, this.createStateName("WaitBetweenRetries"), {
            time: sfn.WaitTime.secondsPath("$.WaitSeconds"),
        });
        // Need this state to "unwrap" original input to the state.
        // Also, CDK does not support outputPath for Wait state, which I would use for retry path
        const unwrapArguments = new sfn.Pass(this, this.createStateName('Unwrap Input'), {
            outputPath: "$.Input"
        });
        const retryPath = calculateJitterTask.next(waitTask).next(unwrapArguments);
        const choiceState = new sfn.Choice(this, this.createStateName("CheckRetryCount"))
            .when(sfn.Condition.numberGreaterThan("$.RetryCount", 0), retryPath)
            .otherwise(unwrapArguments)
            .afterwards()
            .next(props.tryProcess);
        this.branch(choiceState);
        this.addRetry({
            interval: cdk.Duration.seconds(0),
            maxAttempts: props.retryProps.maxAttempts,
            errors: props.retryProps.errors
        });
    }
    createStateName(name) {
        return `${name}_${this.node.uniqueId}`;
    }
}
exports.RetryWithJitterTask = RetryWithJitterTask;
//# sourceMappingURL=data:application/json;base64,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