# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['handpick']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'handpick',
    'version': '0.11.0',
    'description': 'Inspect nested data structures.',
    'long_description': '|ukraine| |build| |coverage| |version| |black| |pyversions| |license| |downloads|\n\n==========\n Handpick\n==========\n\nHandpick is a tool to traverse nested data structures and pick all\nobjects that meet certain criteria.\n\n\nInstallation\n============\n\n.. code::\n\n    pip install handpick\n\n\nQuick introduction\n==================\n\n\nThe ``pick`` function\n---------------------\n\nThe `pick`_ generator function performs the recursive traversal of a\n(presumably nested) data structure and applies the picking criteria provided\nin the form of a predicate function (see below for various examples). Picked\nobjects are retrieved lazily by an iterator.\n\n\nSimple predicate functions\n~~~~~~~~~~~~~~~~~~~~~~~~~~\n\nThe predicate function is passed to ``pick`` as the ``predicate``\nargument. In simple cases, you can use a lambda function as a\npredicate. For example:\n\n.. code-block:: python\n\n    from handpick import pick\n\n    data = [[1, "Py"], [-2, ["", 3.0]], -4]\n\n    non_empty_strings = pick(data, lambda s: isinstance(s, str) and s)\n\n.. code::\n\n    >>> list(non_empty_strings)\n    [\'Py\']\n\n\nNon-callable predicates\n~~~~~~~~~~~~~~~~~~~~~~~\n\nIf ``predicate`` is not callable, equality will be used as the picking\ncriteria. For example:\n\n.. code-block:: python\n\n    from handpick import pick\n\n    data = [1, [1.0, [2, 1.0]], [{"1": 1}, [3]]]\n\n    ones = pick(data, 1)  # equivalent to pick(data, lambda n: n == 1)\n\n.. code::\n\n    >>> list(ones)\n    [1, 1.0, 1.0, 1]\n\n\nHandling dictionary keys\n~~~~~~~~~~~~~~~~~~~~~~~~\n\nWhen inspecting dictionaries or other mappings, you can configure\nwhether or not ``pick`` will inspect dictionary keys using the\n``dict_keys`` keyword argument. Default is False, which means only\ndictionary values are inspected. For example:\n\n.. code-block:: python\n\n    from handpick import pick\n\n    data = {"foo": {"name": "foo"}, "bar": {"name": "bar"}}\n\n    default = pick(data, lambda s: "a" in s)\n    keys_included = pick(data, lambda s: "a" in s, dict_keys=True)\n\n.. code::\n\n    >>> list(default)\n    [\'bar\']\n    >>> list(keys_included)\n    [\'name\', \'bar\', \'name\', \'bar\']\n\n\nPredicates\n----------\n\n\nThe ``Predicate`` decorator\n~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\nThe `Predicate`_ decorator wraps a function in an object that can be\ncombined with other predicates using the operators ``&`` (and) and\n``|`` (or), as well as negated using the operator ``~`` (not).\n\n\nCombining predicates\n~~~~~~~~~~~~~~~~~~~~\n\nFor example:\n\n.. code-block:: python\n\n    from handpick import pick, Predicate\n\n    @Predicate\n    def is_int(n):\n        return isinstance(n, int)\n\n    @Predicate\n    def is_even(n):\n        return n % 2 == 0\n\n    data = [[4, [5.0, 1], 3.0], [[15, []], {17: [7, [8], 0]}]]\n\n    # compound predicate\n    non_even_int = is_int & ~is_even\n\n    odd_integers = pick(data, non_even_int)\n\n.. code::\n\n    >>> list(odd_integers)\n    [1, 15, 7]\n\n\nCombining predicates with functions\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\nIn addition, the ``&`` and ``|`` operations are supported between\npredicates and regular undecorated functions. For example:\n\n.. code-block:: python\n\n    from handpick import pick, Predicate\n\n    @Predicate\n    def is_list(obj):\n        return isinstance(obj, list)\n\n    data = [("1", [2]), {("x",): [(3, [4]), "5"]}, ["x", ["6"]], {7: ("x",)}]\n\n    # compound predicate\n    short_list = (lambda obj: len(obj) < 2) & is_list\n\n    short_lists = pick(data, short_list)\n\n.. code::\n\n    >>> list(short_lists)\n    [[2], [4], [\'6\']]\n\n\nSuppressing errors\n~~~~~~~~~~~~~~~~~~\n\nOne important thing to note: when the predicate\'s underlying function raises\nan exception, the exception is suppressed and instead the call to the predicate\nreturns False. In other words, it is assumed that the object in question does\nnot meet the picking criteria. For example:\n\n.. code-block:: python\n\n    from handpick import pick, Predicate\n\n    @Predicate\n    def above_zero(n):\n        return n > 0\n\n.. code::\n\n    >>> above_zero(1)\n    True\n    >>> above_zero("a")\n    False\n    >>> positive_numbers = pick([[1, "Py", -2], [None, 3.0]], above_zero)\n    >>> list(positive_numbers)\n    [1, 3.0]\n\nIn the example above, several lists and strings were internally compared to ``0``\nbut no ``TypeError`` propagated up to the code that called ``above_zero``.\n\n\nPredicate factories\n~~~~~~~~~~~~~~~~~~~\n\nThe ``is_type`` and ``not_type`` functions can be used to create\npredicates based on an object\'s type. For example:\n\n.. code-block:: python\n\n    from handpick import pick, is_type, not_type\n\n    data = [[1.0, [2, True]], [False, [3]], ["4", {5, True}]]\n\n    strictly_integers = pick(data, is_type(int) & not_type(bool))\n\n.. code::\n\n    >>> list(strictly_integers)\n    [2, 3, 5]\n\n\nBuilt-in predicates\n~~~~~~~~~~~~~~~~~~~\n\nHandpick provides some predefined predicates to be used in common\nscenarios. For example:\n\n.. code-block:: python\n\n    from handpick import pick, NUM_STR\n\n    data = {"id": "01353", "price": 15.42, "quantity": 68, "year": "2011"}\n\n    # pick strings that can be cast to numbers\n    numeric_strings = pick(data, NUM_STR)\n\n.. code::\n\n    >>> list(numeric_strings)\n    [\'01353\', \'2011\']\n\n\nUseful functions\n----------------\n\n\nThe ``values_for_key`` function\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\nWhen inspecting data structures that contain dictionaries or other\nmappings, you can use this function to retrieve values associated with\na specific key, regardless of the nested depth in which these values\nare stored. Values are retrieved lazily by an iterator. For example:\n\n.. code-block:: python\n\n    from handpick import values_for_key\n\n    data = {\n        "node_id": 4,\n        "child_nodes": [\n            {\n                "node_id": 8,\n                "child_nodes": [\n                    {\n                        "node_id": 16,\n                    },\n                ],\n            },\n            {\n                "node_id": 9,\n            },\n        ],\n    }\n\n    node_ids = values_for_key(data, key="node_id")\n\n.. code::\n\n    >>> list(node_ids)\n    [4, 8, 16, 9]\n\nMultiple keys may be specified at a time. For example:\n\n.. code-block:: python\n\n    data = {\n        "node_id": 4,\n        "child_nodes": [\n            {\n                "id": 8,\n                "child_nodes": [\n                    {\n                        "id": 16,\n                    },\n                ],\n            },\n            {\n                "node_id": 9,\n            },\n        ],\n    }\n\n    node_ids = values_for_key(data, key=["node_id", "id"])\n\n.. code::\n\n    >>> list(node_ids)\n    [4, 8, 16, 9]\n\n\nThe ``max_depth`` function\n~~~~~~~~~~~~~~~~~~~~~~~~~~\n\nThis function returns the maximum nested depth of a data structure. For\nexample:\n\n.. code-block:: python\n\n    from handpick import max_depth\n\n    nested_list = [0, [1, [2]]]\n    nested_dict = {0: {1: {2: {3: {4: 4}}}}}\n\n.. code::\n\n    >>> max_depth(nested_list)\n    2\n    >>> max_depth(nested_dict)\n    4\n\n**Note:** Just like non-empty collections, empty collections constitute\nanother level of nested depth. For example:\n\n.. code::\n\n    >>> max_depth([0, [1, []]])\n    2\n\n\nRecipes\n=======\n\n\nFlattening nested data\n----------------------\n\nFor example:\n\n.. code-block:: python\n\n    from handpick import pick\n\n    data = [[], [0], [[[], 1], [2, [3, [4]], []], [5]]]\n    flat_data = pick(data, collections=False)\n\n.. code::\n\n    >>> list(flat_data)\n    [0, 1, 2, 3, 4, 5]\n\n\nAPI reference\n=============\n\npick\n----\n\n*handpick.pick(data, predicate=lambda obj: True, *, collections=True, dict_keys=False, strings=False, bytes_like=False)*\n\nPick objects from ``data`` based on ``predicate``.\n\nTraverse ``data`` recursively and yield all objects for which\n``predicate(obj)`` is True or truthy. ``data`` should be an iterable\ncollection. ``predicate`` should be a callable taking one argument\nand returning a Boolean value.\n\nIf ``predicate`` is omitted, all objects are picked. If ``predicate``\nis not callable, equality is used as criteria, i.e. objects for\nwhich ``obj == predicate`` are picked.\n\nBy default, collections of other objects are yielded just like any\nother objects. To exclude collections, pass ``collections=False``.\n\nWhen traversing a mapping, only its values are inspected by default.\nTo inspect both keys and values of mappings, pass ``dict_keys=True``.\n\nBy default, strings are not treated as collections of other objects\nand therefore not iterated by the recursive algorithm. This can be\nchanged by passing ``strings=True``. Strings of length 1 are never\niterated.\n\nBy default, bytes-like sequences (bytes and bytearrays) are not\ntreated as collections of other objects and therefore not iterated\nby the recursive algorithm. This can be changed by passing\n``bytes_like=True``.\n\nPredicate\n---------\n\n*@handpick.Predicate(func=None, *, suppressed_errors=(TypeError, ValueError, LookupError, AttributeError))*\n\nDecorator wrapping a function in a predicate object.\n\nThe decorated function can be combined with other predicates using\nthe operators ``&`` (and) and ``|`` (or), as well as negated using the\noperator ``~`` (not).\n\n``suppressed_errors`` can be used to customize which exception classes\nwill be suppressed by the predicate.\n\nPredicate objects are intended to be used as the ``predicate``\nargument to the ``pick`` function.\n\nis_type\n-------\n\n*handpick.is_type(type_or_types)*\n\nPredicate factory. Return a predicate that returns True if\nobject is an instance of specified type(s).\n\n``type_or_types`` must be a type or tuple of types.\n\nnot_type\n--------\n\n*handpick.not_type(type_or_types)*\n\nPredicate factory. Return a predicate that returns True if\nobject is not an instance of specified type(s).\n\n``type_or_types`` must be a type or tuple of types.\n\nno_error\n--------\n\n*handpick.no_error(func)*\n\nPredicate factory. Return a predicate that returns True if ``func``\ncan be applied on object without an exception being raised,\nFalse otherwise.\n\nIS_COLLECTION\n-------------\n\n*handpick.IS_COLLECTION*\n\nPredicate that returns True for iterable collections of other\nobjects. Strings and bytes-like objects are not treated as collections.\n\nIS_MAPPING\n----------\n\n*handpick.IS_MAPPING*\n\nPredicate that returns True for dictionaries and other mappings.\n\nINT_STR\n-------\n\n*handpick.INT_STR*\n\nPredicate that returns True for strings that can be converted\nto int.\n\nFLOAT_STR\n---------\n\n*handpick.FLOAT_STR*\n\nPredicate that returns True for strings that can be converted\nto float.\n\nNUM_STR\n-------\n\n*handpick.NUM_STR*\n\nPredicate that returns True for strings that can be converted\nto a number (i.e. an int, float or complex).\n\nvalues_for_key\n--------------\n\n*handpick.values_for_key(data, key)*\n\nPick values associated with a specific key.\n\nTraverse ``data`` recursively and yield a sequence of dictionary\nvalues that are mapped to ``key``. ``key`` may be a list of multiple\nkeys.\n\nmax_depth\n---------\n\n*handpick.max_depth(data)*\n\nReturn maximum nested depth of ``data``.\n\n``data`` should be an iterable collection. Depth is counted from zero,\ni.e. the direct elements of ``data`` are in depth 0.\n\n\n.. |ukraine| image:: https://raw.githubusercontent.com/vshymanskyy/StandWithUkraine/main/badges/StandWithUkraine.svg\n    :target: https://stand-with-ukraine.pp.ua\n.. |build| image:: https://github.com/mportesdev/handpick/actions/workflows/build-test.yml/badge.svg\n    :target: https://github.com/mportesdev/handpick/actions\n.. |coverage| image:: https://img.shields.io/codecov/c/gh/mportesdev/handpick\n    :target: https://codecov.io/gh/mportesdev/handpick\n.. |version| image:: https://img.shields.io/pypi/v/handpick\n    :target: https://pypi.org/project/handpick\n.. |black| image:: https://img.shields.io/badge/code%20style-black-000000.svg\n   :target: https://github.com/psf/black\n.. |pyversions| image:: https://img.shields.io/pypi/pyversions/handpick\n    :target: https://pypi.org/project/handpick\n.. |license| image:: https://img.shields.io/github/license/mportesdev/handpick\n    :target: https://github.com/mportesdev/handpick/blob/main/LICENSE\n.. |downloads| image:: https://pepy.tech/badge/handpick\n    :target: https://pepy.tech/project/handpick\n',
    'author': 'Michal Porteš',
    'author_email': 'michalportes1@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/mportesdev/handpick',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
