# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['fractal', 'fractal.client', 'fractal.common', 'fractal.common.models']

package_data = \
{'': ['*']}

install_requires = \
['PyJWT>=2.4.0,<3.0.0',
 'asyncclick>=8.1.3.2,<9.0.0.0',
 'httpx>=0.23.0,<0.24.0',
 'pydantic>=1.10.1,<2.0.0',
 'python-dotenv>=0.20.0,<0.21.0',
 'rich>=12.5.1,<13.0.0',
 'sqlmodel>=0.0.8,<0.0.9']

entry_points = \
{'console_scripts': ['fractal = fractal.__main__:run']}

setup_kwargs = {
    'name': 'fractal-client',
    'version': '0.1.1',
    'description': 'Client and common components of the Fractal analytics platform',
    'long_description': '# Fractal\nFractal is a framework to process high content screening data at scale and prepares it for interactive visualization.\n\nFractal provides distributed workflows that convert TBs of image data into OME-Zarr files. The platform then processes the 3D image data by applying tasks like illumination correction and maximum intensity projection. The pyramidal OME-Zarr files enable interactive visualization in the napari viewer.\nWe are building towards integrating object segmentation (nuclei, cells, organoids) and feature measurements into Fractal.\n\nFractal is currently an **early alpha build**. We currently support only Yokogawa CV7000 image data as an input. Also, we\'re in the process of refactoring the workflow management into a client-server architecture. Thus, proceed with it at your own risk, there will still be significant breaking changes. While we don\'t have any releases or stable versions and thus don\'t provide user support, we welcome questions and discussions. Open an issue to get in touch.\n\n\n![Fractal_multiwell_plate](https://user-images.githubusercontent.com/18033446/177169496-09784413-6ba9-4041-80e2-c70a70e0a5d9.gif)\n\nShortened movie of browsing an OME-Zarr file generated with Fractal in napari, using the [napari-ome-zarr plugin](https://github.com/ome/napari-ome-zarr). Actual loading times vary and can be a bit slower than in this GIF.\n\n### Contributors\nFractal was conceived in the Liberali Lab at the Friedrich Miescher Institute for Biomedical Research and in the Pelkmans Lab at the University of Zurich (both in Switzerland). The project lead is with [@gusqgm](https://github.com/gusqgm) & [@jluethi](https://github.com/jluethi). The project was originally led by [@dvischi](https://github.com/dvischi).\nThe core development is done under contract by [@mfranzon](https://github.com/mfranzon), [@tcompa](https://github.com/tcompa) & [jacopo-exact](https://github.com/jacopo-exact) from eXact lab S.r.l. <exact-lab.it>.\n\n*Installation instructions below have not been fully updated yet*\n\n-----------------------------\n\n### Requirements and configuration\n\n- Install poetry, if you don\'t already have it. Here the official [guide](https://python-poetry.org/docs/)\n\n- Move into the the folder\n```\ncd mwe_fractal/\n```\n- In your working (virtual) environment install all the dependencies with\n\n```\npoetry install\n```\n(takes few minutes)\n- Install `graphviz`. This can be done via `conda install -c anaconda graphviz`, for instance, or via `sudo apt-get install graphviz` (or equivalent commands on other systems).\n\n- Define some global configuration parameters in `fractal/fractal_config.py`. The essential ones are `partition` (the name of the SLURM partition on your cluster) and `worker_init` (which typically includes the activation of a virtual environment). An example is:\n```\n# Parameters of parsl.executors.HighThroughputExecutor\nmax_workers = 32\n# Parameters of parsl.providers.SlurmProvider\n# Note that worker_init is a command which is included at the beginning of\n# each SLURM submission scripts\nnodes_per_block = 1\ncores_per_node = 16\nmem_per_node_GB = 64\npartition = "main"\nworker_init = "source /opt/easybuild/software/Anaconda3/2019.07/"\nworker_init += "etc/profile.d/conda.sh\\n"\nworker_init += "conda activate fractal"\n```\n\n## `fractal_cmd.py` playbook\n\nOn the previous terminal window, create the first project.\n\nMove into the fractal folder.\n\nFirst of all have a look on all the possible\ncommands:\n\n```\npython fractal_cmd.py\n```\n\nTyping ```--help``` after each command like:\n\n```\npython fractal_cmd.py project --help\n```\nyou will see the arguments you had to pass.\n\n### Let\'s create new project:\n\n```\npython fractal_cmd.py project new mwe-test path_output dstest\n```\n   - In which mwe-test is the name on the project\n   - Then the path in which I want to put all the config files of my project (in this case I create a subfolder in the current folder, called ```path_output```)\n   - Last one is the name of the first dataset of the project (assuming that each project has least one dataset)\n\n- Now you should see two files:\n   - One called fractal.json in the fractal/ directory\n   - The second one, into the project folder you have choose called ```<project_name>.json```, in our case ```mwe-test.json```\n\n```fractal.json``` stores all the general information like:\n - projects names, their path and the datasets name associated, then for each project the user that have created it\n - tasks, in this way tasks could be visible by everyone\n - users and groups\n\n```mwe-test.json``` stores the specific information regarding current project:\n - datasets, their name, resources associated and the type\n - workflow (at this step workflow are project-specific)\n\nWith\n\n```\npython fractal_cmd.py project list\n```\nnow you can see your projects.\n\n### Datasets\nAdd resources to the dataset we have already created.\n\n```\npython fractal_cmd.py dataset add-resources mwe-test dstest absolute/path/of/resource\n```\n\narguments are:\n - mwe-test, name of project\n - dstest, name of dataset\n - path to the folder in which are the files. For example the path to the folder in which there are tif images.\n\n Then you can update the type of the dataset, for example set it as ```tif``` or ```png```\n\n```\npython fractal_cmd.py dataset update-type mwe-test dstest tif\n```\n\nNow you can see the dataset object using list command passing as argument the project name:\n\n```\npython fractal_cmd.py dataset list mwe-test\n```\n\n### Tasks\n\n***At the moment the task name and the unix executable which contains the logic of the tasks should have same name***\n\nThe tasks executable are in ```tasks/``` folder\n\nAdd a task; tasks are into the tasks folder. To add one or more just copy the filename without extension, example:\n\n```\npython fractal_cmd.py task add yokogawa_to_zarr zarr zarr well\n```\nThe last three arguments are the input/output type and the parallelization level. Each task should act at the level of a `plate` or `well`, while `none` means that the task is not parallelized over any of its arguments.\n\nFor the moment the "depends_on" argument is not used.\n\n```\npython fractal_cmd.py task list\n```\nNow you should see the new task.\n\n### Workflow\nLet\'s create a workflow :\n\n```\npython fractal_cmd.py workflow new mwe-test wftest create_zarr_structure\n```\n\n - project name\n - workflow name\n - tasks to add, in this case just one\n\n\n*** Note: at the moment, all workflows must start with the `create_zarr_structure` task. ***\n\n\n### Workflow execution\n\nFirst, we need to specify a set of workflow-dependent parameters, which by now should be stored in a dedicated JSON file. An example of this file (for a workflow that performs yokogawa-to-zarr conversion and maximum intensity projection) reads\n```\n{\n"workflow_name": "uzh_1_well_2x2_sites",\n"dims": [2, 2],\n"coarsening_factor_xy": 3,\n"coarsening_factor_z": 1,\n"num_levels": 5\n}\n```\n\nAfter saving these parameters in `wf_params.json`, we execute the workflow through\n\n```\npython fractal_cmd.py workflow apply mwe-test wftest dstest dstest resource_in path_output wf_params.json\n```\n\n***PAY ATTENTION: `resource_in` should be a folder that is already inserted as resource in the input dataset, otherwise fractal_cmd will raise an error. Instead, the path_out folder will be created, if it does not exist.***\n\nThis command will use [parsl](https://parsl.readthedocs.io/en/stable/index.html) to run the tasks on a SLURM cluster. When the `fractal_cmd.py` script is complete, you will see a `runinfo` folder, in the current directory, with subfolder for each run (`000`, `001`, `002`..). This subfolder includes all logs from parsl, whose structure will be described in detail at a later stage.\n\n\n### Setting up a local SLURM cluster\n\n*** WARNING: This part has not been tested with the new parsl version of fractal! ***\n\nTo test slurm integration it is possible to run Fractal on a slurm node or to test it locally, creating a small slurm cluster using Docker/Podman\n\n#### Docker setup\n\nTo create your testbed with a slurm cluster you have to install Docker :\n[Guide](https://docs.docker.com/get-docker/)\n\nThen in your working environment install docker-compose: [Guide](https://docs.docker.com/compose/install/)\n\nNow just deploy the docker-slurm.yaml file which contains four containers:\n\n- Master node with a jupyter hub; in this way will be easy launch scripts / create ones, and monitor slurm cluster.\n- Three worker nodes.\n\nTo deploy it just use docker-compose :\n```\ndocker-compose -f docker-slurm.yaml up -d\n```\n\n### Podman setup\nAs docker case, it requires Podman and podman-compose installed. Here the two references:\n\nPodman : [Guide](https://podman.io/getting-started/installation)\n\npodman-compose : [Guide](https://github.com/containers/podman-compose)\n\nTo deploy the .yaml file just use podman-compose :\n\n```\npodman-compose -f docker-slurm.yaml up -d\n```\n\n\n## Misc\n\nTo easily load all channels of the 2D example in Napari you can just run this piece of code in a cell of a Jupyter notebook or run it as an executable.\n\n```\nimport napari\nimport zarr\n\nx = zarr.open(\'path/to/201703012-NUPfollowup.zarr\')\nviewer = napari.Viewer()\n[viewer.add_image(zarr.open(f\'path/to/201703012-NUPfollowup.zarr/C22/{n}\', \'r\'), name=f\'{n}\', blending="additive") for n in x.C22]\n```\n\n## Commands cli\n\n```\nfractal project new [project_name] [path] [dataset_name] - creates new project in path and create the first dataset object\n\nfractal project list - lists registered projects\n\nfractal dataset list [project_name] - lists datasets in project\n\nfractal dataset new [project_name] [dataset_name] [resources] [dataset_type] - create a new dataset, add resources and the dataset type\n\nfractal dataset add-resources [project_name] [dataset_name] [resources] - add resources to an existing dataset\n\nfractal dataset update-type [project_name] [dataset_name] [dataset_type] - update type of dataset\n\nfractal task list - list all the existing task\n\nfractal task add [task_name] [input_type] [output_type] [parallelization_level] - create a new task\n\nfractal workflow new [project_name] [workflow_name] [tasks] - create a new workflow to the project\n\nfractal workflow list [project_name] - list all the workflow to that project\n\nfractal workflow add-task [project_name] [workflow_name] [tasks_names] - add new tasks to the workflow\n\nfractal workflow apply [project_name] [workflow_name] [dataset_name] [resource_in] [resource_out] [wf_parameters_json_file] - run workflow using the parameters into the .json\n\n```\n\n## Development\n\nDevelopment takes place on Github. You are welcome to submit an issue and open\npull requests.\n\n### Releasing\n\nBefore release checklist:\n\n- [ ] The `main` branch is checked out\n- [ ] You reviewed dependencies and dev dependencies and the lock file is up to\n      date with `pyproject.toml`.\n- [ ] The current `HEAD` of the main branch passes all the tests\n- [ ] Use\n```\npoetry run bumpver update --dry --[patch|minor] --tag-commit --commit\n```\nto test updating the version bump\n- [ ] If the previous step looks good, use\n```\npoetry run bumpver update --[patch|minor] --tag-commit --commit\n```\nto actually bump the version and commit the changes locally.\n- [ ] Test the build with\n```\npoetry build\n```\n- [ ] If the previous step was successful, push the version bump and tags:\n```\ngit push && git push --tags\n```\n- [ ] Finally, publish the updated package to pypi with\n```\npoetry publish --dry-run\n```\nremoving the `--dry-run` when you made sure that everything looks good.\n',
    'author': 'Jacopo Nespolo',
    'author_email': 'jacopo.nespolo@exact-lab.it',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/fractal-analytics-platform/fractal',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
