import json

import numpy as np
from torch.utils.data import Dataset

from .lxmert_vqa import load_obj_tsv
from imix.data.builder import DATASETS
from imix.data.utils.stream import ItemFeature

# Load part of the dataset for fast checking.
# Notice that here is the number of images instead of the number of data,
# which means all related data to the images would be used.
# TINY_IMG_NUM = 512
# FAST_IMG_NUM = 5000


class NLVR2Dataset:
    """
    An NLVR2 data example in json file:
    {
        "identifier": "train-10171-0-0",
        "img0": "train-10171-0-img0",
        "img1": "train-10171-0-img1",
        "label": 0,
        "sent": "An image shows one leather pencil case, displayed open with writing implements tucked inside.
        ",
        "uid": "nlvr2_train_0"
    }
    """

    def __init__(self, cfg):

        splits = cfg.datasets
        if isinstance(splits, str):
            splits = [splits]
            splits = splits.split(',')

        self.name = splits
        self.splits = splits

        # Loading datasets to data
        self.data = []
        for split in self.splits:
            path = cfg.annotations.get(split, None)
            if path:
                self.data.extend(json.load(open(path)))
        print('Load %d data from split(s) %s.' % (len(self.data), self.name))

        # List to dict (for evaluation and others)
        self.id2datum = {datum['uid']: datum for datum in self.data}

    def __len__(self):
        return len(self.data)


"""
An example in obj36 tsv:
FIELDNAMES = ["img_id", "img_h", "img_w", "objects_id", "objects_conf",
              "attrs_id", "attrs_conf", "num_boxes", "boxes", "features"]
FIELDNAMES would be keys in the dict returned by load_obj_tsv.
"""


@DATASETS.register_module()
class NLVR2TorchDataset(Dataset):

    def __init__(self, reader):
        super().__init__()
        self.raw_dataset = NLVR2Dataset(reader)

        topk = reader.get('topk', -1)

        # Loading detection features to img_data
        img_data = []
        for split in self.raw_dataset.splits:
            path = reader.img_feature.get(split)
            if path:
                img_data.extend(load_obj_tsv(path, topk=topk))
        self.imgid2img = {}
        for img_datum in img_data:
            self.imgid2img[img_datum['img_id']] = img_datum

        # Filter out the dataset
        self.data = []
        for datum in self.raw_dataset.data:
            if datum['img0'] in self.imgid2img and datum['img1'] in self.imgid2img:
                self.data.append(datum)
        print('Use %d data in torch dataset' % (len(self.data)))

    def __len__(self):
        return len(self.data)

    def __getitem__(self, item: int):
        datum = self.data[item]

        ques_id = datum['uid']
        ques = datum['sent']

        # Get image info
        boxes2 = []
        feats2 = []
        for key in ['img0', 'img1']:
            img_id = datum[key]
            img_info = self.imgid2img[img_id]
            boxes = img_info['boxes'].copy()
            feats = img_info['features'].copy()
            assert len(boxes) == len(feats)

            # Normalize the boxes (to 0 ~ 1)
            img_h, img_w = img_info['img_h'], img_info['img_w']
            boxes[..., (0, 2)] /= img_w
            boxes[..., (1, 3)] /= img_h
            np.testing.assert_array_less(boxes, 1 + 1e-5)
            np.testing.assert_array_less(-boxes, 0 + 1e-5)

            boxes2.append(boxes)
            feats2.append(feats)
        feats = np.stack(feats2)
        boxes = np.stack(boxes2)

        item = ItemFeature()
        item.ques_id = ques_id
        item.feats = feats
        item.boxes = boxes
        item.ques = ques

        # Create target
        if 'label' in datum:
            label = datum['label']
            item.target = label
            # return ques_id, feats, boxes, ques, label
        else:
            # return ques_id, feats, boxes, ques
            pass

        return item
