import { Construct, IResource, Resource } from '@aws-cdk/core';
import { IUserPool } from './user-pool';
/**
 * Types of authentication flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-authentication-flow.html
 */
export interface AuthFlow {
    /**
     * Enable admin based user password authentication flow
     * @default false
     */
    readonly adminUserPassword?: boolean;
    /**
     * Enable custom authentication flow
     * @default false
     */
    readonly custom?: boolean;
    /**
     * Enable auth using username & password
     * @default false
     */
    readonly userPassword?: boolean;
    /**
     * Enable SRP based authentication
     * @default false
     */
    readonly userSrp?: boolean;
    /**
     * Enable authflow to refresh tokens
     * @default false
     */
    readonly refreshToken?: boolean;
}
/**
 * OAuth settings to configure the interaction between the app and this client.
 */
export interface OAuthSettings {
    /**
     * OAuth flows that are allowed with this client.
     * @see - the 'Allowed OAuth Flows' section at https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
     * @default - all OAuth flows disabled
     */
    readonly flows: OAuthFlows;
    /**
     * List of allowed redirect URLs for the identity providers.
     * @default - no callback URLs
     */
    readonly callbackUrls?: string[];
    /**
     * OAuth scopes that are allowed with this client.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
     * @default - no OAuth scopes are configured.
     */
    readonly scopes: OAuthScope[];
}
/**
 * Types of OAuth grant flows
 * @see - the 'Allowed OAuth Flows' section at https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
 */
export interface OAuthFlows {
    /**
     * Initiate an authorization code grant flow, which provides an authorization code as the response.
     * @default false
     */
    readonly authorizationCodeGrant?: boolean;
    /**
     * The client should get the access token and ID token directly.
     * @default false
     */
    readonly implicitCodeGrant?: boolean;
    /**
     * Client should get the access token and ID token from the token endpoint
     * using a combination of client and client_secret.
     * @default false
     */
    readonly clientCredentials?: boolean;
}
/**
 * OAuth scopes that are allowed with this client.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
 */
export declare class OAuthScope {
    /**
     * Grants access to the 'phone_number' and 'phone_number_verified' claims.
     * Automatically includes access to `OAuthScope.OPENID`.
     */
    static readonly PHONE: OAuthScope;
    /**
     * Grants access to the 'email' and 'email_verified' claims.
     * Automatically includes access to `OAuthScope.OPENID`.
     */
    static readonly EMAIL: OAuthScope;
    /**
     * Returns all user attributes in the ID token that are readable by the client
     */
    static readonly OPENID: OAuthScope;
    /**
     * Grants access to all user attributes that are readable by the client
     * Automatically includes access to `OAuthScope.OPENID`.
     */
    static readonly PROFILE: OAuthScope;
    /**
     * Grants access to Amazon Cognito User Pool API operations that require access tokens,
     * such as UpdateUserAttributes and VerifyUserAttribute.
     */
    static readonly COGNITO_ADMIN: OAuthScope;
    /**
     * Custom scope is one that you define for your own resource server in the Resource Servers.
     * The format is 'resource-server-identifier/scope'.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-define-resource-servers.html
     */
    static custom(name: string): OAuthScope;
    /**
     * The name of this scope as recognized by CloudFormation.
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolclient.html#cfn-cognito-userpoolclient-allowedoauthscopes
     */
    readonly scopeName: string;
    private constructor();
}
/**
 * Properties for the UserPoolClient construct
 */
export interface UserPoolClientOptions {
    /**
     * Name of the application client
     * @default - cloudformation generated name
     */
    readonly userPoolClientName?: string;
    /**
     * Whether to generate a client secret
     * @default false
     */
    readonly generateSecret?: boolean;
    /**
     * The set of OAuth authentication flows to enable on the client
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-authentication-flow.html
     * @default - all auth flows disabled
     */
    readonly authFlows?: AuthFlow;
    /**
     * OAuth settings for this to client to interact with the app.
     * @default - see defaults in `OAuthSettings`
     */
    readonly oAuth?: OAuthSettings;
}
/**
 * Properties for the UserPoolClient construct
 */
export interface UserPoolClientProps extends UserPoolClientOptions {
    /**
     * The UserPool resource this client will have access to
     */
    readonly userPool: IUserPool;
}
/**
 * Represents a Cognito user pool client.
 */
export interface IUserPoolClient extends IResource {
    /**
     * Name of the application client
     * @attribute
     */
    readonly userPoolClientId: string;
}
/**
 * Define a UserPool App Client
 */
export declare class UserPoolClient extends Resource implements IUserPoolClient {
    /**
     * Import a user pool client given its id.
     */
    static fromUserPoolClientId(scope: Construct, id: string, userPoolClientId: string): IUserPoolClient;
    readonly userPoolClientId: string;
    private readonly _userPoolClientName?;
    constructor(scope: Construct, id: string, props: UserPoolClientProps);
    /**
     * The client name that was specified via the `userPoolClientName` property during initialization,
     * throws an error otherwise.
     */
    get userPoolClientName(): string;
    private configureAuthFlows;
    private configureOAuthFlows;
    private configureOAuthScopes;
}
