# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pyvplm', 'pyvplm.addon', 'pyvplm.core']

package_data = \
{'': ['*']}

install_requires = \
['ipython',
 'ipywidgets',
 'matplotlib',
 'numpy',
 'pandas',
 'pint',
 'pydoe2',
 'scikit-learn',
 'scipy',
 'sympy',
 'xlrd',
 'xlwt']

setup_kwargs = {
    'name': 'pyvplm',
    'version': '0.1.0',
    'description': 'Variable Power-Law regression Models tool',
    'long_description': '![](logo.png) \n\n## About\n\npyVPLM is a package that is developed to help scientist, engineer, etc., to construct power-law and/or polynomial regression models on different type of data such as finite-element simulation results, manufacturer data-sheets...\n\nIt integrates various functionalities such as :\n\n- Model parameters reduction based on Buckingham Theorem dimensional analysis and [Pint](https://pint.readthedocs.io/en/0.9/) package with derived functions.\n- Sensitivity and dependency analysis on dimensionless parameter and limited experiments to simplify further model expressions.\n- Construction of optimized experimental design on feasible-physical variables leading to full-factorial design within dimensionless space. Those DOE are the inputs of parametrized finite-element models.\n- Regression models construction with increasing  complexity (terms sorted based on their impact) and validation based on relative error repartition analysis.\n\n## Capabilities\n\n#### Dimensional analysis\n\nThe dimensional analysis has to be conducted on a defined set of physical parameters. It can be performed using alternatively `buckingham_theorem` which will return the default solution or `automatic_buckingham` which will propose different alternate sets.\n\nBased on the obtained solutions, advanced user can also define manually a new solution set with `force_buckingham` function.\n\n```\nfrom pyvplm.core.definition import PositiveParameter, PositiveParameterSet\nfrom pyvplm.addon.variablepowerlaw import buckingham_theorem\nd = PositiveParameter(\'d\', [10e-3, 150e-3],\'m\', \'pipe internal diameter\')\ne = PositiveParameter(\'e\', [.1e-3, 10e-3], \'m\', \'pipe thickness\')\nparameter_set = PositiveParameterSet(d,e)\npi_set, _ = buckingham_theorem(parameter_set, track=False)\n```\n\n#### Sensitivity & dependency analysis\n\nOnce dimensional analysis is done, there may be still a huge number of dimensionless parameter to describe a performance criteria (mostly form factor) and DOE construction phase may lead to big experiments number and long simulation times.\n\nThis is to answer this problematic that `pi_sensitivity` and `pi_dependency` functions have been designed. The obtained graph for analysis are based on primary vs. secondary parameters analysis that can be easily adapted using configuration parameters:\n\n![](./docs/source/_static/Pictures/variablepowerlaw_pi_sensitivity.png)\n\n------\n\n![](./docs/source/_static/Pictures/variablepowerlaw_pi_dependency.png)\n\n#### Optimized design of experiments\n\nThe non-constrained nor reduced experimental set are defined using [pyDOE2](https://github.com/clicumu/pyDOE2) package. It integrates automatic sizing of physical/dimensionless initial test plans and functions for selection based on distance criteria (dimensionless mapping) and spread quality (physical mapping).\n\n![](./docs/source/_static/Pictures/pixdoe_create_const_doe1.png)\n\n------\n\n![](./docs/source/_static/Pictures/pixdoe_create_const_doe2.png)\n\n#### Regression models construction\n\nThe `regression_models` function interpolate results to fit a given order polynomial model within linear or logarithmic space. Within log space, the model obtained can be transformed into variable power-law model, indeed:\n$$\nlog(\\pi_0) = a_0+a_1 \\cdot log(\\pi_1) + a_{11} \\cdot log(\\pi_1)^2+a_{12} \\cdot log(\\pi_1) \\cdot log(\\pi_2) + a_2 \\cdot log(\\pi_2) +...\n$$\nCan be expressed in the following form:\n$$\n\\pi_0 = 10^{a_0} \\cdot \\pi_1 ^{a_1 + a_{11} \\cdot log(\\pi_1)+a_{12} \\cdot log(\\pi_2)+...} \\cdot  \\pi_2^{a_2+...} \\cdot ...\n$$\nThis is the origin of package name since variable power-law model is one of the current research subject of MS2M team in [ICA](http://institut-clement-ader.org/home/) Laboratory (Toulouse-France). \n\nRegression coefficients are sorted with increasing magnitude while considering standardized values regression (first order terms are selected at the beginning to avoid singularity issues):\n\n![](./docs/source/_static/Pictures/variablepowerlaw_regression_models1.png)\n\n<u>The four criteria to evaluate model fidelity with leave-one-out cross-validation are:</u>\n\n1. Maximal relative error magnitude\n2. Average value of the magnitude of relative error which is a good indicator of both average and standard deviation\n3. Average value of the relative error\n4. Standard deviation of the relative error\n\nOn this example with 2 dimensionless parameters and order 3 polynomial expression, a 5-terms model seems to have good representation capabilities.\n\n------\n\nOnce regression models have been constructed, each one of them can be analyzed trough the analysis of their relative error using `perform_regression` function:\n\n![](./docs/source/_static/Pictures/variablepowerlaw_perform_regression1.png)\n\n## Examples and notes\n\nFour Jupyter Notebooks have been developed to present tool capabilities and functions. They can either be launched using `.batch` files from `master/notebooks` folder for windows users, or launching Jupyter Notebook application and opening `.ipynb` files from `master/docs/source/_static/Pyhton_Notebooks` folder.\n\nAdditional documentation on sub-packages (`pyvplm.core`, `pyvplm.addon`) and functions can be find on the interactive documentation `master/docs/build/index.html` which can be launched within windows operating system with double-click command on `User guide.bat` file.\n\n## Requirements\n\nThe packages uses different additional libraries to the two named before (numpy, pandas...) most of them coming with anaconda install. The complete set of necessary packages are summed up in `requirements.txt` file and checked during the install process.\n\n## Install\n\nTo install pyVPLM, follow those three steps:\n\n1. Download pyVPLM source code folder `pyvplm-master`:\n\t- Go to [GitHub](https://github.com/SizingLab/pyvplm), download the repository in any directory. \n2. Add path to Anaconda:\n\t- Open an Anaconda terminal and navigate to the master directory just copied (use `cd` command to change directory).\n\t- Add path using command `pip install -e ./` (be sure to pip from conda: `conda install pip`).\n2. Or add library permanently to Anaconda:\n\t- Open an Anaconda terminal and navigate to the master/dist directory just copied (use `cd` command to change directory).\n\t- Install library using wheel with command `pip install pyVPLM-version-py3-none-any.whl` (be sure to pip from conda: `conda install pip`).\n3. Launch unittest to check the installation:\n   - Change directory to `./pyvplm/tests` within terminal and launch command `python -m unittest`. \n\n## License\n\nThis project is under MIT License [<https://opensource.org/licenses/MIT>]\n\n## Credit\n\npyVPLM is an adaptation of the work performed by MS2M team at [ICA Laboratory](http://institut-clement-ader.org/) - France and covers the work done during different doctorate thesis:\n\n- Copyright (C) 2014 - 2017 - Florian Sanchez\n- Copyright (C) 2017 - 2019 - Francesco De Giorgi\n\n## Author\n\n[A. Reysset](https://www.researchgate.net/profile/Aurelien_Reysset)\n\n## References\n\n- F. Sanchez, M. Budinger, I. Hazyuk, "*Dimensional analysis and surrogate models for thermal modeling of power electronic components*", Electrimacs conference (2017), Toulouse\n- F. Sanchez, M. Budinger, I. Hazyuk, "*Dimensional analysis and surrogate models for the thermal modeling of Multiphysics systems*", \u2002[Applied Thermal Engineering](https://www.researchgate.net/journal/1359-4311_Applied_Thermal_Engineering) 110 (August 2016)\n\n',
    'author': 'Aurélien REYSSET',
    'author_email': 'aurelien.reysset@insa-toulouse.fr',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/SizingLab/pyvplm',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6.1,<4.0.0',
}


setup(**setup_kwargs)
