# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/01_utils.ipynb.

# %% auto 0
__all__ = ['INS', 'inscls', 'insclsattr', 'setkwsattr', 'update_specs', 'update_aspec', 'update_dspec', 'update_dattr',
           'setdattrkey', 'getdattrkey', 'setdattrval', 'setdattr', 'getdattr', 'setkwsdattr', 'setkwsdef']

# %% ../nbs/01_utils.ipynb 4
from typing import Any, TypeVar, Callable
from atyp import AnyQ, StrQ
from ispec.utils import (kwsobj, arg1st, objtype)
from ispec.types import (isdict)

INS = TypeVar('INS', bound=object)  # INS is a type variable constrained to object

# %% ../nbs/01_utils.ipynb 5
from aspec.cons import (    
    ASPEC, DSPEC, SKIND, DKIND, __SKIND__, __DKIND__, ATTRSPECS,
)
from pstr.nlit import (SPEC, SPECS, INPLACE, __COPYING__, __PASSSELF__, __READONLY__, __ATTRSPECS__,)

# %% ../nbs/01_utils.ipynb 7
def inscls(ins: INS, func: Callable, *args, **kwargs) -> INS:
    '''Call func on instance then fallback to its class if None
    
    Returns
    -------
    AnyQ:
        The value of `func(ins, *args, **kwargs)` or `func(cls, *args, **kwargs)
    '''    
    insval = func(ins, *args, **kwargs)
    if insval is not None: return insval
    cls = objtype(ins)
    clsval = func(cls, *args, **kwargs)
    return clsval

def insclsattr(ins: INS, attr: str, default: AnyQ = None) -> AnyQ:
    '''Get the instance attribute if it is None, get the class attribute.'''
    return inscls(ins, getattr, attr, default)

def setkwsattr(ins: INS, attr: str, default: AnyQ = None, **kwargs) -> dict:
    '''Set default values for `kws` based on the name and value of a given attribute.'''
    kwargs.setdefault(attr, insclsattr(ins, attr, default))
    return kwargs


# %% ../nbs/01_utils.ipynb 9
def update_specs(ins: INS, **kwargs) -> INS:
    '''Update an instance's attribute specifications e.g. (`aspec` and `dspec`).

    Parameters
    ----------
    ins : Any
        The instance whose specifications are to be updated.

    **kwargs : dict
        Initial keyword arguments.

    Returns
    -------
    ins : Any
        The updated instance.

    Notes
    -----
    Assumes that the object has the following attributes:
    - `specs` : list of specifications
    - `aspec` : attribute specification
    - `dspec` : dynamic attribute specification

    Examples
    --------
    >>> class foo:
    >>>     specs = ('aspec', 'dspec',)
    >>>     aspec = ('bar', 'baz')
    >>>     dspec = ('baz', )
    >>> 
    >>>     bar = 'bar'
    >>>     baz = 'qix'
    >>>     qix = 'qux'
    '''
    
    for spec in getattr(ins, SPECS, ()):
        sval = kwsobj(ins, spec, **kwargs)

        if spec not in getattr(ins, __READONLY__, ()):
            setattr(ins, spec, sval)

        match spec:
            case 'aspec': ins = update_aspec(ins, **kwargs)
            case 'dspec': ins = update_dspec(ins, **kwargs)
            case _: pass

    return ins

def update_aspec(ins: INS, **kwargs) -> INS:
    '''Update an instance's attribute specifications i.e. `aspec`.
    This basically just iterates over the attributes in `aspec` and
    calls `setattr` on them.

    Parameters
    ----------
    ins : Any
        The instance whose attribute specification is to be updated.

    **kwargs : dict
        Initial keyword arguments.

    Returns
    -------
    ins : Any
        The updated instance.
    '''
    spec = getattr(ins, ASPEC, ())
    for attr in spec:         
        defval = spec[attr] if isdict(spec) else None
        curval = kwsobj(ins, attr, **kwargs)
        setattr(ins, attr, arg1st(curval, defval))
    return ins

def update_dspec(ins: INS, **kwargs) -> INS:
    for attr in getattr(ins, DSPEC, ()):
        ins = update_dattr(ins, attr, **kwargs)
    return ins

def update_dattr(ins: INS, dattr: str, **kwargs) -> INS:
    '''Update the attribute key and attribute value corresponding to a given dynamic attribute.
    Given a dictionary `{dattr: akey, akey: aval}` this updates both `akey` and `aval`.

    Parameters
    ----------
    ins : Any
        The instance whose dynamic attribute you want to update.

    dattr : str
        The dynamic attribute to update.

    **kwargs : dict
        Keyword arguments Since the dynamic attribute's attribute key and value are updated
        the keyword arguments must contain both the new attribute key and value. i.e.
        given a dictionary `{dattr: akey, akey: aval}` the keyword arguments must contain
        `{dattr}` and `{akey}`.
    '''
    ins = setdattrkey(ins, dattr, **kwargs)
    akey = getdattrkey(ins, dattr)
    ins = setdattrval(ins, akey, **kwargs)
    return ins

def setdattrkey(ins: INS, dattr: str, val: StrQ = None, **kwargs) -> INS:
    '''Update the attribute key corresponding to a given dynamic attribute.
    Given a dictionary `{dattr: akey, akey: aval}` set the attribute key of `dattr` to `akey`.
    i.e. `setattr(getattr(self, dattr), kwargs.get(dattr, getattr(self, dattr)))`

    Parameters
    ----------
    ins : Any
        The instance whose attribute you want to update.

    dattr : str
        Dynamic attribute key.

    val : str, optional
        The new attribute name.

    **kwargs : dict
        Keyword arguments.

    Notes
    -----
    This does **not** update the value `aval` of the dynamic attribute.
    '''     
    spec = getattr(ins, DSPEC, kwargs.get(SPEC, ()))
    sval = spec[dattr] if isdict(spec) else None
    cur = getdattrkey(ins, dattr)
    kws = kwargs.get(dattr, None)
    setattr(ins, dattr, arg1st(val, kws, cur, sval))
    return ins

def getdattrkey(ins: INS, dattr: str) -> StrQ:
    '''Get the dynamic attribute's attribute key.
        
    Parameters
    ----------
    dattr : str
        The name of the dynamic attribute.
        
    Returns
    -------
    key : str, None
        The name of the dynamic attribute.
    
    Notes
    -----
    This is a helper function for `getdattr` and `getdattrval`. 
        It is just a wrapper for `getattr(obj, dattr, None)`.    
    '''  
    return getattr(ins, dattr, None)

def setdattrval(ins: INS, key: str, val: StrQ = None, **kwargs) -> INS:
    '''Update the value of a given dynamic attribute's key.
    Given a dictionary `{dattr: akey, akey: aval}` `akey` is the dynamic attribute key.

    Parameters
    ----------
    ins : Any
        The instance whose attribute you want to update.

    key : str
        Dynamic attribute key i.e. given dict `{dattr: key, key: val}`
        `key` is the dynamic attribute key.

    val : str, optional
        The new attribute value.

    **kwargs : dict
        Keyword arguments.

    Notes
    -----
    This does **not** update the value `aval` of the dynamic attribute.
    '''       
    if key is None: return
    cur = getattr(ins, key, None)
    kws = kwargs.get(key, None)
    setattr(ins, key, arg1st(val, kws, cur))
    return ins

def setdattr(ins: INS, dattr: str, val: AnyQ = None) -> INS:
    '''Set the val of a dynamic attribute based on its name.
    Given a dictionary `{dattr: akey, akey: aval}` this sets `aval`.
    i.e. `setattr(self, getattr(self, dattr), default)`

    Parameters
    ----------
    dattr : str
        The dynamic attribute's name.

    val : AnyQ
        The val to set the attribute to.

    Notes
    -----
    This is a helper function for `setdattrkey` and `setdattrval`.
    '''
    akey = getdattrkey(ins, dattr) # get the dyanmic attribute's attribute name
    if akey is None: return        # if the attr name doesn't exist return default
    setattr(ins, akey, val)        # get the dynamic attribute's value
    return ins

def getdattr(ins: INS, dattr: str, default: AnyQ = None) -> AnyQ:
    '''Fetch the value of a dynamic attribute based on its name.
    Given a dictionary `{dattr: akey, akey: aval}` this returns `aval`.
    i.e. `getattr(self, getattr(self, dattr), default)`

    Parameters
    ----------
    dattr : str
        The dynamic attribute's name.

    default : AnyQ
        The default value to return if the attribute does not exist.

    Returns
    -------
    AnyQ
        The value of the attribute named by the value of the attribute `dattr`.

    Examples
    --------
    >>> # These are equivalent
    >>> getdattr(obj, dattr, default)
    >>>
    >>> # Get the value of the attr named by the value of the attribute `dattr`
    >>> try:
    >>>     return getattr(obj, getattr(obj, attr, None), default)
    >>> except TypeError:
    >>>     return default
    '''
    akey = getdattrkey(ins, dattr)     # get the dyanmic attribute's attribute key
    if akey is None: return default    # if the attr name doesn't exist return default        
    return getattr(ins, akey, default) # get the dynamic attribute's value


# %% ../nbs/01_utils.ipynb 11
def setkwsdattr(ins: INS, dattr: str, **kwargs) -> dict:
    '''Set default values for `kws` based on the name and value of a given dynamic attribute.'''
    attr = insclsattr(ins, dattr, None)
    kwargs.setdefault(dattr, attr)
    if attr is None: return kwargs

    # kwargs = setkwsattr(ins, attr, None, **kwargs)
    aval = insclsattr(ins, attr, None)
    kwargs.setdefault(attr, aval)
    return kwargs

# %% ../nbs/01_utils.ipynb 12
def setkwsdef(ins: INS = None, **kwargs):
    '''Sets default keyword arguments based on the object's specifications (`aspec` and `dspec`).

    Parameters
    ----------
    obj : Any
        The object whose specifications are considered.

    **kwargs : dict
        Initial keyword arguments.

    Other Parameters
    ----------------
    inplace : bool, default: False
        Whether to modify the passed kwargs in place or return a new dictionary.

    __skind__ : set, default: SKIND
        Set of keys that are considered static attributes.
        i.e. `{'static',  'attr',  'aspec', 's',  'a'}`
    
    __dkind__ : set, default: DKIND
        Set of keys that are considered dynamic attributes.
        i.e. `{'dynamic', 'dattr', 'dspec', 'dyn', 'd', }`

    __attrspecs__ : set, default: ATTRSPECS
        Set of keys that are considered attribute specifications.
        i.e. `{'aspec', 'dspec', }`

    Returns
    -------
    kws: dict
        Dictionary of keyword arguments with defaults filled in.

    Notes
    -----
    Assumes that the object has the following attributes:
    - `aspec` : attribute specification
    - `dspec` : dynamic attribute specification
    - `specs` : list of specifications

    Examples
    --------
    >>> class foo:
    >>>     specs = ('aspec', 'dspec',)
    >>>     aspec = ('bar', 'baz')
    >>>     dspec = ('baz', )
    >>> 
    >>>     bar = 'bar'
    >>>     baz = 'qix'
    >>>     qix = 'qux'
    >>> setkwsdef(foo, inplace=False, dict())
    >>> {'bar': 'bar', 'baz': 'qix', 'qix': 'qux'}
    
    '''
    __skind__ = kwargs.get(__SKIND__, SKIND)
    __dkind__ = kwargs.get(__DKIND__, DKIND)
    __specs__ = kwargs.get(__ATTRSPECS__, ATTRSPECS)

    kws = kwargs if kwargs.get(INPLACE, False) else kwargs.copy()
    
    specs = getattr(ins, SPECS, ())
    for spec in specs:
        if spec not in __specs__: continue        
        
        attrs = getattr(ins, spec, ())
        for attr in attrs:            
            if spec in __skind__:
                kws = setkwsattr(ins, attr, **kws)
            elif spec in __dkind__:
                kws = setkwsdattr(ins, attr, **kws)        
    return kws
