# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/02_core.ipynb.

# %% auto 0
__all__ = ['SpecType', 'readonlymeta', 'specabc', 'spec', 'aspec']

# %% ../nbs/02_core.ipynb 4
import copy, itertools
from contextlib import contextmanager

from abc import ABC, ABCMeta
from typing import Any, Dict, Tuple, TypeVar, Optional
from atyp import AnyQ, StrQ

SpecType = TypeVar('SpecType', bound='specabc')  # INS is a type variable constrained to object

# %% ../nbs/02_core.ipynb 5
from ispec.utils import (getattrs, kwsopt, kwsobj, arg1st)
from ispec.types import AttrSpec, islist, isdict, istuple, isallstr

# %% ../nbs/02_core.ipynb 6
from aspec.cons import (
    Spec, ASPEC, DSPEC,  ATTRSPECS, __SKIND__, __DKIND__, 
    # SPECS, __COPYING__, __PASSSELF__, __READONLY__, __ATTRSPECS__,
)
from pstr.nlit import (SPEC, SPECS, INPLACE, __SLOTS__, __COPYING__, __PASSSELF__, __READONLY__, __ATTRSPECS__,)

from aspec.utils import (
    update_specs, update_aspec, update_dspec,
    update_dattr, setdattrkey, getdattrkey, setdattrval, 
    setdattr, getdattr, setkwsdattr, setkwsdef
)

# %% ../nbs/02_core.ipynb 8
class readonlymeta(ABCMeta):
    '''
    A meta-class that extends ABCMeta to make certain attributes read-only.
    
    Methods
    -------
    __new__(cls, name, bases, dct) -> readonlymeta
        Create and return a new object of the defined class.
    '''
    def __new__(cls, name, bases, dct):
        '''
        Create and return a new object of the defined class.
        
        Parameters
        ----------
        name : str
            Name of the new class.
        bases : tuple
            Tuple of base classes.
        dct : dict
            Dictionary of class attributes and methods.
        
        Returns
        -------
        readonlymeta
            A new object of the defined class.
        '''
        new_cls = super().__new__(cls, name, bases, dct)
        new_cls.__slots__ = tuple(set(getattr(new_cls, __SLOTS__, ())) | ATTRSPECS)
        return new_cls

# %% ../nbs/02_core.ipynb 10
class specabc(ABC, metaclass=readonlymeta):
    '''
    Abstract base class for attribute specifications with read-only meta class.
    
    Methods
    -------
    __init_subclass__(*args, **kwargs) -> None
        Ensures that subclasses have an appropriate `aspec` defined.

    __setattr__(self, name, value) -> None
        Set the class attribute if it's not read-only.

    __copy__(self, *args, **kwargs) -> object
        Return a shallow copy of the instance.

    __deepcopy__(self, memo, *args, **kwargs) -> object
        Return a deep copy of the instance.

    copy(self) -> object
        Return a shallow copy of the instance.

    deepcopy(self) -> object
        Return a deep copy of the instance.

    makecopy(self, *args, **kwargs) -> object
        Call class constructor with the same attributes as the current instance.
    '''
    __readonly__ = ()    
    def __init_subclass__(cls, *args, **kwargs):       
        '''
        Ensures that subclasses have an appropriate `aspec` defined.
        
        Checks if the 'aspec' attribute is defined and validates it.
        It also takes care of `dspec` if it is provided in kwargs.
        
        Parameters
        ----------
        args : tuple
            Positional arguments.
        kwargs : dict
            Keyword arguments.
        
        Raises
        ------
        TypeError
            If `aspec` is not an iterable of strings or a dictionary of str: Any.
        '''  
        
        # Check if 'aspec' is defined and valid
        super().__init_subclass__(*args, **kwargs)

        for spec in getattr(cls, SPECS, ()):
            attrs = kwsobj(cls, spec, **kwargs)
            if not isinstance(attrs, (list, tuple, dict)):
                raise TypeError(f"{spec} must be either an iterable of strings or a dictionary of str: Any, got {type(attrs)}")
            
            if istuple(attrs) and not isallstr(attrs):
                raise TypeError(f"If type({spec}) == tuple, all items must be strings")
            
            if isdict(attrs) and not isallstr(attrs.keys()):
                raise TypeError(f"If type({spec}) == dict, all keys must be strings")
        
            # Read-only property for {a | d}-spec
            if kwargs.get(spec, None):
                setattr(cls, spec, kwargs[spec])


    def __setattr__(self, name, value):
        '''
        Set the class attribute if it's not read-only.
        
        Parameters
        ----------
        name : str
            The name of the attribute.
            
        value : Any
            The value to set the attribute to.
        
        Raises
        ------
        AttributeError
            If the attribute is read-only.
        '''
        if name in self.__readonly__:
            if not hasattr(self, __COPYING__):
                raise AttributeError(f"can't set attribute '{name}'")
        super().__setattr__(name, value)

    @contextmanager
    def copying(self):
        self.__copying__ = True
        try: yield
        finally: del self.__copying__

    def __copy__(self: SpecType, *args, **kwargs)  -> SpecType:
        '''Return a shallow copy of the instance.
        Override `makecopy` in subclasses to change the behavior.

        Parameters
        ----------
        *args : tuple
            Positional arguments.

        **kwargs : dict
            Keyword arguments.

        Returns
        -------
        object
            A new instance of the class with shallow-copied attributes.
        '''
        with self.copying():
            new = self.makecopy(*args, **kwargs)
            new.__dict__.update(self.__dict__)
        return new

    def __deepcopy__(self: SpecType, memo: Optional[Dict[int, Any]] = None, *args, **kwargs)  -> SpecType:
        '''Return a deep copy of the instance.
        Override `makecopy` in subclasses to change the behavior.
        
        Parameters
        ----------
        memo : dict
            A dictionary of objects already copied during the current copying pass.

        *args : tuple
            Positional arguments.

        **kwargs : dict
            Keyword arguments.

        Returns
        -------
        object
            A new instance of the class with deeply copied attributes.
        '''
        with self.copying():
            if memo is None: memo = dict()
            new = self.makecopy(*args, **kwargs)
            memo[id(self)] = new
            new.__dict__.update(copy.deepcopy(self.__dict__, memo))
        return new
    
    def copy(self, *args, **kwargs) -> SpecType: 
        if not args and not kwargs:
            return self.__copy__(*args, **kwargs)
        return copy.copy(self)
    
    def deepcopy(self, *args, **kwargs) -> SpecType: 
        if not args and not kwargs:
            return self.__deepcopy__(*args, **kwargs)
        return copy.deepcopy(self)

    def makecopy(self, *args, **kwargs):
        '''Call class constructor with the same attributes as the current instance.
        This is to be used in the `__copy__` and `__deepcopy__` methods of subclasses.

        Parameters
        ----------
        *args : tuple
            Positional arguments.

        **kwargs : dict
            Keyword arguments.

        Other Parameters
        ----------------
        __passself__ : bool, default: False
            Whether to pass `self` as the first argument to the constructor.

        Returns
        -------
        object
            A new instance of the class with the same attributes as the current instance.
        '''
        if kwargs.get(__PASSSELF__, False):
            return type(self)(self, *args, **self.getattrs(**kwargs))
        return type(self)(*args, **self.getattrs(**kwargs))

# %% ../nbs/02_core.ipynb 12
class spec(specabc):
    '''Base class mixin with attribute utilities.
    
    Attributes
    ----------
    specs: {(), ('aspec', 'dspec), ('aspec', ), ('dspec', ), }
        The attribute specifications to use.
        
    __readonly__ : {(), ('aspec', 'dspec), ('aspec', ), ('dspec', ), }
        Attributes that cannot be set.

    attrs : dict
        A dictionary containing the instance attributes determined by 
        the specs in `specs` e.g. `aspec` and / or `dspec`.
        This is a read-only property.

    clsname : str
        The name of the class. This is a read-only property.

    Methods
    -------
    getattrkeys(spec: str, dyn: bool = False) -> tuple[str, ...]:
        Return attribute keys stored in `spec`.

    getattrvals(self, spec: str, dyn: bool = False) -> tuple[Any, ...]:
        Return default attribute values stored in `spec`.
    
    skeys() -> tuple[str, ...]:
        Return all attribute keys stored for each spec stored in `specs`.

    svals() -> tuple[str, ...]:
        Return default attribute values stored for each spec stored in `specs`.
    
    getattrs(**kwargs):
        Get instance parameters with optional overrides.
        
    makesame(*args, **kwargs):
        Call class constructor with the same attributes as the current instance.
    
    isinst(other):
        Check if the provided value is an instance of this class.
        
    sameattrs(other):
        Check if the provided value is an instance of this class with the same attributes.
        
    diffattrs(other):
        Check if the provided value is an instance of this class with different attributes.

    getdattrkey(dattr: str) -> str:
        Get the name of the dynamic attribute.
    
    getdattr(dattr: str, default: Any = None) -> Any:
        Get the value of the dynamic attribute.

    setdattr(dattr: str, value: Any = None):
        Set the value of the dynamic attribute.

    setdattrkey(dattr: str, **kwargs):
        Update the name of the dynamic attribute.
    
    setdattrval(aname: str, **kwargs):
        Update the value of the dynamic attribute.
    
    update_dattr(dattr: str, **kwargs):
        Update the name and value of the dynamic attribute.

    update_aspec(**kwargs):
        Update the attribute specification.

    update_dspec(**kwargs):
        Update the dynamic attribute specification.

    update_specs(**kwargs):
        Update the specifications in `specs` e.g. `aspec` and / or `dspec`.

    getclsattr(attr: str, default: Any = None) -> Any:
        Get the value of the class attribute.

    setclsattr(attr: str, val: Any = None):
        Set the value of the class attribute.

    setkwsdef(obj: Any = None, **kwargs) -> dict:
        Sets default keyword arguments based on the object's specifications (`aspec` and `dspec`).

    setkwsclsdef(**kwargs) -> dict:
        Give kwargs default values for attributes in the class's specs.

    setkwsinsdef(**kwargs) -> dict:
        Give kwargs default values for attributes in the instance's specs.

    getattr(self, attr: str, default: Any = None) -> Any:
        Checks aspec / dspec to call getattr or getdattr.

    setattr(self, attr: str, default: Any = None) -> Any:
        Checks aspec / dspec to call getattr or getdattr.

    __init_subclass__(*args, **kwargs) -> None
        Ensures that subclasses have an appropriate `aspec` defined.

    __setattr__(self, name, value) -> None
        Set the class attribute if it's not read-only.

    __copy__(self, *args, **kwargs) -> object
        Return a shallow copy of the instance.

    __deepcopy__(self, memo, *args, **kwargs) -> object
        Return a deep copy of the instance.

    copy(self) -> object
        Return a shallow copy of the instance.

    deepcopy(self) -> object
        Return a deep copy of the instance.

    makecopy(self, *args, **kwargs) -> object
        Call class constructor with the same attributes as the current instance.

    Raises
    ------
    AttributeError
        If the attribute is read-only.
    '''
    specs: Tuple[str] = ()
    __readonly__: Tuple[str] = ()

    def __init__(self, *args, **kwargs):
        '''Initialize the spec object and update its attributes based on 'aspec' and 'dspec'.

        Parameters
        ----------
        *args : tuple
            Positional arguments.
        **kwargs : dict
            Keyword arguments.

        Returns
        -------
        None
        '''
        self.update_specs(**kwargs)

    def getattrkeys(self, spec: str, dyn: bool = False) -> Tuple[str]:
        '''
        Return attribute keys stored in `aspec`.

        Parameters
        ----------
        spec : str
            The attribute specification to use.
        dyn : bool, default: False
            If True, return the dynamic attribute keys.
        
        Returns
        -------
        tuple
            A tuple containing the keys in `aspec`.
        '''
        keys = ()
        spec = getattr(self, spec, None)
        if spec is None:     pass
        elif isdict(spec):   keys = tuple(spec.keys())
        elif islist(spec):   keys = tuple(spec)
        elif istuple(spec):  keys = tuple(spec)
        elif isallstr(spec): keys = tuple(spec)
        else:                pass

        if dyn: keys = tuple(getattrs(self, keys).values())
        
        return keys
    
    def getattrvals(self, spec: str, dyn: bool = False) -> Tuple[str]:
        '''
        Return default attribute values stored in `spec`.
        
        Returns
        -------
        tuple
            A tuple containing the default attribute values.
        '''        
        keys = self.getattrkeys(spec, dyn=dyn)
        spec = getattr(self, spec, None)
        if isinstance(spec, dict): return tuple(spec.values())
        else:                      return tuple([None] * len(keys))
        
    def skeys(self) -> Tuple[str]:
        '''
        Return all attribute keys stored for each spec stored in `specs`.
        
        Returns
        -------
        tuple
            A tuple containing the keys in `aspec` and `dspec`.
        '''        
        return tuple(itertools.chain(*(
            self.getattrkeys(spec, dyn=spec.startswith('d')) 
            for spec in self.specs
        )))
    
    def svals(self) -> tuple:
        '''
        Return default attribute values stored for each spec stored in `specs`.        
        
        Returns
        -------
        tuple
            A tuple containing the default attribute values.
        '''
        return tuple(itertools.chain(*(
            self.getattrvals(spec, dyn=spec.startswith('d')) 
            for spec in self.specs
        )))
        
    def getattrs(self, **kwargs) -> dict:
        '''
        Get instance parameters with optional overrides.
        
        Parameters
        ----------
        kwargs : dict
            Dictionary containing overrides for the instance attributes.
        
        Returns
        -------
        dict
            A dictionary containing the instance attributes.
        '''        
        return kwsopt(self.attrs, **getattrs(kwargs, self.skeys()))
     
    def makesame(self: SpecType, *args, **kwargs) -> SpecType:
        '''
        Call class constructor with the same attributes as the current instance.
        
        Parameters
        ----------
        args : Any
            Positional arguments to pass to the class constructor.
            
        kwargs : Any
            Keyword arguments to pass to the class constructor.
            
        Returns
        -------
        object
            A new instance of the class with the same attributes.
        '''
        return type(self)(*args, **self.getattrs(**kwargs))
    
    @classmethod
    def isinst(cls, other: AnyQ) -> bool:
        '''
        Check if the provided value is an instance of this class.
        
        Parameters
        ----------
        other : Any
            The object to check.
            
        Returns
        -------
        bool
            True if `other` is an instance of this class, otherwise False.
        '''
        return isinstance(other, cls)

    def sameattrs(self, other: AnyQ) -> bool:
        '''
        Check if the provided value is an instance of this class with the same attributes.
        
        Parameters
        ----------
        other : Any
            The object to check against.
            
        Returns
        -------
        bool
            True if `other` is an instance with the same attributes, otherwise False.
        '''
        if self.isinst(other): return self.attrs == other.attrs
        return False
    
    def diffattrs(self, other: AnyQ) -> bool:
        '''
        Check if the provided value is an instance of this class with different attributes.
        
        Parameters
        ----------
        other : Any
            The object to check against.
            
        Returns
        -------
        bool
            True if `other` is an instance with different attributes, otherwise False.
        '''
        return not self.sameargs(other)

    @property
    def attrs(self) -> dict:
        '''
        Fetches the values of attributes as specified in `aspec`.
        
        Returns
        -------
        dict
            A dictionary containing the instance attributes.
        '''      
        return getattrs(self, self.skeys(), self.svals())
    @attrs.setter
    def attrs(self, value):
        '''
        Raises
        ------
        AttributeError
            Attempting to set the 'attrs' property will raise an AttributeError.
        '''
        raise AttributeError("Cannot set the 'attrs' property")

    @property    
    def clsname(self) -> str:
        '''
        The name of the class.
        
        Returns
        -------
        str
            The class name.
        '''     
        return type(self).__name__
    @clsname.setter
    def clsname(self, value):
        '''
        Raises
        ------
        AttributeError
            Attempting to set the 'clsname' property will raise an AttributeError.
        '''
        raise AttributeError("Cannot set the 'clsname' property")
    
    def getdattrkey(self, dattr: str) -> str:
        '''Get the dynamic attribute's attribute key.
        
        Parameters
        ----------
        dattr : str
            The name of the dynamic attribute.
            
        Returns
        -------
        key : str, None
            The name of the dynamic attribute.
        
        Notes
        -----
        This is a helper function for `getdattr` and `getdattrval`. 
            It is just a wrapper for `getattr(obj, dattr, None)`.    
        '''  
        return getdattrkey(self, dattr)

    def getdattr(self, dattr: str, default: AnyQ = None) -> AnyQ:        
        '''Fetch the value of a dynamic attribute based on its name.
        Given a dictionary `{dattr: akey, akey: aval}` this returns `aval`.
        i.e. `getattr(self, getattr(self, dattr), default)`

        Parameters
        ----------
        dattr : str
            The dynamic attribute's name.

        default : AnyQ
            The default value to return if the attribute does not exist.

        Returns
        -------
        AnyQ
            The value of the attribute named by the value of the attribute `dattr`.

        Examples
        --------
        >>> # These are equivalent
        >>> getdattr(obj, dattr, default)
        >>>
        >>> # Get the value of the attr named by the value of the attribute `dattr`
        >>> try:
        >>>     return getattr(obj, getattr(obj, attr, None), default)
        >>> except TypeError:
        >>>     return default
        '''
        return getdattr(self, dattr, default)

    def setdattr(self: SpecType, dattr: str, val: AnyQ = None) -> SpecType:
        '''Set the val of a dynamic attribute based on its name.
        Given a dictionary `{dattr: akey, akey: aval}` this sets `aval`.
        i.e. `setattr(self, getattr(self, dattr), default)`

        Parameters
        ----------
        dattr : str
            The dynamic attribute's name.

        val : AnyQ
            The val to set the attribute to.

        Notes
        -----
        This is a helper function for `setdattrkey` and `setdattrval`.
        '''
        return setdattr(self, dattr, val)

    def setdattrkey(self: SpecType, dattr: str, val: StrQ = None, **kwargs) -> SpecType:
        '''Update the attribute key corresponding to a given dynamic attribute.
        Given a dictionary `{dattr: akey, akey: aval}` set the attribute key of `dattr` to `akey`.
        i.e. `setattr(getattr(self, dattr), kwargs.get(dattr, getattr(self, dattr)))`

        Parameters
        ----------
        ins : Any
            The instance whose attribute you want to update.

        dattr : str
            Dynamic attribute key.

        val : str, optional
            The new attribute name.

        **kwargs : dict
            Keyword arguments.

        Notes
        -----
        This does **not** update the value `aval` of the dynamic attribute.
        '''   
        return setdattrkey(self, dattr, val, **kwargs)

    def setdattrval(self: SpecType, key: str, val: StrQ = None, **kwargs) -> SpecType: 
        '''Update the value of a given dynamic attribute's key.
        Given a dictionary `{dattr: akey, akey: aval}` `akey` is the dynamic attribute key.

        Parameters
        ----------
        ins : Any
            The instance whose attribute you want to update.

        key : str
            Dynamic attribute key i.e. given dict `{dattr: key, key: val}`
            `key` is the dynamic attribute key.

        val : str, optional
            The new attribute value.

        **kwargs : dict
            Keyword arguments.

        Notes
        -----
        This does **not** update the value `aval` of the dynamic attribute.
        '''     
        return setdattrval(self, key, val, **kwargs)

    def update_dattr(self: SpecType, dattr: str, **kwargs) -> SpecType:
        '''Update the attribute key and attribute value corresponding to a given dynamic attribute.
        Given a dictionary `{dattr: akey, akey: aval}` this updates both `akey` and `aval`.

        Parameters
        ----------
        ins : Any
            The instance whose dynamic attribute you want to update.

        dattr : str
            The dynamic attribute to update.

        **kwargs : dict
            Keyword arguments Since the dynamic attribute's attribute key and value are updated
            the keyword arguments must contain both the new attribute key and value. i.e.
            given a dictionary `{dattr: akey, akey: aval}` the keyword arguments must contain
            `{dattr}` and `{akey}`.
        '''
        return update_dattr(self, dattr, **kwargs)
    
    def update_aspec(self: SpecType, **kwargs) -> SpecType:
        '''Update an instance's attribute specifications i.e. `aspec`.
        This basically just iterates over the attributes in `aspec` and
        calls `setattr` on them.

        Parameters
        ----------
        ins : Any
            The instance whose attribute specification is to be updated.

        **kwargs : dict
            Initial keyword arguments.

        Returns
        -------
        ins : Any
            The updated instance.
        '''
        return update_aspec(self, **kwargs)

    def update_dspec(self: SpecType, **kwargs) -> SpecType:
        '''Update an instance's attribute specifications i.e. `dspec`.
        This basically just iterates over the dynamic attributes in `dspec` and
        calls `update_dattr` on them.

        Parameters
        ----------
        ins : Any
            The instance whose attribute specification is to be updated.

        **kwargs : dict
            Initial keyword arguments.

        Returns
        -------
        ins : Any
            The updated instance.
        '''
        return update_dspec(self, **kwargs)

    def update_specs(self: SpecType, **kwargs) -> SpecType:
        '''Update an instance's attribute specifications e.g. (`aspec` and `dspec`).

        Parameters
        ----------
        ins : Any
            The instance whose specifications are to be updated.

        **kwargs : dict
            Initial keyword arguments.

        Returns
        -------
        ins : Any
            The updated instance.

        Notes
        -----
        Assumes that the object has the following attributes:
        - `specs` : list of specifications
        - `aspec` : attribute specification
        - `dspec` : dynamic attribute specification

        Examples
        --------
        >>> class foo:
        >>>     specs = ('aspec', 'dspec',)
        >>>     aspec = ('bar', 'baz')
        >>>     dspec = ('baz', )
        >>> 
        >>>     bar = 'bar'
        >>>     baz = 'qix'
        >>>     qix = 'qux'
        '''
        return update_specs(self, **kwargs)

    @classmethod
    def getclsattr(cls, attr: str, default: AnyQ = None):
        '''
        Get a class-level attribute.

        Parameters
        ----------
        attr : str
            Name of the class-level attribute.
        default : AnyQ, optional
            Default value to return if attribute is not found.

        Returns
        -------
        AnyQ
            Value of the class attribute, or default value if attribute not found.
        '''
        if not hasattr(cls, attr):
            raise AttributeError(f'Class {cls} has not attribute {attr}')
        return getattr(cls, attr, default)
    
    @classmethod
    def setclsattr(cls, attr: str, val: AnyQ = None):
        '''
        Set a class-level attribute.

        Parameters
        ----------
        attr : str
            Name of the class-level attribute.
        val : AnyQ, optional
            Value to set for the class attribute.
        '''
        if not hasattr(cls, attr): return
        setattr(cls, attr, val)
        
    @staticmethod
    def setkwsdef(ins: Any, **kwargs) -> dict:    
        '''Sets default keyword arguments based on the object's specifications (`aspec` and `dspec`).

        Parameters
        ----------
        ins : Any
            The object whose specifications are considered.

        inplace : bool, default: False
            Whether to modify the passed kwargs in place or return a new dictionary.

        **kwargs : dict
            Initial keyword arguments.

        Returns
        -------
        dict
            Dictionary of keyword arguments with defaults filled in.
        '''    
        return setkwsdef(ins, **kwargs)
    
    @classmethod
    def setkwsclsdef(cls, **kwargs) -> dict:
        '''Give kwargs default values for attributes in the class's specs'''
        return setkwsdef(cls, **kwargs)
    
    def setkwsinsdef(self, **kwargs) -> dict:
        '''Give kwargs default values for attributes in the instance's specs'''
        return setkwsdef(self, **kwargs)

    def getattr(self: SpecType, attr: str, default: AnyQ = None) -> SpecType:
        '''Checks aspec / dspec to call getattr or getdattr'''
        if attr in self.aspec: return getattr(self, attr, default)
        if attr in self.dspec: return getdattr(self, attr, default)
        return getattr(self, attr, default)

    def setattr(self: SpecType, attr: str, default: AnyQ = None) -> SpecType:
        '''Checks aspec / dspec to call setattr or setdattr'''
        if attr in self.aspec: 
            setattr(self, attr, default)
            return self
        if attr in self.dspec: 
            self = setdattr(self, attr, default)
            return self
        setattr(self, attr, default)
        return self
    
    def update(self, **kwargs):
        '''Update the instance's attributes with the provided keyword arguments.
        
        Notes
        -----
        This is a convenience method that calls `setattr` for each keyword argument.
            It only works on attributes in `aspec` and `dspec`
        '''
        inplace = kwargs.get(INPLACE, False)
        updated = self if inplace else self.deepcopy()
        for attr, aval in kwargs.items():
            if (attr in updated.aspec or attr in updated.dspec) and attr not in updated.__readonly__: 
                updated = updated.setattr(attr, aval)
                continue        
        return updated
    

# %% ../nbs/02_core.ipynb 14
class aspec(spec):
    '''apsec (Attribute SPECification) is a base class with attribute utilities.
    
    Attributes
    ----------
    aspec : list | tuple | dict, optional
        The attribute specification. Must be either an iterable of strings 
        or a dictionary of `str: Any`. Defaults to an empty tuple.
    
    dspec : list | tuple | dict, optional
        The dynamic attribute specification. Must be either an iterable of strings 
        or a dictionary of `str: str`. Defaults to an empty tuple.

    specs: {{('aspec', 'dspec), ('aspec', ), ('dspec', ), (),}
        The attribute specifications to use.
        
    __readonly__ : {('aspec', 'dspec), ('aspec', ), ('dspec', ), (),}
        Attributes that cannot be set.

    attrs : dict
        A dictionary containing the instance attributes determined by 
        the specs in `specs` e.g. `aspec` and / or `dspec`.
        This is a read-only property.

    clsname : str
        The name of the class. This is a read-only property.

    Methods
    -------
    getattrkeys(spec: str, dyn: bool = False) -> tuple[str, ...]:
        Return attribute keys stored in `spec`.

    getattrvals(self, spec: str, dyn: bool = False) -> tuple[Any, ...]:
        Return default attribute values stored in `spec`.
    
    skeys() -> tuple[str, ...]:
        Return all attribute keys stored for each spec stored in `specs`.

    svals() -> tuple[str, ...]:
        Return default attribute values stored for each spec stored in `specs`.
    
    getattrs(**kwargs):
        Get instance parameters with optional overrides.
        
    makesame(*args, **kwargs):
        Call class constructor with the same attributes as the current instance.
    
    isinst(other):
        Check if the provided value is an instance of this class.
        
    sameattrs(other):
        Check if the provided value is an instance of this class with the same attributes.
        
    diffattrs(other):
        Check if the provided value is an instance of this class with different attributes.

    getdattrkey(dattr: str) -> str:
        Get the name of the dynamic attribute.
    
    getdattr(dattr: str, default: Any = None) -> Any:
        Get the value of the dynamic attribute.

    setdattr(dattr: str, value: Any = None):
        Set the value of the dynamic attribute.

    setdattrkey(dattr: str, **kwargs):
        Update the name of the dynamic attribute.
    
    setdattrval(aname: str, **kwargs):
        Update the value of the dynamic attribute.
    
    update_dattr(dattr: str, **kwargs):
        Update the name and value of the dynamic attribute.

    update_aspec(**kwargs):
        Update the attribute specification.

    update_dspec(**kwargs):
        Update the dynamic attribute specification.

    update_specs(**kwargs):
        Update the specifications in `specs` e.g. `aspec` and / or `dspec`.

    getclsattr(attr: str, default: Any = None) -> Any:
        Get the value of the class attribute.

    setclsattr(attr: str, val: Any = None):
        Set the value of the class attribute.

    setkwsdef(obj: Any = None, **kwargs) -> dict:
        Sets default keyword arguments based on the object's specifications (`aspec` and `dspec`).

    setkwsclsdef(**kwargs) -> dict:
        Give kwargs default values for attributes in the class's specs.

    setkwsinsdef(**kwargs) -> dict:
        Give kwargs default values for attributes in the instance's specs.

    __init_subclass__(*args, **kwargs) -> None
        Ensures that subclasses have an appropriate `aspec` defined.

    __setattr__(self, name, value) -> None
        Set the class attribute if it's not read-only.

    __copy__(self, *args, **kwargs) -> object
        Return a shallow copy of the instance.

    __deepcopy__(self, memo, *args, **kwargs) -> object
        Return a deep copy of the instance.

    copy(self) -> object
        Return a shallow copy of the instance.

    deepcopy(self) -> object
        Return a deep copy of the instance.

    makecopy(self, *args, **kwargs) -> object
        Call class constructor with the same attributes as the current instance.

    Raises
    ------
    AttributeError
        If the attribute is read-only.
    '''
    aspec: AttrSpec = ()
    dspec: AttrSpec = ()
    specs: tuple = (Spec.aspec.value, Spec.dspec.value)
    __readonly__ = (Spec.aspec.value, Spec.dspec.value)

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
