"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3RunFargateTask = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
class S3RunFargateTask extends constructs_1.Construct {
    /**
     * @summary Construct to run a Fargate task when files have been added to S3
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {S3RunFargateTaskProps} props - user provided props for the construct.
     * @since 0.8.0
     * @access private
    */
    constructor(scope, id, props) {
        super(scope, id);
        const account = aws_cdk_lib_1.Stack.of(this).account;
        const region = aws_cdk_lib_1.Stack.of(this).region;
        // Use Bucket Cfn class to enable EventBridge notifications
        const cfnBucket = props.bucket.node.defaultChild;
        cfnBucket.addPropertyOverride('NotificationConfiguration.EventBridgeConfiguration.EventBridgeEnabled', true);
        // DeadLetter queue
        const dlq = new aws_cdk_lib_1.aws_sqs.Queue(this, 'DLQ', {
            queueName: `${props.ruleName}-DLQ`,
            encryption: aws_cdk_lib_1.aws_sqs.QueueEncryption.KMS_MANAGED,
        });
        this.dlq = dlq;
        // EventBridge rule role
        const ruleRole = new aws_cdk_lib_1.aws_iam.Role(this, 'RuleRole', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('events.amazonaws.com'),
        });
        ruleRole.addToPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['ecs:RunTask'],
            resources: [props.taskDefinitionArn],
        }));
        ruleRole.addToPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: ['*'],
            conditions: {
                StringLike: {
                    'iam:PassedToService': 'ecs-tasks.amazonaws.com',
                },
            },
        }));
        // EventBridge Rule
        const s3Rule = new aws_cdk_lib_1.aws_events.CfnRule(this, 'RunFargateRule', {
            eventBusName: props.eventBusName,
            name: props.ruleName,
            description: props.ruleDescription,
            eventPattern: {
                'source': ['aws.s3'],
                'detail-type': ['Object Created'],
                'detail': {
                    bucket: {
                        name: [props.bucket.bucketName],
                    },
                },
            },
            state: 'ENABLED',
            targets: [
                {
                    arn: `arn:aws:ecs:${region}:${account}:cluster/${props.clusterName}`,
                    id: 'Id1',
                    roleArn: ruleRole.roleArn,
                    ecsParameters: {
                        taskDefinitionArn: props.taskDefinitionArn,
                        launchType: 'FARGATE',
                        networkConfiguration: {
                            awsVpcConfiguration: {
                                subnets: props.subnetIds,
                                assignPublicIp: props.assignPublicIp ? 'ENABLED' : 'DISABLED',
                                securityGroups: props.securityGroups,
                            },
                        },
                        taskCount: 1,
                    },
                    inputTransformer: {
                        inputPathsMap: {
                            bucket: '$.detail.bucket.name',
                            s3Key: '$.detail.object.key',
                        },
                        inputTemplate: `{
              \"containerOverrides\": [
                {
                  \"name\":\"${props.containerName}\",
                  \"environment\":[
                    {\"name\":\"BUCKET\",\"value\":<bucket>},
                    {\"name\":\"KEY\",\"value\":<s3Key>}
                  ]
                }
              ]
            }`,
                    },
                    retryPolicy: {
                        maximumRetryAttempts: 3,
                    },
                    deadLetterConfig: {
                        arn: dlq.queueArn,
                    },
                },
            ],
        });
        this.rule = s3Rule;
    }
}
exports.S3RunFargateTask = S3RunFargateTask;
_a = JSII_RTTI_SYMBOL_1;
S3RunFargateTask[_a] = { fqn: "@sudocdkconstructs/s3-run-fargate-task.S3RunFargateTask", version: "1.0.1" };
//# sourceMappingURL=data:application/json;base64,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