"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OriginBase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
/**
 * Represents a distribution origin, that describes the Amazon S3 bucket, HTTP server (for example, a web server), Amazon MediaStore, or other server from which CloudFront gets your files.
 *
 * @stability stable
 */
class OriginBase {
    /**
     * @stability stable
     */
    constructor(domainName, props = {}) {
        var _b;
        validateIntInRangeOrUndefined('connectionTimeout', 1, 10, (_b = props.connectionTimeout) === null || _b === void 0 ? void 0 : _b.toSeconds());
        validateIntInRangeOrUndefined('connectionAttempts', 1, 3, props.connectionAttempts, false);
        this.domainName = domainName;
        this.originPath = this.validateOriginPath(props.originPath);
        this.connectionTimeout = props.connectionTimeout;
        this.connectionAttempts = props.connectionAttempts;
        this.customHeaders = props.customHeaders;
        this.originShieldRegion = props.originShieldRegion;
    }
    /**
     * Binds the origin to the associated Distribution.
     *
     * Can be used to grant permissions, create dependent resources, etc.
     *
     * @stability stable
     */
    bind(_scope, options) {
        var _b;
        const s3OriginConfig = this.renderS3OriginConfig();
        const customOriginConfig = this.renderCustomOriginConfig();
        if (!s3OriginConfig && !customOriginConfig) {
            throw new Error('Subclass must override and provide either s3OriginConfig or customOriginConfig');
        }
        return {
            originProperty: {
                domainName: this.domainName,
                id: options.originId,
                originPath: this.originPath,
                connectionAttempts: this.connectionAttempts,
                connectionTimeout: (_b = this.connectionTimeout) === null || _b === void 0 ? void 0 : _b.toSeconds(),
                originCustomHeaders: this.renderCustomHeaders(),
                s3OriginConfig,
                customOriginConfig,
                originShield: this.renderOriginShield(this.originShieldRegion),
            },
        };
    }
    // Overridden by sub-classes to provide S3 origin config.
    /**
     * @stability stable
     */
    renderS3OriginConfig() {
        return undefined;
    }
    // Overridden by sub-classes to provide custom origin config.
    /**
     * @stability stable
     */
    renderCustomOriginConfig() {
        return undefined;
    }
    renderCustomHeaders() {
        if (!this.customHeaders || Object.entries(this.customHeaders).length === 0) {
            return undefined;
        }
        return Object.entries(this.customHeaders).map(([headerName, headerValue]) => {
            return { headerName, headerValue };
        });
    }
    /**
     * If the path is defined, it must start with a '/' and not end with a '/'.
     * This method takes in the originPath, and returns it back (if undefined) or adds/removes the '/' as appropriate.
     */
    validateOriginPath(originPath) {
        if (core_1.Token.isUnresolved(originPath)) {
            return originPath;
        }
        if (originPath === undefined) {
            return undefined;
        }
        let path = originPath;
        if (!path.startsWith('/')) {
            path = '/' + path;
        }
        if (path.endsWith('/')) {
            path = path.substr(0, path.length - 1);
        }
        return path;
    }
    /**
     * Takes origin shield region and converts to CfnDistribution.OriginShieldProperty
     */
    renderOriginShield(originShieldRegion) {
        return originShieldRegion
            ? { enabled: true, originShieldRegion }
            : undefined;
    }
}
exports.OriginBase = OriginBase;
_a = JSII_RTTI_SYMBOL_1;
OriginBase[_a] = { fqn: "@aws-cdk/aws-cloudfront.OriginBase", version: "1.119.0" };
/**
 * Throws an error if a value is defined and not an integer or not in a range.
 */
function validateIntInRangeOrUndefined(name, min, max, value, isDuration = true) {
    if (value === undefined) {
        return;
    }
    if (!Number.isInteger(value) || value < min || value > max) {
        const seconds = isDuration ? ' seconds' : '';
        throw new Error(`${name}: Must be an int between ${min} and ${max}${seconds} (inclusive); received ${value}.`);
    }
}
//# sourceMappingURL=data:application/json;base64,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