import * as acm from '@aws-cdk/aws-certificatemanager';
import * as lambda from '@aws-cdk/aws-lambda';
import * as s3 from '@aws-cdk/aws-s3';
import { IResource, Resource, Duration } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { ICachePolicy } from './cache-policy';
import { FunctionAssociation } from './function';
import { GeoRestriction } from './geo-restriction';
import { IKeyGroup } from './key-group';
import { IOrigin } from './origin';
import { IOriginRequestPolicy } from './origin-request-policy';
/**
 * Interface for CloudFront distributions.
 *
 * @stability stable
 */
export interface IDistribution extends IResource {
    /**
     * (deprecated) The domain name of the Distribution, such as d111111abcdef8.cloudfront.net.
     *
     * @deprecated - Use `distributionDomainName` instead.
     * @attribute true
     */
    readonly domainName: string;
    /**
     * The domain name of the Distribution, such as d111111abcdef8.cloudfront.net.
     *
     * @stability stable
     * @attribute true
     */
    readonly distributionDomainName: string;
    /**
     * The distribution ID for this distribution.
     *
     * @stability stable
     * @attribute true
     */
    readonly distributionId: string;
}
/**
 * Attributes used to import a Distribution.
 *
 * @stability stable
 */
export interface DistributionAttributes {
    /**
     * The generated domain name of the Distribution, such as d111111abcdef8.cloudfront.net.
     *
     * @stability stable
     * @attribute true
     */
    readonly domainName: string;
    /**
     * The distribution ID for this distribution.
     *
     * @stability stable
     * @attribute true
     */
    readonly distributionId: string;
}
/**
 * Properties for a Distribution.
 *
 * @stability stable
 */
export interface DistributionProps {
    /**
     * The default behavior for the distribution.
     *
     * @stability stable
     */
    readonly defaultBehavior: BehaviorOptions;
    /**
     * Additional behaviors for the distribution, mapped by the pathPattern that specifies which requests to apply the behavior to.
     *
     * @default - no additional behaviors are added.
     * @stability stable
     */
    readonly additionalBehaviors?: Record<string, BehaviorOptions>;
    /**
     * A certificate to associate with the distribution.
     *
     * The certificate must be located in N. Virginia (us-east-1).
     *
     * @default - the CloudFront wildcard certificate (*.cloudfront.net) will be used.
     * @stability stable
     */
    readonly certificate?: acm.ICertificate;
    /**
     * Any comments you want to include about the distribution.
     *
     * @default - no comment
     * @stability stable
     */
    readonly comment?: string;
    /**
     * The object that you want CloudFront to request from your origin (for example, index.html) when a viewer requests the root URL for your distribution. If no default object is set, the request goes to the origin's root (e.g., example.com/).
     *
     * @default - no default root object
     * @stability stable
     */
    readonly defaultRootObject?: string;
    /**
     * Alternative domain names for this distribution.
     *
     * If you want to use your own domain name, such as www.example.com, instead of the cloudfront.net domain name,
     * you can add an alternate domain name to your distribution. If you attach a certificate to the distribution,
     * you must add (at least one of) the domain names of the certificate to this list.
     *
     * @default - The distribution will only support the default generated name (e.g., d111111abcdef8.cloudfront.net)
     * @stability stable
     */
    readonly domainNames?: string[];
    /**
     * Enable or disable the distribution.
     *
     * @default true
     * @stability stable
     */
    readonly enabled?: boolean;
    /**
     * Whether CloudFront will respond to IPv6 DNS requests with an IPv6 address.
     *
     * If you specify false, CloudFront responds to IPv6 DNS requests with the DNS response code NOERROR and with no IP addresses.
     * This allows viewers to submit a second request, for an IPv4 address for your distribution.
     *
     * @default true
     * @stability stable
     */
    readonly enableIpv6?: boolean;
    /**
     * Enable access logging for the distribution.
     *
     * @default - false, unless `logBucket` is specified.
     * @stability stable
     */
    readonly enableLogging?: boolean;
    /**
     * Controls the countries in which your content is distributed.
     *
     * @default - No geographic restrictions
     * @stability stable
     */
    readonly geoRestriction?: GeoRestriction;
    /**
     * Specify the maximum HTTP version that you want viewers to use to communicate with CloudFront.
     *
     * For viewers and CloudFront to use HTTP/2, viewers must support TLS 1.2 or later, and must support server name identification (SNI).
     *
     * @default HttpVersion.HTTP2
     * @stability stable
     */
    readonly httpVersion?: HttpVersion;
    /**
     * The Amazon S3 bucket to store the access logs in.
     *
     * @default - A bucket is created if `enableLogging` is true
     * @stability stable
     */
    readonly logBucket?: s3.IBucket;
    /**
     * Specifies whether you want CloudFront to include cookies in access logs.
     *
     * @default false
     * @stability stable
     */
    readonly logIncludesCookies?: boolean;
    /**
     * An optional string that you want CloudFront to prefix to the access log filenames for this distribution.
     *
     * @default - no prefix
     * @stability stable
     */
    readonly logFilePrefix?: string;
    /**
     * The price class that corresponds with the maximum price that you want to pay for CloudFront service.
     *
     * If you specify PriceClass_All, CloudFront responds to requests for your objects from all CloudFront edge locations.
     * If you specify a price class other than PriceClass_All, CloudFront serves your objects from the CloudFront edge location
     * that has the lowest latency among the edge locations in your price class.
     *
     * @default PriceClass.PRICE_CLASS_ALL
     * @stability stable
     */
    readonly priceClass?: PriceClass;
    /**
     * Unique identifier that specifies the AWS WAF web ACL to associate with this CloudFront distribution.
     *
     * To specify a web ACL created using the latest version of AWS WAF, use the ACL ARN, for example
     * `arn:aws:wafv2:us-east-1:123456789012:global/webacl/ExampleWebACL/473e64fd-f30b-4765-81a0-62ad96dd167a`.
     * To specify a web ACL created using AWS WAF Classic, use the ACL ID, for example `473e64fd-f30b-4765-81a0-62ad96dd167a`.
     *
     * @default - No AWS Web Application Firewall web access control list (web ACL).
     * @see https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateDistribution.html#API_CreateDistribution_RequestParameters.
     * @stability stable
     */
    readonly webAclId?: string;
    /**
     * How CloudFront should handle requests that are not successful (e.g., PageNotFound).
     *
     * @default - No custom error responses.
     * @stability stable
     */
    readonly errorResponses?: ErrorResponse[];
    /**
     * The minimum version of the SSL protocol that you want CloudFront to use for HTTPS connections.
     *
     * CloudFront serves your objects only to browsers or devices that support at
     * least the SSL version that you specify.
     *
     * @default - SecurityPolicyProtocol.TLS_V1_2_2021 if the '
     * @stability stable
     * @aws-cdk /aws-cloudfront:defaultSecurityPolicyTLSv1.2_2021' feature flag is set; otherwise, SecurityPolicyProtocol.TLS_V1_2_2019.
     */
    readonly minimumProtocolVersion?: SecurityPolicyProtocol;
}
/**
 * A CloudFront distribution with associated origin(s) and caching behavior(s).
 *
 * @stability stable
 */
export declare class Distribution extends Resource implements IDistribution {
    /**
     * Creates a Distribution construct that represents an external (imported) distribution.
     *
     * @stability stable
     */
    static fromDistributionAttributes(scope: Construct, id: string, attrs: DistributionAttributes): IDistribution;
    /**
     * The domain name of the Distribution, such as d111111abcdef8.cloudfront.net.
     *
     * @stability stable
     */
    readonly domainName: string;
    /**
     * The domain name of the Distribution, such as d111111abcdef8.cloudfront.net.
     *
     * @stability stable
     */
    readonly distributionDomainName: string;
    /**
     * The distribution ID for this distribution.
     *
     * @stability stable
     */
    readonly distributionId: string;
    private readonly defaultBehavior;
    private readonly additionalBehaviors;
    private readonly boundOrigins;
    private readonly originGroups;
    private readonly errorResponses;
    private readonly certificate?;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: DistributionProps);
    /**
     * Adds a new behavior to this distribution for the given pathPattern.
     *
     * @param pathPattern the path pattern (e.g., 'images/*') that specifies which requests to apply the behavior to.
     * @param origin the origin to use for this behavior.
     * @param behaviorOptions the options for the behavior at this path.
     * @stability stable
     */
    addBehavior(pathPattern: string, origin: IOrigin, behaviorOptions?: AddBehaviorOptions): void;
    private addOrigin;
    private addOriginGroup;
    private renderOrigins;
    private renderOriginGroups;
    private renderCacheBehaviors;
    private renderErrorResponses;
    private renderLogging;
    private renderRestrictions;
    private renderViewerCertificate;
}
/**
 * Maximum HTTP version to support.
 *
 * @stability stable
 */
export declare enum HttpVersion {
    /**
     * HTTP 1.1.
     *
     * @stability stable
     */
    HTTP1_1 = "http1.1",
    /**
     * HTTP 2.
     *
     * @stability stable
     */
    HTTP2 = "http2"
}
/**
 * The price class determines how many edge locations CloudFront will use for your distribution.
 *
 * See https://aws.amazon.com/cloudfront/pricing/ for full list of supported regions.
 *
 * @stability stable
 */
export declare enum PriceClass {
    /**
     * USA, Canada, Europe, & Israel.
     *
     * @stability stable
     */
    PRICE_CLASS_100 = "PriceClass_100",
    /**
     * PRICE_CLASS_100 + South Africa, Kenya, Middle East, Japan, Singapore, South Korea, Taiwan, Hong Kong, & Philippines.
     *
     * @stability stable
     */
    PRICE_CLASS_200 = "PriceClass_200",
    /**
     * All locations.
     *
     * @stability stable
     */
    PRICE_CLASS_ALL = "PriceClass_All"
}
/**
 * How HTTPs should be handled with your distribution.
 *
 * @stability stable
 */
export declare enum ViewerProtocolPolicy {
    /**
     * HTTPS only.
     *
     * @stability stable
     */
    HTTPS_ONLY = "https-only",
    /**
     * Will redirect HTTP requests to HTTPS.
     *
     * @stability stable
     */
    REDIRECT_TO_HTTPS = "redirect-to-https",
    /**
     * Both HTTP and HTTPS supported.
     *
     * @stability stable
     */
    ALLOW_ALL = "allow-all"
}
/**
 * Defines what protocols CloudFront will use to connect to an origin.
 *
 * @stability stable
 */
export declare enum OriginProtocolPolicy {
    /**
     * Connect on HTTP only.
     *
     * @stability stable
     */
    HTTP_ONLY = "http-only",
    /**
     * Connect with the same protocol as the viewer.
     *
     * @stability stable
     */
    MATCH_VIEWER = "match-viewer",
    /**
     * Connect on HTTPS only.
     *
     * @stability stable
     */
    HTTPS_ONLY = "https-only"
}
/**
 * The SSL method CloudFront will use for your distribution.
 *
 * Server Name Indication (SNI) - is an extension to the TLS computer networking protocol by which a client indicates
 *   which hostname it is attempting to connect to at the start of the handshaking process. This allows a server to present
 *   multiple certificates on the same IP address and TCP port number and hence allows multiple secure (HTTPS) websites
 * (or any other service over TLS) to be served by the same IP address without requiring all those sites to use the same certificate.
 *
 * CloudFront can use SNI to host multiple distributions on the same IP - which a large majority of clients will support.
 *
 * If your clients cannot support SNI however - CloudFront can use dedicated IPs for your distribution - but there is a prorated monthly charge for
 * using this feature. By default, we use SNI - but you can optionally enable dedicated IPs (VIP).
 *
 * See the CloudFront SSL for more details about pricing : https://aws.amazon.com/cloudfront/custom-ssl-domains/
 *
 * @stability stable
 */
export declare enum SSLMethod {
    /**
     * @stability stable
     */
    SNI = "sni-only",
    /**
     * @stability stable
     */
    VIP = "vip"
}
/**
 * The minimum version of the SSL protocol that you want CloudFront to use for HTTPS connections.
 *
 * CloudFront serves your objects only to browsers or devices that support at least the SSL version that you specify.
 *
 * @stability stable
 */
export declare enum SecurityPolicyProtocol {
    /**
     * @stability stable
     */
    SSL_V3 = "SSLv3",
    /**
     * @stability stable
     */
    TLS_V1 = "TLSv1",
    /**
     * @stability stable
     */
    TLS_V1_2016 = "TLSv1_2016",
    /**
     * @stability stable
     */
    TLS_V1_1_2016 = "TLSv1.1_2016",
    /**
     * @stability stable
     */
    TLS_V1_2_2018 = "TLSv1.2_2018",
    /**
     * @stability stable
     */
    TLS_V1_2_2019 = "TLSv1.2_2019",
    /**
     * @stability stable
     */
    TLS_V1_2_2021 = "TLSv1.2_2021"
}
/**
 * The HTTP methods that the Behavior will accept requests on.
 *
 * @stability stable
 */
export declare class AllowedMethods {
    /**
     * HEAD and GET.
     *
     * @stability stable
     */
    static readonly ALLOW_GET_HEAD: AllowedMethods;
    /**
     * HEAD, GET, and OPTIONS.
     *
     * @stability stable
     */
    static readonly ALLOW_GET_HEAD_OPTIONS: AllowedMethods;
    /**
     * All supported HTTP methods.
     *
     * @stability stable
     */
    static readonly ALLOW_ALL: AllowedMethods;
    /**
     * HTTP methods supported.
     *
     * @stability stable
     */
    readonly methods: string[];
    private constructor();
}
/**
 * The HTTP methods that the Behavior will cache requests on.
 *
 * @stability stable
 */
export declare class CachedMethods {
    /**
     * HEAD and GET.
     *
     * @stability stable
     */
    static readonly CACHE_GET_HEAD: CachedMethods;
    /**
     * HEAD, GET, and OPTIONS.
     *
     * @stability stable
     */
    static readonly CACHE_GET_HEAD_OPTIONS: CachedMethods;
    /**
     * HTTP methods supported.
     *
     * @stability stable
     */
    readonly methods: string[];
    private constructor();
}
/**
 * Options for configuring custom error responses.
 *
 * @stability stable
 */
export interface ErrorResponse {
    /**
     * The minimum amount of time, in seconds, that you want CloudFront to cache the HTTP status code specified in ErrorCode.
     *
     * @default - the default caching TTL behavior applies
     * @stability stable
     */
    readonly ttl?: Duration;
    /**
     * The HTTP status code for which you want to specify a custom error page and/or a caching duration.
     *
     * @stability stable
     */
    readonly httpStatus: number;
    /**
     * The HTTP status code that you want CloudFront to return to the viewer along with the custom error page.
     *
     * If you specify a value for `responseHttpStatus`, you must also specify a value for `responsePagePath`.
     *
     * @default - the error code will be returned as the response code.
     * @stability stable
     */
    readonly responseHttpStatus?: number;
    /**
     * The path to the custom error page that you want CloudFront to return to a viewer when your origin returns the `httpStatus`, for example, /4xx-errors/403-forbidden.html.
     *
     * @default - the default CloudFront response is shown.
     * @stability stable
     */
    readonly responsePagePath?: string;
}
/**
 * The type of events that a Lambda@Edge function can be invoked in response to.
 *
 * @stability stable
 */
export declare enum LambdaEdgeEventType {
    /**
     * The origin-request specifies the request to the origin location (e.g. S3).
     *
     * @stability stable
     */
    ORIGIN_REQUEST = "origin-request",
    /**
     * The origin-response specifies the response from the origin location (e.g. S3).
     *
     * @stability stable
     */
    ORIGIN_RESPONSE = "origin-response",
    /**
     * The viewer-request specifies the incoming request.
     *
     * @stability stable
     */
    VIEWER_REQUEST = "viewer-request",
    /**
     * The viewer-response specifies the outgoing response.
     *
     * @stability stable
     */
    VIEWER_RESPONSE = "viewer-response"
}
/**
 * Represents a Lambda function version and event type when using Lambda@Edge.
 *
 * The type of the {@link AddBehaviorOptions.edgeLambdas} property.
 *
 * @stability stable
 */
export interface EdgeLambda {
    /**
     * The version of the Lambda function that will be invoked.
     *
     * **Note**: it's not possible to use the '$LATEST' function version for Lambda@Edge!
     *
     * @stability stable
     */
    readonly functionVersion: lambda.IVersion;
    /**
     * The type of event in response to which should the function be invoked.
     *
     * @stability stable
     */
    readonly eventType: LambdaEdgeEventType;
    /**
     * Allows a Lambda function to have read access to the body content.
     *
     * Only valid for "request" event types (`ORIGIN_REQUEST` or `VIEWER_REQUEST`).
     * See https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/lambda-include-body-access.html
     *
     * @default false
     * @stability stable
     */
    readonly includeBody?: boolean;
}
/**
 * Options for adding a new behavior to a Distribution.
 *
 * @stability stable
 */
export interface AddBehaviorOptions {
    /**
     * HTTP methods to allow for this behavior.
     *
     * @default AllowedMethods.ALLOW_GET_HEAD
     * @stability stable
     */
    readonly allowedMethods?: AllowedMethods;
    /**
     * HTTP methods to cache for this behavior.
     *
     * @default CachedMethods.CACHE_GET_HEAD
     * @stability stable
     */
    readonly cachedMethods?: CachedMethods;
    /**
     * The cache policy for this behavior.
     *
     * The cache policy determines what values are included in the cache key,
     * and the time-to-live (TTL) values for the cache.
     *
     * @default CachePolicy.CACHING_OPTIMIZED
     * @see https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html.
     * @stability stable
     */
    readonly cachePolicy?: ICachePolicy;
    /**
     * Whether you want CloudFront to automatically compress certain files for this cache behavior.
     *
     * See https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/ServingCompressedFiles.html#compressed-content-cloudfront-file-types
     * for file types CloudFront will compress.
     *
     * @default true
     * @stability stable
     */
    readonly compress?: boolean;
    /**
     * The origin request policy for this behavior.
     *
     * The origin request policy determines which values (e.g., headers, cookies)
     * are included in requests that CloudFront sends to the origin.
     *
     * @default - none
     * @stability stable
     */
    readonly originRequestPolicy?: IOriginRequestPolicy;
    /**
     * Set this to true to indicate you want to distribute media files in the Microsoft Smooth Streaming format using this behavior.
     *
     * @default false
     * @stability stable
     */
    readonly smoothStreaming?: boolean;
    /**
     * The protocol that viewers can use to access the files controlled by this behavior.
     *
     * @default ViewerProtocolPolicy.ALLOW_ALL
     * @stability stable
     */
    readonly viewerProtocolPolicy?: ViewerProtocolPolicy;
    /**
     * The CloudFront functions to invoke before serving the contents.
     *
     * @default - no functions will be invoked
     * @stability stable
     */
    readonly functionAssociations?: FunctionAssociation[];
    /**
     * The Lambda@Edge functions to invoke before serving the contents.
     *
     * @default - no Lambda functions will be invoked
     * @see https://aws.amazon.com/lambda/edge
     * @stability stable
     */
    readonly edgeLambdas?: EdgeLambda[];
    /**
     * A list of Key Groups that CloudFront can use to validate signed URLs or signed cookies.
     *
     * @default - no KeyGroups are associated with cache behavior
     * @see https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html
     * @stability stable
     */
    readonly trustedKeyGroups?: IKeyGroup[];
}
/**
 * Options for creating a new behavior.
 *
 * @stability stable
 */
export interface BehaviorOptions extends AddBehaviorOptions {
    /**
     * The origin that you want CloudFront to route requests to when they match this behavior.
     *
     * @stability stable
     */
    readonly origin: IOrigin;
}
