/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
import * as api from '@aws-cdk/aws-apigateway';
import * as kinesis from '@aws-cdk/aws-kinesis';
import * as iam from '@aws-cdk/aws-iam';
import { Construct } from '@aws-cdk/core';
import * as logs from '@aws-cdk/aws-logs';
/**
 * @summary The properties for the ApiGatewayToKinesisStreamsProps class.
 */
export interface ApiGatewayToKinesisStreamsProps {
    /**
     * Optional user-provided props to override the default props for the API Gateway.
     *
     * @default - Default properties are used.
     */
    readonly apiGatewayProps?: api.RestApiProps;
    /**
     * API Gateway request template for the PutRecord action.
     *
     * If not provided, a default one will be used.
     *
     * @default - { "StreamName": "${this.kinesisStream.streamName}", "Data": "$util.base64Encode($input.json('$.data'))",
     * "PartitionKey": "$input.path('$.partitionKey')" }
     */
    readonly putRecordRequestTemplate?: string;
    /**
     * API Gateway request model for the PutRecord action.
     *
     * If not provided, a default one will be created.
     *
     * @default - {"$schema":"http://json-schema.org/draft-04/schema#","title":"PutRecord proxy single-record payload","type":"object",
     * "required":["data","partitionKey"],"properties":{"data":{"type":"string"},"partitionKey":{"type":"string"}}}
     */
    readonly putRecordRequestModel?: api.ModelOptions;
    /**
     * API Gateway request template for the PutRecords action.
     *
     * If not provided, a default one will be used.
     *
     * @default - { "StreamName": "${this.kinesisStream.streamName}", "Records": [ #foreach($elem in $input.path('$.records'))
     * { "Data": "$util.base64Encode($elem.data)", "PartitionKey": "$elem.partitionKey"}#if($foreach.hasNext),#end #end ] }
     */
    readonly putRecordsRequestTemplate?: string;
    /**
     * API Gateway request model for the PutRecords action.
     *
     * If not provided, a default one will be created.
     *
     * @default - {"$schema":"http://json-schema.org/draft-04/schema#","title":"PutRecords proxy payload data","type":"object","required":["records"],
     * "properties":{"records":{"type":"array","items":{"type":"object",
     * "required":["data","partitionKey"],"properties":{"data":{"type":"string"},"partitionKey":{"type":"string"}}}}}}
     */
    readonly putRecordsRequestModel?: api.ModelOptions;
    /**
     * Existing instance of Kinesis Stream, if this is set then kinesisStreamProps is ignored.
     *
     * @default - None
     */
    readonly existingStreamObj?: kinesis.Stream;
    /**
     * Optional user-provided props to override the default props for the Kinesis Data Stream.
     *
     * @default - Default properties are used.
     */
    readonly kinesisStreamProps?: kinesis.StreamProps;
    /**
     * User provided props to override the default props for the CloudWatchLogs LogGroup.
     *
     * @default - Default props are used
     */
    readonly logGroupProps?: logs.LogGroupProps;
}
/**
 * @summary The ApiGatewayToKinesisStreams class.
 */
export declare class ApiGatewayToKinesisStreams extends Construct {
    readonly apiGateway: api.RestApi;
    readonly apiGatewayRole: iam.Role;
    readonly apiGatewayCloudWatchRole: iam.Role;
    readonly apiGatewayLogGroup: logs.LogGroup;
    readonly kinesisStream: kinesis.Stream;
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the ApiGatewayToKinesisStreams class.
     * @since 1.62.0
     * @access public
     */
    constructor(scope: Construct, id: string, props: ApiGatewayToKinesisStreamsProps);
    private getPutRecordTemplate;
    private getPutRecordModel;
    private getPutRecordsTemplate;
    private getPutRecordsModel;
}
