# lunar-sdk

## Contents 
- [Description](#description)
- [Installation](#installation)
- [How to use](#how-to-use)
  - [credentials](#credentials)
  - [SDK](#sdk)
  - [CLI](#cli)
- [Development](#development)
  - [Requirements for development](#requirements-for-development)
  - [Python environment](#python-environment)
  - [Documents generation](#documents-generation)


SDK for lunar microservices

## Description

A Python based SDK & CLI for Lunar microservices

This python package includes
- Functions for [lunar-rec](https://github.com/sktaiflow/lunar-rec)
- CLI (Command Line Interface) named `lunar`

## Installation

Build package and wheel files with `poetry`.

```bash
# installation
poetry install

poetry run lunar -h
```

## How to use

### Credentials
`lunar` uses a credentials file to setup a config(`env`, `apikey`) with a `init` file type.  
The path for credentials file and format must be as follow:

```bash
[Path]

.lunar/credentials
```

```bash
[default]
env=local
apikey=ABCDEFG123

[dev]
env=dev
apikey=AERSDF123

[stg]
env=stg
apikey=HIJKLMNO456

[prd]
env=prd
apikey=PQRSTUV789
```
The first [default] is used when SDK / CLI is run without a specified profile.

To specify a profile, use `AWS_PROFILE` to overrides the default profile for commands that run in that session. 

```bash
export AWS_PROFILE=stg
```

### SDK

`lunar` package provides functions for using all lunar microservices.
For the detailed information related to each function, please refer to [Docs](https://sktaiflow.github.io/lunar-sdk/lunar/).

Here is an example for CRUDing channels:

```python
import lunar

# Environment for Lunar-rec is automatically set with credential file
client = lunar.client("channel")

# Get list of channels
channels = client.get_channels()

# Get a channel
channel = client.get_channel(id="test_channel")

# Create a channel
channel = client.create_channel(id="new_channel", experiment_id="new_experiment")

# Update(or partial) a channel 
updated_channel = client.update_channel(id="test_channel", experiment_id="a")
partially_updated_channel = client.update_channel_partial(id="test_channel", experiment_id="ab")

# Delete a channel
client.delete_channel(channel.id)
```

All functions on the SDK also supports asyncio. If necessary, use an async version of each function.
- Naming rule: Suffix with **_async**. (e.g. `list_channels()` → `list_channels_async()`)

### CLI

`lunar` also provides CLI (Command Line Interface) generated by [click](https://click.palletsprojects.com/) package.

```zsh
poetry run lunar -h
```

should show something like the below:

``` zsh
Usage: lunar [OPTIONS] COMMAND [ARGS]...

Options:
  -h, --help  Show this message and exit.

Commands:
  channel     Channels on lunar-rec
  experiment  Experiments on lunar-rec
  recommend   Recommendation on lunar-rec
```

Each option also has its own usage instruction.

``` zsh
poetry run lunar experiments -h

Usage: lunar experiment [OPTIONS] {CREATE|LIST|READ|UPDATE|DELETE}

  Experiments on lunar-rec

  Send a request to Lunar Recommendation API (`/v1/experiemnts/`).

  Return: dict or list(dict)

Options:
  -i, --id TEXT                   Unique identifier of a channel
  -b, --buckets <TEXT INTEGER>...
                                  Bucket list
  -s, --bucketing-seed TEXT       Random seed for bucketing
  -p, --partial                   Partial update or not (default: False)
  -h, --help                      Show this message and exit.
```

## Development

### Requirements for development

- Python 3.8
- Poetry

Installing Python 3.8 with [pyenv](https://github.com/pyenv/pyenv) is recommended. Check [this](https://github.com/pyenv/pyenv#installation).

For zsh,

```zsh
# Install Xcode command line tools (Only if you don't have it already)
xcode-select --install

# Install pyenv and its dependencies
brew update
brew install pyenv openssl readline sqlite3 xz zlib

echo 'eval "$(pyenv init --path)"' >> ~/.zshrc
eval '"$(pyenv init -)"' >> ~/.zshrc

# Restart the terminal

# Install Python 3.8
pyenv install 3.8.10
pyenv global 3.8.10

# Restart the terminal

# Check if the version is right
python -V

# Install Poetry
curl -sSL https://raw.githubusercontent.com/python-poetry/poetry/master/get-poetry.py | python -
poetry config virtualenvs.in-project true
```

should work.

### Python environment

```zsh
poetry install
```

If you need additional deps,

```zsh
# For production
poetry add package_name

# For development only
poetry add --dev package_name
```

### Documents generation

```bash
rm -rf docs
pdoc --html --config show_source_code=False -f -o ./docs lunar
```
