# AUTOGENERATED! DO NOT EDIT! File to edit: notebooks/04_hirise.ipynb (unless otherwise specified).

__all__ = ['storage_root', 'baseurl', 'rdrindex', 'OBSERVATION_ID', 'check_url_exists', 'ProductPathfinder',
           'RGB_NOMAP', 'RGB_NOMAPCollection']

# Cell

import warnings

import hvplot.xarray  # noqa
import rasterio
import requests
import rioxarray as rxr
from dask import compute, delayed
from fastcore.utils import Path
from .config import config
from .pds.apps import get_index
from .utils import url_retrieve
from pyrise import products as prod
from yarl import URL

warnings.filterwarnings("ignore", category=rasterio.errors.NotGeoreferencedWarning)

# Cell
storage_root = config.storage_root / "missions/mro/hirise"
baseurl = URL("https://hirise-pds.lpl.arizona.edu/download/PDS")
rdrindex = get_index("mro.hirise", "rdr")

# Cell
class OBSERVATION_ID:
    """Manage HiRISE observation ids.

    For example PSP_003092_0985.

    `phase` is set to PSP for orbits < 11000, no setting required.

    Parameters
    ----------
    obsid : str, optional
        One can optionally also create an 'empty' OBSERVATION_ID object and set the
        properties accordingly to create a new obsid.
    """

    def __init__(self, obsid=None):
        if obsid is not None:
            phase, orbit, targetcode = obsid.split("_")
            self._orbit = int(orbit)
            self._targetcode = targetcode
        else:
            self._orbit = None
            self._targetcode = None

    @property
    def orbit(self):
        return str(self._orbit).zfill(6)

    @orbit.setter
    def orbit(self, value):
        if value > 999999:
            raise ValueError("Orbit cannot be larger than 999999")
        elif len(value) != 6:
            raise ValueError("Orbit string must be 6 digits.")
        self._orbit = value

    @property
    def targetcode(self):
        return self._targetcode

    @targetcode.setter
    def targetcode(self, value):
        if len(str(value)) != 4:
            raise ValueError("Targetcode must be exactly 4 characters.")
        self._targetcode = value

    @property
    def phase(self):
        return "PSP" if int(self.orbit) < 11000 else "ESP"

    def __str__(self):
        return "{}_{}_{}".format(self.phase, self.orbit, self.targetcode)

    def __repr__(self):
        return self.__str__()

    @property
    def s(self):
        return self.__str__()

    @property
    def upper_orbit_folder(self):
        """
        get the upper folder name where the given orbit folder is residing on the
        hisync server
        """
        lower = int(self.orbit) // 100 * 100
        return "_".join(["ORB", str(lower).zfill(6), str(lower + 99).zfill(6)])

    @property
    def storage_path_stem(self):
        s = "{phase}/{orbitfolder}/{obsid}".format(
            phase=self.phase, orbitfolder=self.upper_orbit_folder, obsid=self.s
        )
        return s

# Cell
def check_url_exists(url):
    response = requests.head(url)
    if response.status_code < 400:
        return True
    else:
        return False

# Cell
class ProductPathfinder:
    """Determine paths and URLs for HiRISE RDR products (also EXTRAS.)

    Attributes `jp2_path` and `label_path` get you the official RDR mosaic product,
    with `kind` steering if you get the COLOR or the RED product.
    All other properties go to the RDR/EXTRAS folder.

    Parameters
    ----------
    initstr : str, optional
        PRODUCT_ID string like PSP_003902_0985_RED

    Note
    ----
    The "PDS" part of the path is handled in the OBSERVATION_ID class.

    """

    kinds = ["RED", "BG", "IR", "COLOR", "IRB", "MIRB", "MRGB", "RGB", "RGB.NOMAP"]

    @classmethod
    def from_path(cls, path):
        path = Path(path)
        return cls(path.stem)

    def __init__(self, initstr=None):
        if initstr is not None:
            tokens = initstr.split("_")
            self._obsid = OBSERVATION_ID("_".join(tokens[:3]))
            try:
                self.kind = tokens[3]
            except IndexError:
                self._kind = None
        else:
            self._kind = None

    @property
    def obsid(self):
        return self._obsid

    @obsid.setter
    def obsid(self, value):
        self._obsid = OBSERVATION_ID(value)

    @property
    def kind(self):
        return self._kind

    @kind.setter
    def kind(self, value):
        if value not in self.kinds:
            raise ValueError(f"kind must be in {self.kinds}")
        self._kind = value

    def __str__(self):
        return f"{self.obsid}_{self.kind}"

    def __repr__(self):
        return self.__str__()

    @property
    def s(self):
        return self.__str__()

    @property
    def storage_stem(self):
        return f"{self.obsid.storage_path_stem}/{self.s}"

    @property
    def label_fname(self):
        return f"{self.s}.LBL"

    @property
    def label_path(self):
        return "RDR/" + self.storage_stem + ".LBL"

    def _make_url(self, obj):
        path = getattr(self, f"{obj}_path")
        url = baseurl / str(path)
        if not check_url_exists(url):
            warnings.warn(f"{url} does not exist on the server.")
        return url

    def __getattr__(self, item):
        tokens = item.split("_")
        try:
            if tokens[-1] == "url":
                return self._make_url("_".join(tokens[:-1]))
        except IndexError:
            raise AttributeError(f"No attribute named '{item}' found.")

    @property
    def jp2_fname(self):
        return self.s + ".JP2"

    @property
    def jp2_path(self):
        prefix = "RDR/"
        postfix = ""
        if self.kind not in ["RED", "COLOR"]:
            prefix += "EXTRAS/"
        if self.kind in ["IRB"]:
            postfix = ".NOMAP"
        return prefix + self.storage_stem + postfix + ".JP2"

    @property
    def nomap_jp2_path(self):
        if self.kind in ["RED", "IRB", "RGB"]:
            return f"EXTRAS/RDR/{self.storage_stem}.NOMAP.JP2"
        else:
            raise ValueError(f"No NOMAP exists for {self.kind}.")

    @property
    def quicklook_path(self):
        if self.kind in ["COLOR", "RED"]:
            return Path("EXTRAS/RDR/") / (self.storage_stem + ".QLOOK.JP2")
        else:
            raise ValueError(f"No quicklook exists for {self.kind} products.")

    @property
    def abrowse_path(self):
        if self.kind in ["COLOR", "MIRB", "MRGB", "RED"]:
            return Path("EXTRAS/RDR/") / (self.storage_stem + ".abrowse.jpg")
        else:
            raise ValueError(f"No abrowse exists for {self.kind}")

    @property
    def browse_path(self):
        inset = ""
        if self.kind in ["IRB", "RGB"]:
            inset = ".NOMAP"
        if self.kind not in ["COLOR", "MIRB", "MRGB", "RED", "IRB", "RGB"]:
            raise ValueError(f"No browse exists for {self.kind}")
        else:
            return Path("EXTRAS/RDR/") / (self.storage_stem + inset + ".browse.jpg")

    @property
    def thumbnail_path(self):
        if self.kind in ["BG", "IR"]:
            raise ValueError(f"No thumbnail exists for {self.kind}")
        inset = ""
        if self.kind in ["IRB", "RGB"]:
            inset = ".NOMAP"
        return Path("EXTRAS/RDR/") / (self.storage_stem + inset + ".thumb.jpg")

    @property
    def nomap_thumbnail_path(self):
        if self.kind in ["RED", "IRB", "RGB"]:
            return Path("EXTRAS/RDR") / (self.storage_stem + ".NOMAP.thumb.jpg")
        else:
            raise ValueError(f"No NOMAP thumbnail exists for {self.kind} images.")

    @property
    def nomap_browse_path(self):
        if self.kind in ["RED", "IRB", "RGB"]:
            return Path("EXTRAS/RDR") / (self.storage_stem + ".NOMAP.browse.jpg")
        else:
            raise ValueError(f"No NOMAP browse exists for {self.kind} images.")

    @property
    def edr_storage_stem(self):
        return "EDR/" + self.storage_stem

# Cell
class RGB_NOMAP:
    def __init__(self, obsid):
        self.obsid = obsid
        if self.local_path.exists():
            self.read()  # this is fine, as it's using dask chunks, cheap op

    @property
    def product_id(self):
        return self.obsid + "_COLOR"

    @property
    def filename(self):
        return self.obsid + ".JP2"

    @property
    def pid(self):
        pid = prod.PRODUCT_ID(self.obsid)
        pid.kind = "RGB"
        return pid

    @property
    def meta(self):
        s = rdrindex.query("PRODUCT_ID == @self.product_id").squeeze()
        s.index = s.index.str.lower()
        return s

    # several things in the PDS path either have changed or I did it wrong in pyrise
    @property
    def nomap_jp2_path(self):
        p = Path("EXTRAS/") / self.pid.nomap_jp2_path.replace("/EXTRAS", "")
        return p

    @property
    def url(self):
        return baseurl / str(self.nomap_jp2_path)

    @property
    def local_path(self):
        full = self.nomap_jp2_path
        return storage_root / (f"EXTRAS/RDR/{full.parent.name}/{full.name}")

    def download(self, overwrite=False):
        self.local_path.parent.mkdir(parents=True, exist_ok=True)
        if self.local_path.exists() and not overwrite:
            print("File exists. Use `overwrite=True` to download fresh.")
            return
        url_retrieve(self.url, self.local_path)

    def read(self):
        self.da = rxr.open_rasterio(self.local_path, chunks=(1, 2024, 2024))
        return self.da

    def plot_da(self, xslice=None, yslice=None):
        if xslice is not None or yslice is not None:
            data = self.da.isel(x=xslice, y=yslice)
        else:
            data = self.da

        return data.hvplot.image(
            x="x",
            y="y",
            rasterize=True,
            widget_location="top_left",
            cmap="gray",
            frame_height=800,
            frame_width=800,
            flip_yaxis=True,
        )

# Cell


class RGB_NOMAPCollection:
    """Class to deal with a set of RGB_NOMAP products."""

    def __init__(self, obsids):
        self.obsids = obsids

    def get_urls(self):
        """Get URLs for list of obsids.

        Returns
        -------
        List[yarl.URL]
            List of URL objects with the respective PDS URL for download.
        """
        urls = []
        for obsid in self.obsids:
            rgb = RGB_NOMAP(obsid)
            urls.append(rgb.url)
        self.urls = urls
        return urls

    @property
    def local_paths(self):
        paths = []
        for obsid in self.obsids:
            rgb = RGB_NOMAP(obsid)
            paths.append(rgb.local_path)
        return paths

    def download_collection(self):
        lazys = []
        for obsid in self.obsids:
            rgb = RGB_NOMAP(obsid)
            lazys.append(delayed(rgb.download)())
        print("Launching parallel download...")
        compute(*lazys)
        print("Done.")