"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _JSONStream = _interopRequireDefault(require("JSONStream"));

var _debug = _interopRequireDefault(require("debug"));

var _lodash = _interopRequireDefault(require("lodash"));

var _request = _interopRequireDefault(require("request"));

var _stream = _interopRequireDefault(require("stream"));

var _url = _interopRequireDefault(require("url"));

var _zlib = _interopRequireDefault(require("zlib"));

var _streams = require("@verdaccio/streams");

var _constants = require("./constants");

var _logger = require("./logger");

var _utils = require("./utils");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const debug = (0, _debug.default)('verdaccio:up-storage');

const encode = function (thing) {
  return encodeURIComponent(thing).replace(/^%40/, '@');
};

const jsonContentType = _constants.HEADERS.JSON;
const contentTypeAccept = `${jsonContentType};`;
/**
 * Just a helper (`config[key] || default` doesn't work because of zeroes)
 */

const setConfig = (config, key, def) => {
  return _lodash.default.isNil(config[key]) === false ? config[key] : def;
};
/**
 * Implements Storage interface
 * (same for storage.js, local-storage.js, up-storage.js)
 */


class ProxyStorage {
  // FIXME: upname is assigned to each instance
  // @ts-ignore
  // FIXME: proxy can be boolean or object, something smells here
  // @ts-ignore
  // @ts-ignore

  /**
   * Constructor
   * @param {*} config
   * @param {*} mainConfig
   */
  constructor(config, mainConfig) {
    _defineProperty(this, "config", void 0);

    _defineProperty(this, "failed_requests", void 0);

    _defineProperty(this, "userAgent", void 0);

    _defineProperty(this, "ca", void 0);

    _defineProperty(this, "logger", void 0);

    _defineProperty(this, "server_id", void 0);

    _defineProperty(this, "url", void 0);

    _defineProperty(this, "maxage", void 0);

    _defineProperty(this, "timeout", void 0);

    _defineProperty(this, "max_fails", void 0);

    _defineProperty(this, "fail_timeout", void 0);

    _defineProperty(this, "agent_options", void 0);

    _defineProperty(this, "upname", void 0);

    _defineProperty(this, "proxy", void 0);

    _defineProperty(this, "last_request_time", void 0);

    _defineProperty(this, "strict_ssl", void 0);

    this.config = config;
    this.failed_requests = 0; // @ts-ignore

    this.userAgent = mainConfig.user_agent;
    this.ca = config.ca;
    this.logger = _logger.logger;
    this.server_id = mainConfig.server_id;
    this.url = _url.default.parse(this.config.url); // $FlowFixMe

    this._setupProxy(this.url.hostname, config, mainConfig, this.url.protocol === 'https:');

    this.config.url = this.config.url.replace(/\/$/, '');

    if (this.config.timeout && Number(this.config.timeout) >= 1000) {
      this.logger.warn(['Too big timeout value: ' + this.config.timeout, 'We changed time format to nginx-like one', '(see http://nginx.org/en/docs/syntax.html)', 'so please update your config accordingly'].join('\n'));
    } // a bunch of different configurable timers


    this.maxage = (0, _utils.parseInterval)(setConfig(this.config, 'maxage', '2m'));
    this.timeout = (0, _utils.parseInterval)(setConfig(this.config, 'timeout', '30s'));
    this.max_fails = Number(setConfig(this.config, 'max_fails', 2));
    this.fail_timeout = (0, _utils.parseInterval)(setConfig(this.config, 'fail_timeout', '5m'));
    this.strict_ssl = Boolean(setConfig(this.config, 'strict_ssl', true));
    this.agent_options = setConfig(this.config, 'agent_options', {
      keepAlive: true,
      maxSockets: 40,
      maxFreeSockets: 10
    });
  }
  /**
   * Fetch an asset.
   * @param {*} options
   * @param {*} cb
   * @return {Request}
   */


  request(options, cb) {
    let json;

    if (this._statusCheck() === false) {
      const streamRead = new _stream.default.Readable();
      process.nextTick(function () {
        if (cb) {
          cb(_utils.ErrorCode.getInternalError(_constants.API_ERROR.UPLINK_OFFLINE));
        }

        streamRead.emit('error', _utils.ErrorCode.getInternalError(_constants.API_ERROR.UPLINK_OFFLINE));
      }); // $FlowFixMe

      streamRead._read = function () {}; // preventing 'Uncaught, unspecified "error" event'


      streamRead.on('error', function () {});
      return streamRead;
    }

    const self = this;

    const headers = this._setHeaders(options);

    this._addProxyHeaders(options.req, headers);

    this._overrideWithUpLinkConfLocaligHeaders(headers);

    const method = options.method || 'GET';
    const uri = options.uri_full || this.config.url + options.uri;
    self.logger.info({
      method: method,
      headers: headers,
      uri: uri
    }, "making request: '@{method} @{uri}'");

    if ((0, _utils.isObject)(options.json)) {
      json = JSON.stringify(options.json);
      headers['Content-Type'] = headers['Content-Type'] || _constants.HEADERS.JSON;
    }

    const requestCallback = cb ? function (err, res, body) {
      let error;
      const responseLength = err ? 0 : body.length; // $FlowFixMe

      processBody();
      logActivity(); // $FlowFixMe

      cb(err, res, body);
      /**
       * Perform a decode.
       */

      function processBody() {
        if (err) {
          error = err.message;
          return;
        }

        if (options.json && res.statusCode < 300) {
          try {
            // $FlowFixMe
            body = JSON.parse(body.toString(_constants.CHARACTER_ENCODING.UTF8));
          } catch (_err) {
            body = {};
            err = _err;
            error = err.message;
          }
        }

        if (!err && (0, _utils.isObject)(body)) {
          if (_lodash.default.isString(body.error)) {
            error = body.error;
          }
        }
      }
      /**
       * Perform a log.
       */


      function logActivity() {
        let message = "@{!status}, req: '@{request.method} @{request.url}'";
        message += error ? ', error: @{!error}' : ', bytes: @{bytes.in}/@{bytes.out}';
        self.logger.http({
          err: err || undefined,
          // if error is null/false change this to undefined so it wont log
          request: {
            method: method,
            url: uri
          },
          status: res != null ? res.statusCode : 'ERR',
          error: error,
          bytes: {
            in: json ? json.length : 0,
            out: responseLength || 0
          }
        }, message);
      }
    } : undefined;
    let requestOptions = {
      url: uri,
      method: method,
      headers: headers,
      body: json,
      proxy: this.proxy,
      encoding: null,
      gzip: true,
      timeout: this.timeout,
      strictSSL: this.strict_ssl,
      agentOptions: this.agent_options
    };

    if (this.ca) {
      requestOptions = Object.assign({}, requestOptions, {
        ca: this.ca
      });
    }

    const req = (0, _request.default)(requestOptions, requestCallback);
    let statusCalled = false;
    req.on('response', function (res) {
      // FIXME: _verdaccio_aborted seems not used
      // @ts-ignore
      if (!req._verdaccio_aborted && !statusCalled) {
        statusCalled = true;

        self._statusCheck(true);
      }

      if (_lodash.default.isNil(requestCallback) === false) {
        (function do_log() {
          const message = "@{!status}, req: '@{request.method} @{request.url}' (streaming)";
          self.logger.http({
            request: {
              method: method,
              url: uri
            },
            status: _lodash.default.isNull(res) === false ? res.statusCode : 'ERR'
          }, message);
        })();
      }
    });
    req.on('error', function (_err) {
      // FIXME: _verdaccio_aborted seems not used
      // @ts-ignore
      if (!req._verdaccio_aborted && !statusCalled) {
        statusCalled = true;

        self._statusCheck(false);
      }
    }); // @ts-ignore

    return req;
  }
  /**
   * Set default headers.
   * @param {Object} options
   * @return {Object}
   * @private
   */


  _setHeaders(options) {
    var _options$req;

    const headers = options.headers || {};
    const accept = _constants.HEADERS.ACCEPT;
    const acceptEncoding = _constants.HEADERS.ACCEPT_ENCODING;
    const userAgent = _constants.HEADERS.USER_AGENT;
    headers[accept] = headers[accept] || contentTypeAccept;
    headers[acceptEncoding] = headers[acceptEncoding] || 'gzip'; // registry.npmjs.org will only return search result if user-agent include string 'npm'

    headers[userAgent] = this.userAgent ? `npm (${this.userAgent})` : (_options$req = options.req) === null || _options$req === void 0 ? void 0 : _options$req.get('user-agent');
    return this._setAuth(headers);
  }
  /**
   * Validate configuration auth and assign Header authorization
   * @param {Object} headers
   * @return {Object}
   * @private
   */


  _setAuth(headers) {
    const {
      auth
    } = this.config;

    if (_lodash.default.isNil(auth) || headers[_constants.HEADERS.AUTHORIZATION]) {
      return headers;
    }

    if (_lodash.default.isObject(auth) === false && _lodash.default.isObject(auth.token) === false) {
      this._throwErrorAuth('Auth invalid');
    } // get NPM_TOKEN http://blog.npmjs.org/post/118393368555/deploying-with-npm-private-modules
    // or get other variable export in env
    // https://github.com/verdaccio/verdaccio/releases/tag/v2.5.0


    let token;
    const tokenConf = auth;

    if (_lodash.default.isNil(tokenConf.token) === false && _lodash.default.isString(tokenConf.token)) {
      token = tokenConf.token;
    } else if (_lodash.default.isNil(tokenConf.token_env) === false) {
      if (_lodash.default.isString(tokenConf.token_env)) {
        token = process.env[tokenConf.token_env];
      } else if (_lodash.default.isBoolean(tokenConf.token_env) && tokenConf.token_env) {
        token = process.env.NPM_TOKEN;
      } else {
        this.logger.error(_constants.ERROR_CODE.token_required);

        this._throwErrorAuth(_constants.ERROR_CODE.token_required);
      }
    } else {
      token = process.env.NPM_TOKEN;
    }

    if (_lodash.default.isNil(token)) {
      this._throwErrorAuth(_constants.ERROR_CODE.token_required);
    } // define type Auth allow basic and bearer


    const type = tokenConf.type || _constants.TOKEN_BASIC;

    this._setHeaderAuthorization(headers, type, token);

    return headers;
  }
  /**
   * @param {string} message
   * @throws {Error}
   * @private
   */


  _throwErrorAuth(message) {
    this.logger.error(message);
    throw new Error(message);
  }
  /**
   * Assign Header authorization with type authentication
   * @param {Object} headers
   * @param {string} type
   * @param {string} token
   * @private
   */


  _setHeaderAuthorization(headers, type, token) {
    const _type = type.toLowerCase();

    if (_type !== _constants.TOKEN_BEARER.toLowerCase() && _type !== _constants.TOKEN_BASIC.toLowerCase()) {
      this._throwErrorAuth(`Auth type '${_type}' not allowed`);
    }

    type = _lodash.default.upperFirst(type);
    headers[_constants.HEADERS.AUTHORIZATION] = (0, _utils.buildToken)(type, token);
  }
  /**
   * It will add or override specified headers from config file.
   *
   * Eg:
   *
   * uplinks:
   npmjs:
   url: https://registry.npmjs.org/
   headers:
   Accept: "application/vnd.npm.install-v2+json; q=1.0"
   verdaccio-staging:
   url: https://mycompany.com/npm
   headers:
   Accept: "application/json"
   authorization: "Basic YourBase64EncodedCredentials=="
    * @param {Object} headers
   * @private
   */


  _overrideWithUpLinkConfLocaligHeaders(headers) {
    if (!this.config.headers) {
      return headers;
    } // add/override headers specified in the config

    /* eslint guard-for-in: 0 */


    for (const key in this.config.headers) {
      headers[key] = this.config.headers[key];
    }
  }
  /**
   * Determine whether can fetch from the provided URL
   * @param {*} url
   * @return {Boolean}
   */


  isUplinkValid(url) {
    // $FlowFixMe
    const urlParsed = _url.default.parse(url);

    const isHTTPS = urlDomainParsed => urlDomainParsed.protocol === 'https:' && (urlParsed.port === null || urlParsed.port === '443');

    const getHost = urlDomainParsed => isHTTPS(urlDomainParsed) ? urlDomainParsed.hostname : urlDomainParsed.host;

    const isMatchProtocol = urlParsed.protocol === this.url.protocol;
    const isMatchHost = getHost(urlParsed) === getHost(this.url); // @ts-ignore

    const isMatchPath = urlParsed.path.indexOf(this.url.path) === 0;
    return isMatchProtocol && isMatchHost && isMatchPath;
  }
  /**
   * Get a remote package metadata
   * @param {*} name package name
   * @param {*} options request options, eg: eTag.
   * @param {*} callback
   */


  getRemoteMetadata(name, options, callback) {
    const headers = {};

    if (_lodash.default.isNil(options.etag) === false) {
      headers['If-None-Match'] = options.etag;
      headers[_constants.HEADERS.ACCEPT] = contentTypeAccept;
    }

    this.request({
      uri: `/${encode(name)}`,
      json: true,
      headers: headers,
      req: options.req
    }, (err, res, body) => {
      if (err) {
        return callback(err);
      }

      if (res.statusCode === _constants.HTTP_STATUS.NOT_FOUND) {
        return callback(_utils.ErrorCode.getNotFound(_constants.API_ERROR.NOT_PACKAGE_UPLINK));
      }

      if (!(res.statusCode >= _constants.HTTP_STATUS.OK && res.statusCode < _constants.HTTP_STATUS.MULTIPLE_CHOICES)) {
        const error = _utils.ErrorCode.getInternalError(`${_constants.API_ERROR.BAD_STATUS_CODE}: ${res.statusCode}`); // $FlowFixMe


        error.remoteStatus = res.statusCode;
        return callback(error);
      }

      callback(null, body, res.headers.etag);
    });
  }
  /**
   * Fetch a tarball from the uplink.
   * @param {String} url
   * @return {Stream}
   */


  fetchTarball(url) {
    const stream = new _streams.ReadTarball({});
    let current_length = 0;
    let expected_length;

    stream.abort = () => {};

    const readStream = this.request({
      uri_full: url,
      encoding: null,
      headers: {
        Accept: contentTypeAccept
      }
    });
    readStream.on('response', function (res) {
      if (res.statusCode === _constants.HTTP_STATUS.NOT_FOUND) {
        return stream.emit('error', _utils.ErrorCode.getNotFound(_constants.API_ERROR.NOT_FILE_UPLINK));
      }

      if (!(res.statusCode >= _constants.HTTP_STATUS.OK && res.statusCode < _constants.HTTP_STATUS.MULTIPLE_CHOICES)) {
        return stream.emit('error', _utils.ErrorCode.getInternalError(`bad uplink status code: ${res.statusCode}`));
      }

      if (res.headers[_constants.HEADER_TYPE.CONTENT_LENGTH]) {
        expected_length = res.headers[_constants.HEADER_TYPE.CONTENT_LENGTH];
        stream.emit(_constants.HEADER_TYPE.CONTENT_LENGTH, res.headers[_constants.HEADER_TYPE.CONTENT_LENGTH]);
      }

      readStream.pipe(stream);
    });
    readStream.on('error', function (err) {
      stream.emit('error', err);
    });
    readStream.on('data', function (data) {
      current_length += data.length;
    });
    readStream.on('end', function (data) {
      if (data) {
        current_length += data.length;
      }

      if (expected_length && current_length != expected_length) {
        stream.emit('error', _utils.ErrorCode.getInternalError(_constants.API_ERROR.CONTENT_MISMATCH));
      }
    });
    return stream;
  }
  /**
   * Perform a stream search.
   * @param {*} options request options
   * @return {Stream}
   */


  search(options) {
    const transformStream = new _stream.default.PassThrough({
      objectMode: true
    });
    const requestStream = this.request({
      uri: options.req.url,
      req: options.req,
      headers: {
        // query for search
        referer: options.req.get('referer')
      }
    });

    const parsePackage = pkg => {
      if ((0, _utils.isObjectOrArray)(pkg)) {
        transformStream.emit('data', pkg);
      }
    };

    requestStream.on('response', res => {
      if (!String(res.statusCode).match(/^2\d\d$/)) {
        return transformStream.emit('error', _utils.ErrorCode.getInternalError(`bad status code ${res.statusCode} from uplink`));
      } // See https://github.com/request/request#requestoptions-callback
      // Request library will not decode gzip stream.


      let jsonStream;

      if (res.headers[_constants.HEADER_TYPE.CONTENT_ENCODING] === _constants.HEADERS.GZIP) {
        jsonStream = res.pipe(_zlib.default.createUnzip());
      } else {
        jsonStream = res;
      }

      jsonStream.pipe(_JSONStream.default.parse('*')).on('data', parsePackage);
      jsonStream.on('end', () => {
        transformStream.emit('end');
      });
    });
    requestStream.on('error', err => {
      transformStream.emit('error', err);
    });

    transformStream.abort = () => {
      // FIXME: this is clearly a potential issue
      // there is no abort method on Stream.Readable
      // @ts-ignore
      requestStream.abort();
      transformStream.emit('end');
    };

    return transformStream;
  }
  /**
   * Add proxy headers.
   * FIXME: object mutations, it should return an new object
   * @param {*} req the http request
   * @param {*} headers the request headers
   */


  _addProxyHeaders(req, headers) {
    if (req) {
      // Only submit X-Forwarded-For field if we don't have a proxy selected
      // in the config file.
      //
      // Otherwise misconfigured proxy could return 407:
      // https://github.com/rlidwka/sinopia/issues/254
      //
      // FIXME: proxy logic is odd, something is wrong here.
      // @ts-ignore
      if (!this.proxy) {
        headers['x-forwarded-for'] = (req.get('x-forwarded-for') ? req.get('x-forwarded-for') + ', ' : '') + req.connection.remoteAddress;
      }
    } // always attach Via header to avoid loops, even if we're not proxying


    headers['via'] = req && req.get('via') ? req.get('via') + ', ' : '';
    headers['via'] += '1.1 ' + this.server_id + ' (Verdaccio)';
  }
  /**
   * Check whether the remote host is available.
   * @param {*} alive
   * @return {Boolean}
   */


  _statusCheck(alive) {
    if (arguments.length === 0) {
      return this._ifRequestFailure() === false;
    }

    if (alive) {
      if (this.failed_requests >= this.max_fails) {
        this.logger.warn({
          host: this.url.host
        }, 'host @{host} is back online');
      }

      this.failed_requests = 0;
    } else {
      this.failed_requests++;

      if (this.failed_requests === this.max_fails) {
        this.logger.warn({
          host: this.url.host
        }, 'host @{host} is now offline');
      }
    }

    this.last_request_time = Date.now();
  }
  /**
   * If the request failure.
   * @return {boolean}
   * @private
   */


  _ifRequestFailure() {
    return this.failed_requests >= this.max_fails && Math.abs(Date.now() - this.last_request_time) < this.fail_timeout;
  }
  /**
   * Set up a proxy.
   * @param {*} hostname
   * @param {*} config
   * @param {*} mainconfig
   * @param {*} isHTTPS
   */


  _setupProxy(hostname, config, mainconfig, isHTTPS) {
    let noProxyList;
    const proxy_key = isHTTPS ? 'https_proxy' : 'http_proxy'; // get http_proxy and no_proxy configs

    if (proxy_key in config) {
      this.proxy = config[proxy_key];
    } else if (proxy_key in mainconfig) {
      this.proxy = mainconfig[proxy_key];
    }

    if ('no_proxy' in config) {
      // $FlowFixMe
      noProxyList = config.no_proxy;
    } else if ('no_proxy' in mainconfig) {
      noProxyList = mainconfig.no_proxy;
    } // use wget-like algorithm to determine if proxy shouldn't be used


    if (hostname[0] !== '.') {
      hostname = '.' + hostname;
    }

    if (_lodash.default.isString(noProxyList) && noProxyList.length) {
      // $FlowFixMe
      noProxyList = noProxyList.split(',');
    }

    if (_lodash.default.isArray(noProxyList)) {
      for (let i = 0; i < noProxyList.length; i++) {
        let noProxyItem = noProxyList[i];

        if (noProxyItem[0] !== '.') {
          noProxyItem = '.' + noProxyItem;
        }

        if (hostname.lastIndexOf(noProxyItem) === hostname.length - noProxyItem.length) {
          if (this.proxy) {
            debug('not using proxy for %o, excluded by %o rule', this.url.href, noProxyItem); // @ts-ignore

            this.proxy = false;
          }

          break;
        }
      }
    } // if it's non-string (i.e. "false"), don't use it


    if (_lodash.default.isString(this.proxy) === false) {
      // @ts-ignore
      delete this.proxy;
    } else {
      debug('using proxy %o for %o', this.url.href, this.proxy);
    }
  }

}

var _default = ProxyStorage;
exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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