"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _assert = _interopRequireDefault(require("assert"));

var _async = _interopRequireDefault(require("async"));

var _lodash = _interopRequireDefault(require("lodash"));

var _stream = _interopRequireDefault(require("stream"));

var _streams = require("@verdaccio/streams");

var _logger = require("../lib/logger");

var _configUtils = require("./config-utils");

var _constants = require("./constants");

var _localStorage = _interopRequireDefault(require("./local-storage"));

var _metadataUtils = require("./metadata-utils");

var _search = _interopRequireDefault(require("./search"));

var _storageUtils = require("./storage-utils");

var _upStorage = _interopRequireDefault(require("./up-storage"));

var _uplinkUtil = require("./uplink-util");

var _utils = require("./utils");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class Storage {
  constructor(config) {
    _defineProperty(this, "localStorage", void 0);

    _defineProperty(this, "config", void 0);

    _defineProperty(this, "logger", void 0);

    _defineProperty(this, "uplinks", void 0);

    _defineProperty(this, "filters", void 0);

    this.config = config;
    this.uplinks = (0, _uplinkUtil.setupUpLinks)(config);
    this.logger = _logger.logger;
    this.filters = []; // @ts-ignore

    this.localStorage = null;
  }

  init(config, filters = []) {
    this.filters = filters;
    this.localStorage = new _localStorage.default(this.config, _logger.logger);
    return this.localStorage.getSecret(config);
  }
  /**
   *  Add a {name} package to a system
   Function checks if package with the same name is available from uplinks.
   If it isn't, we create package locally
   Used storages: local (write) && uplinks
   */


  async addPackage(name, metadata, callback) {
    try {
      await (0, _storageUtils.checkPackageLocal)(name, this.localStorage);
      await (0, _storageUtils.checkPackageRemote)(name, this._isAllowPublishOffline(), this._syncUplinksMetadata.bind(this));
      await (0, _storageUtils.publishPackage)(name, metadata, this.localStorage);
      callback();
    } catch (err) {
      callback(err);
    }
  }

  _isAllowPublishOffline() {
    return typeof this.config.publish !== 'undefined' && _lodash.default.isBoolean(this.config.publish.allow_offline) && this.config.publish.allow_offline;
  }

  readTokens(filter) {
    return this.localStorage.readTokens(filter);
  }

  saveToken(token) {
    return this.localStorage.saveToken(token);
  }

  deleteToken(user, tokenKey) {
    return this.localStorage.deleteToken(user, tokenKey);
  }
  /**
   * Add a new version of package {name} to a system
   Used storages: local (write)
   */


  addVersion(name, version, metadata, tag, callback) {
    this.localStorage.addVersion(name, version, metadata, tag, callback);
  }
  /**
   * Tags a package version with a provided tag
   Used storages: local (write)
   */


  mergeTags(name, tagHash, callback) {
    this.localStorage.mergeTags(name, tagHash, callback);
  }
  /**
   * Change an existing package (i.e. unpublish one version)
   Function changes a package info from local storage and all uplinks with write access./
   Used storages: local (write)
   */


  changePackage(name, metadata, revision, callback) {
    this.localStorage.changePackage(name, metadata, revision, callback);
  }
  /**
   * Remove a package from a system
   Function removes a package from local storage
   Used storages: local (write)
   */


  removePackage(name, callback) {
    this.localStorage.removePackage(name, callback); // update the indexer

    _search.default.remove(name);
  }
  /**
   Remove a tarball from a system
   Function removes a tarball from local storage.
   Tarball in question should not be linked to in any existing
   versions, i.e. package version should be unpublished first.
   Used storage: local (write)
   */


  removeTarball(name, filename, revision, callback) {
    this.localStorage.removeTarball(name, filename, revision, callback);
  }
  /**
   * Upload a tarball for {name} package
   Function is synchronous and returns a WritableStream
   Used storages: local (write)
   */


  addTarball(name, filename) {
    return this.localStorage.addTarball(name, filename);
  }

  hasLocalTarball(name, filename) {
    const self = this;
    return new Promise((resolve, reject) => {
      let localStream = self.localStorage.getTarball(name, filename);
      let isOpen = false;
      localStream.on('error', err => {
        if (isOpen || err.status !== _constants.HTTP_STATUS.NOT_FOUND) {
          reject(err);
        } // local reported 404 or request was aborted already


        if (localStream) {
          localStream.abort();
          localStream = null;
        }

        resolve(false);
      });
      localStream.on('open', function () {
        isOpen = true;
        localStream.abort();
        localStream = null;
        resolve(true);
      });
    });
  }
  /**
   Get a tarball from a storage for {name} package
   Function is synchronous and returns a ReadableStream
   Function tries to read tarball locally, if it fails then it reads package
   information in order to figure out where we can get this tarball from
   Used storages: local || uplink (just one)
   */


  getTarball(name, filename) {
    const readStream = new _streams.ReadTarball({});

    readStream.abort = function () {};

    const self = this; // if someone requesting tarball, it means that we should already have some
    // information about it, so fetching package info is unnecessary
    // trying local first
    // flow: should be IReadTarball

    let localStream = self.localStorage.getTarball(name, filename);
    let isOpen = false;
    localStream.on('error', err => {
      if (isOpen || err.status !== _constants.HTTP_STATUS.NOT_FOUND) {
        return readStream.emit('error', err);
      } // local reported 404


      const err404 = err;
      localStream.abort();
      localStream = null; // we force for garbage collector

      self.localStorage.getPackageMetadata(name, (err, info) => {
        if (_lodash.default.isNil(err) && info._distfiles && _lodash.default.isNil(info._distfiles[filename]) === false) {
          // information about this file exists locally
          serveFile(info._distfiles[filename]);
        } else {
          // we know nothing about this file, trying to get information elsewhere
          self._syncUplinksMetadata(name, info, {}, (err, info) => {
            if (_lodash.default.isNil(err) === false) {
              return readStream.emit('error', err);
            }

            if (_lodash.default.isNil(info._distfiles) || _lodash.default.isNil(info._distfiles[filename])) {
              return readStream.emit('error', err404);
            }

            serveFile(info._distfiles[filename]);
          });
        }
      });
    });
    localStream.on('content-length', function (v) {
      readStream.emit('content-length', v);
    });
    localStream.on('open', function () {
      isOpen = true;
      localStream.pipe(readStream);
    });
    return readStream;
    /**
     * Fetch and cache local/remote packages.
     * @param {Object} file define the package shape
     */

    function serveFile(file) {
      let uplink = null;

      for (const uplinkId in self.uplinks) {
        if ((0, _configUtils.hasProxyTo)(name, uplinkId, self.config.packages)) {
          uplink = self.uplinks[uplinkId];
        }
      }

      if (uplink == null) {
        uplink = new _upStorage.default({
          url: file.url,
          cache: true,
          _autogenerated: true
        }, self.config);
      }

      let savestream = null;

      if (uplink.config.cache) {
        savestream = self.localStorage.addTarball(name, filename);
      }

      let on_open = function () {
        // prevent it from being called twice
        on_open = function () {};

        const rstream2 = uplink.fetchTarball(file.url);
        rstream2.on('error', function (err) {
          if (savestream) {
            savestream.abort();
          }

          savestream = null;
          readStream.emit('error', err);
        });
        rstream2.on('end', function () {
          if (savestream) {
            savestream.done();
          }
        });
        rstream2.on('content-length', function (v) {
          readStream.emit('content-length', v);

          if (savestream) {
            savestream.emit('content-length', v);
          }
        });
        rstream2.pipe(readStream);

        if (savestream) {
          rstream2.pipe(savestream);
        }
      };

      if (savestream) {
        savestream.on('open', function () {
          on_open();
        });
        savestream.on('error', function (err) {
          self.logger.warn({
            err: err,
            fileName: file
          }, 'error saving file @{fileName}: @{err.message}\n@{err.stack}');

          if (savestream) {
            savestream.abort();
          }

          savestream = null;
          on_open();
        });
      } else {
        on_open();
      }
    }
  }
  /**
   Retrieve a package metadata for {name} package
   Function invokes localStorage.getPackage and uplink.get_package for every
   uplink with proxy_access rights against {name} and combines results
   into one json object
   Used storages: local && uplink (proxy_access)
    * @param {object} options
   * @property {string} options.name Package Name
   * @property {object}  options.req Express `req` object
   * @property {boolean} options.keepUpLinkData keep up link info in package meta, last update, etc.
   * @property {function} options.callback Callback for receive data
   */


  getPackage(options) {
    this.localStorage.getPackageMetadata(options.name, (err, data) => {
      if (err && (!err.status || err.status >= _constants.HTTP_STATUS.INTERNAL_ERROR)) {
        // report internal errors right away
        return options.callback(err);
      }

      this._syncUplinksMetadata(options.name, data, {
        req: options.req,
        uplinksLook: options.uplinksLook
      }, function getPackageSynUpLinksCallback(err, result, uplinkErrors) {
        if (err) {
          return options.callback(err);
        }

        (0, _utils.normalizeDistTags)((0, _storageUtils.cleanUpLinksRef)(options.keepUpLinkData, result)); // npm can throw if this field doesn't exist

        result._attachments = {};

        if (options.abbreviated === true) {
          options.callback(null, (0, _storageUtils.convertAbbreviatedManifest)(result), uplinkErrors);
        } else {
          options.callback(null, result, uplinkErrors);
        }
      });
    });
  }
  /**
   Retrieve remote and local packages more recent than {startkey}
   Function streams all packages from all uplinks first, and then
   local packages.
   Note that local packages could override registry ones just because
   they appear in JSON last. That's a trade-off we make to avoid
   memory issues.
   Used storages: local && uplink (proxy_access)
   * @param {*} startkey
   * @param {*} options
   * @return {Stream}
   */


  search(startkey, options) {
    const self = this;
    const searchStream = new _stream.default.PassThrough({
      objectMode: true
    });

    _async.default.eachSeries(Object.keys(this.uplinks), function (up_name, cb) {
      var _options$req, _options$req$query;

      // shortcut: if `local=1` is supplied, don't call uplinks
      if (((_options$req = options.req) === null || _options$req === void 0 ? void 0 : (_options$req$query = _options$req.query) === null || _options$req$query === void 0 ? void 0 : _options$req$query.local) !== undefined) {
        return cb();
      }

      _logger.logger.info(`search for uplink ${up_name}`); // search by keyword for each uplink


      const uplinkStream = self.uplinks[up_name].search(options); // join uplink stream with streams PassThrough

      uplinkStream.pipe(searchStream, {
        end: false
      });
      uplinkStream.on('error', function (err) {
        self.logger.error({
          err: err
        }, 'uplink error: @{err.message}');
        cb(); // to avoid call callback more than once

        cb = function () {};
      });
      uplinkStream.on('end', function () {
        cb(); // to avoid call callback more than once

        cb = function () {};
      });

      searchStream.abort = function () {
        if (uplinkStream.abort) {
          uplinkStream.abort();
        }

        cb(); // to avoid call callback more than once

        cb = function () {};
      };
    }, // executed after all series
    function () {
      // attach a local search results
      const localSearchStream = self.localStorage.search(startkey, options);

      searchStream.abort = function () {
        localSearchStream.abort();
      };

      localSearchStream.pipe(searchStream, {
        end: true
      });
      localSearchStream.on('error', function (err) {
        self.logger.error({
          err: err
        }, 'search error: @{err.message}');
        searchStream.end();
      });
    });

    return searchStream;
  }
  /**
   * Retrieve only private local packages
   * @param {*} callback
   */


  getLocalDatabase(callback) {
    const self = this;
    this.localStorage.storagePlugin.get((err, locals) => {
      if (err) {
        callback(err);
      }

      const packages = [];

      const getPackage = function (itemPkg) {
        self.localStorage.getPackageMetadata(locals[itemPkg], function (err, pkgMetadata) {
          if (_lodash.default.isNil(err)) {
            const latest = pkgMetadata[_constants.DIST_TAGS].latest;

            if (latest && pkgMetadata.versions[latest]) {
              const version = pkgMetadata.versions[latest];
              const timeList = pkgMetadata.time;
              const time = timeList[latest]; // @ts-ignore

              version.time = time; // Add for stars api
              // @ts-ignore

              version.users = pkgMetadata.users;
              packages.push(version);
            } else {
              self.logger.warn({
                package: locals[itemPkg]
              }, 'package @{package} does not have a "latest" tag?');
            }
          }

          if (itemPkg >= locals.length - 1) {
            callback(null, packages);
          } else {
            getPackage(itemPkg + 1);
          }
        });
      };

      if (locals.length) {
        getPackage(0);
      } else {
        callback(null, []);
      }
    });
  }
  /**
   * Function fetches package metadata from uplinks and synchronizes it with local data
   if package is available locally, it MUST be provided in pkginfo
   returns callback(err, result, uplink_errors)
   */


  _syncUplinksMetadata(name, packageInfo, options, callback) {
    let found = true;
    const self = this;
    const upLinks = [];
    const hasToLookIntoUplinks = _lodash.default.isNil(options.uplinksLook) || options.uplinksLook;

    if (!packageInfo) {
      found = false;
      packageInfo = (0, _storageUtils.generatePackageTemplate)(name);
    }

    for (const uplink in this.uplinks) {
      if ((0, _configUtils.hasProxyTo)(name, uplink, this.config.packages) && hasToLookIntoUplinks) {
        upLinks.push(this.uplinks[uplink]);
      }
    }

    _async.default.map(upLinks, (upLink, cb) => {
      const _options = Object.assign({}, options);

      const upLinkMeta = packageInfo._uplinks[upLink.upname];

      if ((0, _utils.isObject)(upLinkMeta)) {
        const fetched = upLinkMeta.fetched;

        if (fetched && Date.now() - fetched < upLink.maxage) {
          return cb();
        }

        _options.etag = upLinkMeta.etag;
      }

      upLink.getRemoteMetadata(name, _options, (err, upLinkResponse, eTag) => {
        if (err && err.remoteStatus === 304) {
          upLinkMeta.fetched = Date.now();
        }

        if (err || !upLinkResponse) {
          return cb(null, [err || _utils.ErrorCode.getInternalError('no data')]);
        }

        try {
          (0, _utils.validateMetadata)(upLinkResponse, name);
        } catch (err) {
          self.logger.error({
            sub: 'out',
            err: err
          }, 'package.json validating error @{!err.message}\n@{err.stack}');
          return cb(null, [err]);
        }

        packageInfo._uplinks[upLink.upname] = {
          etag: eTag,
          fetched: Date.now()
        };
        packageInfo.time = (0, _storageUtils.mergeUplinkTimeIntoLocal)(packageInfo, upLinkResponse);
        (0, _uplinkUtil.updateVersionsHiddenUpLink)(upLinkResponse.versions, upLink);

        try {
          (0, _metadataUtils.mergeVersions)(packageInfo, upLinkResponse);
        } catch (err) {
          self.logger.error({
            sub: 'out',
            err: err
          }, 'package.json parsing error @{!err.message}\n@{err.stack}');
          return cb(null, [err]);
        } // if we got to this point, assume that the correct package exists
        // on the uplink


        found = true;
        cb();
      });
    }, // @ts-ignore
    (err, upLinksErrors) => {
      (0, _assert.default)(!err && Array.isArray(upLinksErrors)); // Check for connection timeout or reset errors with uplink(s)
      // (these should be handled differently from the package not being found)

      if (!found) {
        let uplinkTimeoutError;

        for (let i = 0; i < upLinksErrors.length; i++) {
          if (upLinksErrors[i]) {
            for (let j = 0; j < upLinksErrors[i].length; j++) {
              if (upLinksErrors[i][j]) {
                const code = upLinksErrors[i][j].code;

                if (code === 'ETIMEDOUT' || code === 'ESOCKETTIMEDOUT' || code === 'ECONNRESET') {
                  uplinkTimeoutError = true;
                  break;
                }
              }
            }
          }
        }

        if (uplinkTimeoutError) {
          return callback(_utils.ErrorCode.getServiceUnavailable(), null, upLinksErrors);
        }

        return callback(_utils.ErrorCode.getNotFound(_constants.API_ERROR.NO_PACKAGE), null, upLinksErrors);
      }

      if (upLinks.length === 0) {
        return callback(null, packageInfo);
      }

      self.localStorage.updateVersions(name, packageInfo, async (err, packageJsonLocal) => {
        if (err) {
          return callback(err);
        } // Any error here will cause a 404, like an uplink error. This is likely the right thing to do
        // as a broken filter is a security risk.


        const filterErrors = []; // This MUST be done serially and not in parallel as they modify packageJsonLocal

        for (const filter of self.filters) {
          try {
            // These filters can assume it's save to modify packageJsonLocal and return it directly for
            // performance (i.e. need not be pure)
            packageJsonLocal = await filter.filter_metadata(packageJsonLocal);
          } catch (err) {
            filterErrors.push(err);
          }
        }

        callback(null, packageJsonLocal, _lodash.default.concat(upLinksErrors, filterErrors));
      });
    });
  }
  /**
   * Set a hidden value for each version.
   * @param {Array} versions list of version
   * @param {String} upLink uplink name
   * @private
   */


  _updateVersionsHiddenUpLink(versions, upLink) {
    for (const i in versions) {
      if (Object.prototype.hasOwnProperty.call(versions, i)) {
        const version = versions[i]; // holds a "hidden" value to be used by the package storage.
        // $FlowFixMe

        version[Symbol.for('__verdaccio_uplink')] = upLink.upname;
      }
    }
  }

}

var _default = Storage;
exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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