"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WHITELIST = void 0;
exports.checkPackageLocal = checkPackageLocal;
exports.checkPackageRemote = checkPackageRemote;
exports.cleanUpLinksRef = cleanUpLinksRef;
exports.cleanUpReadme = cleanUpReadme;
exports.convertAbbreviatedManifest = convertAbbreviatedManifest;
exports.generatePackageTemplate = generatePackageTemplate;
exports.generateRevision = generateRevision;
exports.getLatestReadme = getLatestReadme;
exports.isPublishablePackage = isPublishablePackage;
exports.mergeUplinkTimeIntoLocal = mergeUplinkTimeIntoLocal;
exports.normalizeContributors = normalizeContributors;
exports.normalizePackage = normalizePackage;
exports.prepareSearchPackage = prepareSearchPackage;
exports.publishPackage = publishPackage;

var _lodash = _interopRequireDefault(require("lodash"));

var _cryptoUtils = require("../lib/crypto-utils");

var _constants = require("./constants");

var _search = _interopRequireDefault(require("./search"));

var _utils = require("./utils");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function generatePackageTemplate(name) {
  return {
    // standard things
    name,
    versions: {},
    time: {},
    [_constants.USERS]: {},
    [_constants.DIST_TAGS]: {},
    _uplinks: {},
    _distfiles: {},
    _attachments: {},
    _rev: ''
  };
}
/**
 * Normalize package properties, tags, revision id.
 * @param {Object} pkg package reference.
 */


function normalizePackage(pkg) {
  const pkgProperties = ['versions', 'dist-tags', '_distfiles', '_attachments', '_uplinks', 'time'];
  pkgProperties.forEach(key => {
    const pkgProp = pkg[key];

    if (_lodash.default.isNil(pkgProp) || (0, _utils.isObject)(pkgProp) === false) {
      pkg[key] = {};
    }
  });

  if (_lodash.default.isString(pkg._rev) === false) {
    pkg._rev = _constants.STORAGE.DEFAULT_REVISION;
  }

  if (_lodash.default.isString(pkg._id) === false) {
    pkg._id = pkg.name;
  } // normalize dist-tags


  (0, _utils.normalizeDistTags)(pkg);
  return pkg;
}

function generateRevision(rev) {
  const _rev = rev.split('-');

  return (+_rev[0] || 0) + 1 + '-' + (0, _cryptoUtils.generateRandomHexString)();
}

function getLatestReadme(pkg) {
  const versions = pkg['versions'] || {};
  const distTags = pkg[_constants.DIST_TAGS] || {}; // FIXME: here is a bit tricky add the types

  const latestVersion = distTags['latest'] ? versions[distTags['latest']] || {} : {};

  let readme = _lodash.default.trim(pkg.readme || latestVersion.readme || '');

  if (readme) {
    return readme;
  } // In case of empty readme - trying to get ANY readme in the following order: 'next','beta','alpha','test','dev','canary'


  const readmeDistTagsPriority = ['next', 'beta', 'alpha', 'test', 'dev', 'canary'];
  readmeDistTagsPriority.map(function (tag) {
    if (readme) {
      return readme;
    }

    const version = distTags[tag] ? versions[distTags[tag]] || {} : {};
    readme = _lodash.default.trim(version.readme || readme);
  });
  return readme;
}

function cleanUpReadme(version) {
  if (_lodash.default.isNil(version) === false) {
    // @ts-ignore
    delete version.readme;
  }

  return version;
}

function normalizeContributors(contributors) {
  if (_lodash.default.isNil(contributors)) {
    return [];
  } else if (contributors && _lodash.default.isArray(contributors) === false) {
    // FIXME: this branch is clearly no an array, still tsc complains
    // @ts-ignore
    return [contributors];
  } else if (_lodash.default.isString(contributors)) {
    return [{
      name: contributors
    }];
  }

  return contributors;
}

const WHITELIST = ['_rev', 'name', 'versions', 'dist-tags', 'readme', 'time', '_id', 'users'];
exports.WHITELIST = WHITELIST;

function cleanUpLinksRef(keepUpLinkData, result) {
  const propertyToKeep = [...WHITELIST];

  if (keepUpLinkData === true) {
    propertyToKeep.push('_uplinks');
  }

  for (const i in result) {
    if (propertyToKeep.indexOf(i) === -1) {
      // Remove sections like '_uplinks' from response
      delete result[i];
    }
  }

  return result;
}
/**
 * Check whether a package it is already a local package
 * @param {*} name
 * @param {*} localStorage
 */


function checkPackageLocal(name, localStorage) {
  return new Promise((resolve, reject) => {
    localStorage.getPackageMetadata(name, (err, results) => {
      if (!_lodash.default.isNil(err) && err.status !== _constants.HTTP_STATUS.NOT_FOUND) {
        return reject(err);
      }

      if (results) {
        return reject(_utils.ErrorCode.getConflict(_constants.API_ERROR.PACKAGE_EXIST));
      }

      return resolve();
    });
  });
}

function publishPackage(name, metadata, localStorage) {
  return new Promise((resolve, reject) => {
    localStorage.addPackage(name, metadata, (err, latest) => {
      if (!_lodash.default.isNull(err)) {
        return reject(err);
      } else if (!_lodash.default.isUndefined(latest)) {
        _search.default.add(latest);
      }

      return resolve();
    });
  });
}

function checkPackageRemote(name, isAllowPublishOffline, syncMetadata) {
  return new Promise((resolve, reject) => {
    syncMetadata(name, null, {}, (err, packageJsonLocal, upLinksErrors) => {
      // something weird
      if (err && err.status !== _constants.HTTP_STATUS.NOT_FOUND) {
        return reject(err);
      } // checking package exist already


      if (_lodash.default.isNil(packageJsonLocal) === false) {
        return reject(_utils.ErrorCode.getConflict(_constants.API_ERROR.PACKAGE_EXIST));
      }

      for (let errorItem = 0; errorItem < upLinksErrors.length; errorItem++) {
        // checking error
        // if uplink fails with a status other than 404, we report failure
        if (_lodash.default.isNil(upLinksErrors[errorItem][0]) === false) {
          if (upLinksErrors[errorItem][0].status !== _constants.HTTP_STATUS.NOT_FOUND) {
            if (isAllowPublishOffline) {
              return resolve();
            }

            return reject(_utils.ErrorCode.getServiceUnavailable(_constants.API_ERROR.UPLINK_OFFLINE_PUBLISH));
          }
        }
      }

      return resolve();
    });
  });
}

function mergeUplinkTimeIntoLocal(localMetadata, remoteMetadata) {
  if ('time' in remoteMetadata) {
    return Object.assign({}, localMetadata.time, remoteMetadata.time);
  }

  return localMetadata.time;
}

function prepareSearchPackage(data, time) {
  const listVersions = Object.keys(data.versions);
  const versions = (0, _utils.semverSort)(listVersions);
  const latest = data[_constants.DIST_TAGS] && data[_constants.DIST_TAGS].latest ? data[_constants.DIST_TAGS].latest : versions.pop();

  if (latest && data.versions[latest]) {
    const version = data.versions[latest];
    const versions = {
      [latest]: 'latest'
    };
    const pkg = {
      name: version.name,
      description: version.description,
      [_constants.DIST_TAGS]: {
        latest
      },
      maintainers: version.maintainers || [version.author].filter(Boolean),
      author: version.author,
      repository: version.repository,
      readmeFilename: version.readmeFilename || '',
      homepage: version.homepage,
      keywords: version.keywords,
      bugs: version.bugs,
      license: version.license,
      time: {
        modified: time
      },
      versions
    };
    return pkg;
  }
}
/**
 * Check whether the package metadta has enough data to be published
 * @param pkg metadata
 */


function isPublishablePackage(pkg) {
  const keys = Object.keys(pkg);
  return _lodash.default.includes(keys, 'versions');
}

function convertAbbreviatedManifest(manifest) {
  var _manifest$time;

  const abbreviatedVersions = Object.keys(manifest.versions).reduce((acc, version) => {
    const _version = manifest.versions[version]; // This should be align with this document
    // https://github.com/npm/registry/blob/master/docs/responses/package-metadata.md#abbreviated-version-object

    const _version_abbreviated = {
      name: _version.name,
      version: _version.version,
      description: _version.description,
      deprecated: _version.deprecated,
      bin: _version.bin,
      dist: _version.dist,
      engines: _version.engines,
      cpu: _version.cpu,
      os: _version.os,
      funding: _version.funding,
      directories: _version.directories,
      dependencies: _version.dependencies,
      devDependencies: _version.devDependencies,
      peerDependencies: _version.peerDependencies,
      peerDependenciesMeta: _version.peerDependenciesMeta,
      optionalDependencies: _version.optionalDependencies,
      bundleDependencies: _version.bundleDependencies,
      // npm cli specifics
      _hasShrinkwrap: _version._hasShrinkwrap,
      hasInstallScript: _version.hasInstallScript
    };
    acc[version] = _version_abbreviated;
    return acc;
  }, {});
  const convertedManifest = {
    name: manifest['name'],
    [_constants.DIST_TAGS]: manifest[_constants.DIST_TAGS],
    versions: abbreviatedVersions,
    // @ts-ignore
    modified: manifest === null || manifest === void 0 ? void 0 : (_manifest$time = manifest.time) === null || _manifest$time === void 0 ? void 0 : _manifest$time.modified,
    // NOTE: special case for pnpm https://github.com/pnpm/rfcs/pull/2
    time: manifest === null || manifest === void 0 ? void 0 : manifest.time
  }; // @ts-ignore

  return convertedManifest;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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