"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = loadPlugin;

var _debug = _interopRequireDefault(require("debug"));

var _lodash = _interopRequireDefault(require("lodash"));

var _path = _interopRequireDefault(require("path"));

var _constants = require("./constants");

var _logger = require("./logger");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const debug = (0, _debug.default)('verdaccio:plugin:loader');
/**
 * Requires a module.
 * @param {*} path the module's path
 * @return {Object}
 */

function tryLoad(path) {
  try {
    debug('loading plugin %s', path);
    return require(path);
  } catch (err) {
    if (err.code === _constants.MODULE_NOT_FOUND) {
      debug('plugin %s not found', path);
      return null;
    }

    _logger.logger.error({
      err: err.msg
    }, 'error loading plugin @{err}');

    throw err;
  }
}

function mergeConfig(appConfig, pluginConfig) {
  return _lodash.default.merge(appConfig, pluginConfig);
}

function isValid(plugin) {
  return _lodash.default.isFunction(plugin) || _lodash.default.isFunction(plugin.default);
}

function isES6(plugin) {
  return Object.keys(plugin).includes('default');
} // export type PluginGeneric<R, T extends IPlugin<R> = ;

/**
 * Load a plugin following the rules
 * - First try to load from the internal directory plugins (which will disappear soon or later).
 * - If the package is scoped eg: @scope/foo, try to load as a package
 * - A second attempt from the external plugin directory
 * - A third attempt from node_modules, in case to have multiple match as for instance verdaccio-ldap
 * and sinopia-ldap. All verdaccio prefix will have preferences.
 * @param {*} config a reference of the configuration settings
 * @param {*} pluginConfigs
 * @param {*} params a set of params to initialize the plugin
 * @param {*} sanityCheck callback that check the shape that should fulfill the plugin
 * @return {Array} list of plugins
 */


function loadPlugin(config, pluginConfigs = {}, params, sanityCheck, prefix = 'verdaccio') {
  return Object.keys(pluginConfigs).map(pluginId => {
    let plugin;
    const isScoped = pluginId.startsWith('@') && pluginId.includes('/');
    debug('isScoped %s', isScoped);

    if (isScoped) {
      plugin = tryLoad(pluginId);
    }

    const localPlugin = _path.default.resolve(__dirname + '/../plugins', pluginId); // try local plugins first


    plugin = tryLoad(localPlugin); // try the external plugin directory

    if (plugin === null && config.plugins) {
      const pluginDir = config.plugins;

      const externalFilePlugin = _path.default.resolve(pluginDir, pluginId);

      plugin = tryLoad(externalFilePlugin); // npm package

      if (plugin === null && pluginId.match(/^[^\.\/]/)) {
        plugin = tryLoad(_path.default.resolve(pluginDir, `${prefix}-${pluginId}`)); // compatibility for old sinopia plugins

        if (!plugin) {
          plugin = tryLoad(_path.default.resolve(pluginDir, `sinopia-${pluginId}`));

          if (plugin) {
            _logger.logger.warn({
              name: pluginId
            }, `plugin names that start with sinopia-* will be removed in the future, please rename package to verdaccio-*`);
          }
        }
      }
    } // npm package


    if (plugin === null && pluginId.match(/^[^\.\/]/)) {
      plugin = tryLoad(`${prefix}-${pluginId}`); // compatibility for old sinopia plugins

      if (!plugin) {
        plugin = tryLoad(`sinopia-${pluginId}`);
      }

      if (plugin) {
        debug('plugin %s is an npm package', pluginId);
      }
    }

    if (plugin === null) {
      plugin = tryLoad(pluginId);
    } // relative to config path


    if (plugin === null && pluginId.match(/^\.\.?($|\/)/)) {
      plugin = tryLoad(_path.default.resolve(_path.default.dirname(config.self_path), pluginId));
    }

    if (plugin === null) {
      if (isScoped) {
        _logger.logger.error({
          content: pluginId
        }, 'plugin not found. try npm install @{content}');
      } else {
        _logger.logger.error({
          content: pluginId,
          prefix
        }, 'plugin not found. try npm install @{prefix}-@{content}');
      }

      const msg = isScoped ? `
      ${pluginId} plugin not found. try "npm install ${pluginId}"` : `
      ${prefix}-${pluginId} plugin not found. try "npm install ${prefix}-${pluginId}"`;
      throw Error(msg);
    }

    if (!isValid(plugin)) {
      _logger.logger.error({
        content: pluginId
      }, '@{prefix}-@{content} plugin does not have the right code structure');

      throw Error(`"${pluginId}" plugin does not have the right code structure`);
    }
    /* eslint new-cap:off */


    try {
      if (isES6(plugin)) {
        debug('plugin is ES6');
        plugin = new plugin.default(mergeConfig(config, pluginConfigs[pluginId]), params);
      } else {
        debug('plugin is commonJS');
        plugin = plugin(pluginConfigs[pluginId], params);
      }
    } catch (error) {
      plugin = null;

      _logger.logger.error({
        error,
        pluginId
      }, 'error loading a plugin @{pluginId}: @{error}');
    }
    /* eslint new-cap:off */


    if (plugin === null || !sanityCheck(plugin)) {
      if (isScoped) {
        _logger.logger.error({
          content: pluginId
        }, "@{content} doesn't look like a valid plugin");
      } else {
        _logger.logger.error({
          content: pluginId,
          prefix
        }, "@{prefix}-@{content} doesn't look like a valid plugin");
      }

      throw Error(`sanity check has failed, "${pluginId}" is not a valid plugin`);
    }

    if (isScoped) {
      _logger.logger.info({
        content: pluginId
      }, 'plugin successfully loaded: @{content}');
    } else {
      _logger.logger.info({
        content: pluginId,
        prefix
      }, 'plugin successfully loaded: @{prefix}-@{content}');
    }

    return plugin;
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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