"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createLogger = createLogger;
exports.getLogger = getLogger;
exports.logger = void 0;
exports.setup = setup;

var _debug = _interopRequireDefault(require("debug"));

var _lodash = _interopRequireDefault(require("lodash"));

var _pino = _interopRequireDefault(require("pino"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function ownKeys(object, enumerableOnly) { var keys = Object.keys(object); if (Object.getOwnPropertySymbols) { var symbols = Object.getOwnPropertySymbols(object); enumerableOnly && (symbols = symbols.filter(function (sym) { return Object.getOwnPropertyDescriptor(object, sym).enumerable; })), keys.push.apply(keys, symbols); } return keys; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = null != arguments[i] ? arguments[i] : {}; i % 2 ? ownKeys(Object(source), !0).forEach(function (key) { _defineProperty(target, key, source[key]); }) : Object.getOwnPropertyDescriptors ? Object.defineProperties(target, Object.getOwnPropertyDescriptors(source)) : ownKeys(Object(source)).forEach(function (key) { Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key)); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

function isProd() {
  return process.env.NODE_ENV === 'production';
}

let logger;
exports.logger = logger;
const debug = (0, _debug.default)('verdaccio:logger');
const DEFAULT_LOG_FORMAT = isProd() ? 'json' : 'pretty';

function createLogger(options = {
  level: 'http'
}, destination = _pino.default.destination(1), format = DEFAULT_LOG_FORMAT, prettyPrintOptions) {
  if (_lodash.default.isNil(format)) {
    format = DEFAULT_LOG_FORMAT;
  }

  let pinoConfig = _objectSpread(_objectSpread({
    customLevels: {
      http: 25
    }
  }, options), {}, {
    level: options.level,
    serializers: {
      err: _pino.default.stdSerializers.err,
      req: _pino.default.stdSerializers.req,
      res: _pino.default.stdSerializers.res
    }
  });

  debug('has prettifier? %o', !isProd()); // pretty logs are not allowed in production for performance reasons

  if ((format === DEFAULT_LOG_FORMAT || format !== 'json') && isProd() === false) {
    pinoConfig = Object.assign({}, pinoConfig, {
      // more info
      // https://github.com/pinojs/pino-pretty/issues/37
      prettyPrint: _objectSpread({
        levelFirst: true,
        prettyStamp: format === 'pretty-timestamped'
      }, prettyPrintOptions),
      prettifier: require('./formatter')
    });
  }

  const logger = (0, _pino.default)(pinoConfig, destination);

  if (process.env.DEBUG) {
    logger.on('level-change', (lvl, val, prevLvl, prevVal) => {
      debug('%s (%d) was changed to %s (%d)', lvl, val, prevLvl, prevVal);
    });
  }

  return logger;
}

function getLogger() {
  if (_lodash.default.isNil(logger)) {
    process.emitWarning('logger is not defined');
    return;
  }

  return logger;
}

const DEFAULT_LOGGER_CONF = {
  type: 'stdout',
  format: 'pretty',
  level: 'http',
  colors: true
};

function setup(options = [DEFAULT_LOGGER_CONF]) {
  var _loggerConfig, _loggerConfig2, _loggerConfig3;

  debug('setup logger');
  const isLegacyConf = Array.isArray(options);

  if (isLegacyConf) {
    const deprecateMessage = 'deprecate: multiple logger configuration is deprecated, please check the migration guide.';
    process.emitWarning(deprecateMessage);
  } // verdaccio 5 does not allow multiple logger configuration
  // backward compatible, pick only the first option
  // next major will thrown an error


  let loggerConfig = isLegacyConf ? options[0] : options;

  if (!((_loggerConfig = loggerConfig) !== null && _loggerConfig !== void 0 && _loggerConfig.level)) {
    loggerConfig = Object.assign({}, {
      level: 'http'
    }, loggerConfig);
  }

  const pinoConfig = {
    level: loggerConfig.level
  };
  let colors = typeof ((_loggerConfig2 = loggerConfig) === null || _loggerConfig2 === void 0 ? void 0 : _loggerConfig2.colors) === 'boolean' ? Boolean((_loggerConfig3 = loggerConfig) === null || _loggerConfig3 === void 0 ? void 0 : _loggerConfig3.colors) : process.stdout.isTTY;

  if ('EXPERIMENTAL_VERDACCIO_LOGGER_COLORS' in process.env) {
    colors = process.env.EXPERIMENTAL_VERDACCIO_LOGGER_COLORS != 'false';
  }

  const prettyPrintOptions = {
    // we hide warning since the prettifier should not be used in production
    // https://getpino.io/#/docs/pretty?id=prettifier-api
    suppressFlushSyncWarning: true,
    colors
  };

  if (loggerConfig.type === 'file') {
    debug('logging file enabled');

    const destination = _pino.default.destination(loggerConfig.path);

    process.on('SIGUSR2', () => destination.reopen());
    exports.logger = logger = createLogger(pinoConfig, destination, loggerConfig.format, prettyPrintOptions);
  } else if (loggerConfig.type === 'rotating-file') {
    process.emitWarning('rotating-file type is not longer supported, consider use [logrotate] instead');
    debug('logging stdout enabled');
    exports.logger = logger = createLogger(pinoConfig, _pino.default.destination(1), loggerConfig.format, prettyPrintOptions);
  } else {
    debug('logging stdout enabled');
    exports.logger = logger = createLogger(pinoConfig, _pino.default.destination(1), loggerConfig.format, prettyPrintOptions);
  }

  if (isProd()) {
    // why only on prod? https://github.com/pinojs/pino/issues/920#issuecomment-710807667
    const finalHandler = _pino.default.final(logger, (err, finalLogger, event) => {
      finalLogger.info(`${event} caught`);

      if (err) {
        finalLogger.error(err, 'error caused exit');
      }

      process.exit(err ? 1 : 0);
    });

    process.on('uncaughtException', err => finalHandler(err, 'uncaughtException'));
    process.on('unhandledRejection', err => finalHandler(err, 'unhandledRejection'));
    process.on('beforeExit', () => finalHandler(null, 'beforeExit'));
    process.on('exit', () => finalHandler(null, 'exit'));
    process.on('uncaughtException', err => finalHandler(err, 'uncaughtException'));
    process.on('SIGINT', () => finalHandler(null, 'SIGINT'));
    process.on('SIGQUIT', () => finalHandler(null, 'SIGQUIT'));
    process.on('SIGTERM', () => finalHandler(null, 'SIGTERM'));
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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