"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.allow_action = allow_action;
exports.buildUserBuffer = buildUserBuffer;
exports.createAnonymousRemoteUser = createAnonymousRemoteUser;
exports.createRemoteUser = createRemoteUser;
exports.createSessionToken = createSessionToken;
exports.expireReasons = exports.defaultUserRateLimiting = exports.defaultSecurity = void 0;
exports.getApiToken = getApiToken;
exports.getAuthenticatedMessage = getAuthenticatedMessage;
exports.getDefaultPlugins = getDefaultPlugins;
exports.getMiddlewareCredentials = getMiddlewareCredentials;
exports.getSecurity = getSecurity;
exports.handleSpecialUnpublish = handleSpecialUnpublish;
exports.isAESLegacy = isAESLegacy;
exports.isAuthHeaderValid = isAuthHeaderValid;
exports.parseAESCredentials = parseAESCredentials;
exports.parseAuthTokenHeader = parseAuthTokenHeader;
exports.parseBasicPayload = parseBasicPayload;
exports.validatePassword = validatePassword;
exports.verifyJWTPayload = verifyJWTPayload;

var _debug = _interopRequireDefault(require("debug"));

var _lodash = _interopRequireDefault(require("lodash"));

var _logger = require("../lib/logger");

var _constants = require("./constants");

var _cryptoUtils = require("./crypto-utils");

var _utils = require("./utils");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const debug = (0, _debug.default)('verdaccio');

function validatePassword(password, // pragma: allowlist secret
minLength = _constants.DEFAULT_MIN_LIMIT_PASSWORD) {
  return typeof password === 'string' && password.length >= minLength;
}
/**
 * Create a RemoteUser object
 * @return {Object} { name: xx, pluginGroups: [], real_groups: [] }
 */


function createRemoteUser(name, pluginGroups) {
  const isGroupValid = Array.isArray(pluginGroups);
  const groups = Array.from(new Set((isGroupValid ? pluginGroups : []).concat([_constants.ROLES.$ALL, _constants.ROLES.$AUTH, _constants.ROLES.DEPRECATED_ALL, _constants.ROLES.DEPRECATED_AUTH, _constants.ROLES.ALL])));
  return {
    name,
    groups,
    real_groups: pluginGroups
  };
}
/**
 * Builds an anonymous remote user in case none is logged in.
 * @return {Object} { name: xx, groups: [], real_groups: [] }
 */


function createAnonymousRemoteUser() {
  return {
    name: undefined,
    // groups without '$' are going to be deprecated eventually
    groups: [_constants.ROLES.$ALL, _constants.ROLES.$ANONYMOUS, _constants.ROLES.DEPRECATED_ALL, _constants.ROLES.DEPRECATED_ANONYMOUS],
    real_groups: []
  };
}

function allow_action(action) {
  return function (user, pkg, callback) {
    debug('[auth/allow_action]: user: %o', user === null || user === void 0 ? void 0 : user.name);
    const {
      name,
      groups
    } = user;
    const groupAccess = pkg[action];
    const hasPermission = groupAccess.some(group => name === group || groups.includes(group));
    debug('[auth/allow_action]: hasPermission? %o} for user: %o', hasPermission, user === null || user === void 0 ? void 0 : user.name);

    if (hasPermission) {
      _logger.logger.info({
        remote: user.name
      }, `auth/allow_action: access granted to: @{user}`);

      return callback(null, true);
    }

    if (name) {
      callback(_utils.ErrorCode.getForbidden(`user ${name} is not allowed to ${action} package ${pkg.name}`));
    } else {
      callback(_utils.ErrorCode.getUnauthorized(`authorization required to ${action} package ${pkg.name}`));
    }
  };
}
/**
 *
 */


function handleSpecialUnpublish() {
  return function (user, pkg, callback) {
    const action = 'unpublish'; // verify whether the unpublish prop has been defined

    const isUnpublishMissing = _lodash.default.isNil(pkg[action]);

    const hasGroups = isUnpublishMissing ? false : pkg[action].length > 0;
    debug('fallback unpublish for @{name} has groups: %o for %o', hasGroups, user === null || user === void 0 ? void 0 : user.name);

    if (isUnpublishMissing || hasGroups === false) {
      return callback(null, undefined);
    }

    debug('allow_action for %o for %o has groups: %o for %o', action, user === null || user === void 0 ? void 0 : user.name, hasGroups, user);
    return allow_action(action)(user, pkg, callback);
  };
}

function getDefaultPlugins(logger) {
  return {
    authenticate(_user, _password, cb) {
      // pragma: allowlist secret
      cb(_utils.ErrorCode.getForbidden(_constants.API_ERROR.BAD_USERNAME_PASSWORD));
    },

    add_user(_user, _password, cb) {
      // pragma: allowlist secret
      return cb(_utils.ErrorCode.getConflict(_constants.API_ERROR.BAD_USERNAME_PASSWORD));
    },

    // FIXME: allow_action and allow_publish should be in the @verdaccio/types
    // @ts-ignore
    allow_access: allow_action('access', logger),
    // @ts-ignore
    allow_publish: allow_action('publish', logger),
    allow_unpublish: handleSpecialUnpublish()
  };
}

function createSessionToken() {
  const tenHoursTime = 10 * 60 * 60 * 1000;
  return {
    // npmjs.org sets 10h expire
    expires: new Date(Date.now() + tenHoursTime)
  };
}

const defaultWebTokenOptions = {
  sign: {
    // The expiration token for the website is 1 hour
    expiresIn: _constants.TIME_EXPIRATION_1H
  },
  verify: {}
};
const defaultApiTokenConf = {
  legacy: true
}; // we limit max 1000 request per 15 minutes on user endpoints

const defaultUserRateLimiting = {
  windowMs: 15 * 60 * 1000,
  // 15 minutes
  max: 1000
};
exports.defaultUserRateLimiting = defaultUserRateLimiting;
const defaultSecurity = {
  web: defaultWebTokenOptions,
  api: defaultApiTokenConf
};
exports.defaultSecurity = defaultSecurity;

function getSecurity(config) {
  if (_lodash.default.isNil(config.security) === false) {
    return _lodash.default.merge(defaultSecurity, config.security);
  }

  return defaultSecurity;
}

function getAuthenticatedMessage(user) {
  return `you are authenticated as '${user}'`;
}

function buildUserBuffer(name, password) {
  return Buffer.from(`${name}:${password}`, 'utf8');
}

function isAESLegacy(security) {
  const {
    legacy,
    jwt
  } = security.api;
  return _lodash.default.isNil(legacy) === false && _lodash.default.isNil(jwt) && legacy === true;
}

async function getApiToken(auth, config, remoteUser, aesPassword) {
  const security = getSecurity(config);

  if (isAESLegacy(security)) {
    // fallback all goes to AES encryption
    return await new Promise(resolve => {
      resolve(auth.aesEncrypt(buildUserBuffer(remoteUser.name, aesPassword)).toString('base64'));
    });
  } // i am wiling to use here _.isNil but flow does not like it yet.


  const {
    jwt
  } = security.api;

  if (jwt && jwt.sign) {
    return await auth.jwtEncrypt(remoteUser, jwt.sign);
  }

  return await new Promise(resolve => {
    resolve(auth.aesEncrypt(buildUserBuffer(remoteUser.name, aesPassword)).toString('base64'));
  });
}

function parseAuthTokenHeader(authorizationHeader) {
  const parts = authorizationHeader.split(' ');
  const [scheme, token] = parts;
  return {
    scheme,
    token
  };
}

function parseBasicPayload(credentials) {
  const index = credentials.indexOf(':');

  if (index < 0) {
    return;
  }

  const user = credentials.slice(0, index);
  const password = credentials.slice(index + 1);
  return {
    user,
    password
  };
}

function parseAESCredentials(authorizationHeader, secret) {
  const {
    scheme,
    token
  } = parseAuthTokenHeader(authorizationHeader); // basic is deprecated and should not be enforced

  if (scheme.toUpperCase() === _constants.TOKEN_BASIC.toUpperCase()) {
    const credentials = (0, _utils.convertPayloadToBase64)(token).toString();
    return credentials;
  } else if (scheme.toUpperCase() === _constants.TOKEN_BEARER.toUpperCase()) {
    const tokenAsBuffer = (0, _utils.convertPayloadToBase64)(token);
    const credentials = (0, _cryptoUtils.aesDecrypt)(tokenAsBuffer, secret).toString('utf8');
    return credentials;
  }
}

const expireReasons = ['JsonWebTokenError', 'TokenExpiredError'];
exports.expireReasons = expireReasons;

function verifyJWTPayload(token, secret) {
  try {
    const payload = (0, _cryptoUtils.verifyPayload)(token, secret);
    return payload;
  } catch (error) {
    // #168 this check should be removed as soon AES encrypt is removed.
    if (expireReasons.includes(error.name)) {
      // it might be possible the jwt configuration is enabled and
      // old tokens fails still remains in usage, thus
      // we return an anonymous user to force log in.
      return createAnonymousRemoteUser();
    }

    throw _utils.ErrorCode.getCode(_constants.HTTP_STATUS.UNAUTHORIZED, error.message);
  }
}

function isAuthHeaderValid(authorization) {
  return authorization.split(' ').length === 2;
}

function getMiddlewareCredentials(security, secret, authorizationHeader) {
  if (isAESLegacy(security)) {
    const credentials = parseAESCredentials(authorizationHeader, secret);

    if (!credentials) {
      return;
    }

    const parsedCredentials = parseBasicPayload(credentials);

    if (!parsedCredentials) {
      return;
    }

    return parsedCredentials;
  }

  const {
    scheme,
    token
  } = parseAuthTokenHeader(authorizationHeader);

  if (_lodash.default.isString(token) && scheme.toUpperCase() === _constants.TOKEN_BEARER.toUpperCase()) {
    return verifyJWTPayload(token, secret);
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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