"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = _default;
exports.loadTheme = loadTheme;
exports.validatePrimaryColor = validatePrimaryColor;

var _debug = _interopRequireDefault(require("debug"));

var _express = _interopRequireDefault(require("express"));

var _fs = _interopRequireDefault(require("fs"));

var _lodash = _interopRequireDefault(require("lodash"));

var _path = _interopRequireDefault(require("path"));

var _constants = require("../../lib/constants");

var _logger = require("../../lib/logger");

var _pluginLoader = _interopRequireDefault(require("../../lib/plugin-loader"));

var _search = _interopRequireDefault(require("../../lib/search"));

var _utils = require("../../lib/utils");

var _renderHTML = _interopRequireDefault(require("./html/renderHTML"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const {
  setSecurityWebHeaders
} = require('../middleware');

const debug = (0, _debug.default)('verdaccio');

function loadTheme(config) {
  if (_lodash.default.isNil(config.theme) === false) {
    return _lodash.default.head((0, _pluginLoader.default)(config, config.theme, {}, function (plugin) {
      return plugin.staticPath && plugin.manifest && plugin.manifestFiles;
    }, 'verdaccio-theme'));
  }
}

function validatePrimaryColor(primaryColor) {
  const isHex = /^#+([a-fA-F0-9]{6}|[a-fA-F0-9]{3})$/i.test(primaryColor);

  if (!isHex) {
    debug('invalid primary color %o', primaryColor);
    return;
  }

  return primaryColor;
}

const sendFileCallback = next => err => {
  if (!err) {
    return;
  }

  if (err.status === _constants.HTTP_STATUS.NOT_FOUND) {
    next();
  } else {
    next(err);
  }
};

function _default(config, auth, storage) {
  var _config$web, _config$web2;

  let {
    staticPath,
    manifest,
    manifestFiles
  } = loadTheme(config) || require('@verdaccio/ui-theme')();

  debug('static path %o', staticPath);

  _search.default.configureStorage(storage);
  /* eslint new-cap:off */


  const router = _express.default.Router(); // run in production mode by default, just in case
  // it shouldn't make any difference anyway


  router.use(auth.webUIJWTmiddleware());
  router.use(setSecurityWebHeaders); // static assets

  router.get('/-/static/*', function (req, res, next) {
    const filename = req.params[0];
    const file = `${staticPath}/${filename}`;
    debug('render static file %o', file);
    res.sendFile(file, sendFileCallback(next));
  }); // logo

  if (config !== null && config !== void 0 && (_config$web = config.web) !== null && _config$web !== void 0 && _config$web.logo && !(0, _utils.isHTTPProtocol)(config === null || config === void 0 ? void 0 : (_config$web2 = config.web) === null || _config$web2 === void 0 ? void 0 : _config$web2.logo)) {
    // URI related to a local file
    const absoluteLocalFile = _path.default.posix.resolve(config.web.logo);

    debug('serve local logo %s', absoluteLocalFile);

    try {
      if (_fs.default.existsSync(absoluteLocalFile) && typeof _fs.default.accessSync(absoluteLocalFile, _fs.default.constants.R_OK) === 'undefined') {
        // Note: `path.join` will break on Windows, because it transforms `/` to `\`
        // Use POSIX version `path.posix.join` instead.
        config.web.logo = _path.default.posix.join('/-/static/', _path.default.basename(config.web.logo));
        router.get(config.web.logo, function (_req, res, next) {
          // @ts-ignore
          debug('serve custom logo  web:%s - local:%s', config.web.logo, absoluteLocalFile);
          res.sendFile(absoluteLocalFile, sendFileCallback(next));
        });
        debug('enabled custom logo %s', config.web.logo);
      } else {
        config.web.logo = undefined;

        _logger.logger.warn(`web logo is wrong, path ${absoluteLocalFile} does not exist or is not readable`);
      }
    } catch {
      config.web.logo = undefined;

      _logger.logger.warn(`web logo is wrong, path ${absoluteLocalFile} does not exist or is not readable`);
    }
  }

  router.get('/-/web/:section/*', function (req, res) {
    (0, _renderHTML.default)(config, manifest, manifestFiles, req, res);
    debug('render html section');
  });
  router.get('/', function (req, res, next) {
    (0, _renderHTML.default)(config, manifest, manifestFiles, req, res);
    debug('render root');
  });
  return router;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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