"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LOG_VERDACCIO_ERROR = exports.LOG_VERDACCIO_BYTES = exports.LOG_STATUS_MESSAGE = void 0;
exports.allow = allow;
exports.antiLoop = antiLoop;
exports.encodeScopePackage = encodeScopePackage;
exports.errorReportingMiddleware = errorReportingMiddleware;
exports.expectJson = expectJson;
exports.final = final;
exports.log = log;
exports.match = match;
exports.media = media;
exports.serveFavicon = serveFavicon;
exports.setSecurityWebHeaders = setSecurityWebHeaders;
exports.validateName = validateName;
exports.validatePackage = validatePackage;

var _debug = _interopRequireDefault(require("debug"));

var _fs = _interopRequireDefault(require("fs"));

var _lodash = _interopRequireDefault(require("lodash"));

var _path = _interopRequireDefault(require("path"));

var _validator = _interopRequireDefault(require("validator"));

var _constants = require("../lib/constants");

var _cryptoUtils = require("../lib/crypto-utils");

var _logger = require("../lib/logger");

var _utils = require("../lib/utils");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const debug = (0, _debug.default)('verdaccio');

function match(regexp) {
  return function (req, res, next, value) {
    if (regexp.exec(value)) {
      next();
    } else {
      next('route');
    }
  };
}

function serveFavicon(config) {
  return function (req, res, next) {
    try {
      var _config$web;

      // @ts-ignore
      const logoConf = config === null || config === void 0 ? void 0 : (_config$web = config.web) === null || _config$web === void 0 ? void 0 : _config$web.favicon;

      if (logoConf === '') {
        debug('favicon disabled');
        res.status(404);
      } else if (!_lodash.default.isEmpty(logoConf)) {
        debug('custom favicon');

        if (_validator.default.isURL(logoConf, {
          require_host: true,
          require_valid_protocol: true
        })) {
          debug('redirect to %o', logoConf);
          res.redirect(logoConf);
          return;
        } else {
          const faviconPath = _path.default.normalize(logoConf);

          debug('serving favicon from %o', faviconPath);

          _fs.default.access(faviconPath, _fs.default.constants.R_OK, err => {
            if (err) {
              debug('no read permissions to read: %o, reason:', logoConf, err === null || err === void 0 ? void 0 : err.message);
              return res.status(_constants.HTTP_STATUS.NOT_FOUND).end();
            } else {
              res.setHeader('content-type', 'image/x-icon');

              _fs.default.createReadStream(faviconPath).pipe(res);

              debug('rendered custom ico');
            }
          });
        }
      } else {
        res.setHeader('content-type', 'image/x-icon');

        _fs.default.createReadStream(_path.default.posix.join(__dirname, './web/html/favicon.ico')).pipe(res);

        debug('rendered ico');
      }
    } catch (err) {
      debug('error triggered, favicon not found');
      res.status(_constants.HTTP_STATUS.NOT_FOUND).end();
    }
  };
}

function setSecurityWebHeaders(req, res, next) {
  // disable loading in frames (clickjacking, etc.)
  res.header(_constants.HEADERS.FRAMES_OPTIONS, 'deny'); // avoid stablish connections outside of domain

  res.header(_constants.HEADERS.CSP, "connect-src 'self'"); // https://stackoverflow.com/questions/18337630/what-is-x-content-type-options-nosniff

  res.header(_constants.HEADERS.CTO, 'nosniff'); // https://stackoverflow.com/questions/9090577/what-is-the-http-header-x-xss-protection

  res.header(_constants.HEADERS.XSS, '1; mode=block');
  next();
} // flow: express does not match properly
// flow info https://github.com/flowtype/flow-typed/issues?utf8=%E2%9C%93&q=is%3Aissue+is%3Aopen+express


function validateName(req, res, next, value, name) {
  if (value === '-') {
    // special case in couchdb usually
    next('route');
  } else if ((0, _utils.validateName)(value)) {
    next();
  } else {
    next(_utils.ErrorCode.getForbidden('invalid ' + name));
  }
} // flow: express does not match properly
// flow info https://github.com/flowtype/flow-typed/issues?utf8=%E2%9C%93&q=is%3Aissue+is%3Aopen+express


function validatePackage(req, res, next, value, name) {
  if (value === '-') {
    // special case in couchdb usually
    next('route');
  } else if ((0, _utils.validatePackage)(value)) {
    next();
  } else {
    next(_utils.ErrorCode.getForbidden('invalid ' + name));
  }
}

function media(expect) {
  return function (req, res, next) {
    if (req.headers[_constants.HEADER_TYPE.CONTENT_TYPE] !== expect) {
      next(_utils.ErrorCode.getCode(_constants.HTTP_STATUS.UNSUPPORTED_MEDIA, 'wrong content-type, expect: ' + expect + ', got: ' + req.get(_constants.HEADER_TYPE.CONTENT_TYPE)));
    } else {
      next();
    }
  };
}

function encodeScopePackage(req, res, next) {
  if (req.url.indexOf('@') !== -1) {
    // e.g.: /@org/pkg/1.2.3 -> /@org%2Fpkg/1.2.3, /@org%2Fpkg/1.2.3 -> /@org%2Fpkg/1.2.3
    req.url = req.url.replace(/^(\/@[^\/%]+)\/(?!$)/, '$1%2F');
  }

  next();
}

function expectJson(req, res, next) {
  if (!(0, _utils.isObject)(req.body)) {
    return next(_utils.ErrorCode.getBadRequest("can't parse incoming json"));
  }

  next();
}

function antiLoop(config) {
  return function (req, res, next) {
    var _req$headers;

    if ((req === null || req === void 0 ? void 0 : (_req$headers = req.headers) === null || _req$headers === void 0 ? void 0 : _req$headers.via) != null) {
      const arr = req.headers.via.split(',');

      for (let i = 0; i < arr.length; i++) {
        const m = arr[i].match(/\s*(\S+)\s+(\S+)/);

        if (m && m[2] === config.server_id) {
          return next(_utils.ErrorCode.getCode(_constants.HTTP_STATUS.LOOP_DETECTED, 'loop detected'));
        }
      }
    }

    next();
  };
}

function allow(auth) {
  return function (action) {
    return function (req, res, next) {
      req.pause();
      const packageName = req.params.scope ? `@${req.params.scope}/${req.params.package}` : req.params.package;
      let packageVersion = undefined;

      if (req.params.filename) {
        packageVersion = (0, _utils.getVersionFromTarball)(req.params.filename) || undefined;
      } else if (typeof req.body.versions === 'object') {
        packageVersion = Object.keys(req.body.versions)[0];
      }

      const remote = req.remote_user;
      debug('[middleware/allow][%o] allow for %o', action, remote === null || remote === void 0 ? void 0 : remote.name);
      auth['allow_' + action]({
        packageName,
        packageVersion
      }, remote, function (error, allowed) {
        req.resume();

        if (error) {
          next(error);
        } else if (allowed) {
          next();
        } else {
          // last plugin (that's our built-in one) returns either
          // cb(err) or cb(null, true), so this should never happen
          throw _utils.ErrorCode.getInternalError(_constants.API_ERROR.PLUGIN_ERROR);
        }
      });
    };
  };
}

function final(body, req, res, next) {
  if (res.statusCode === _constants.HTTP_STATUS.UNAUTHORIZED && !res.getHeader(_constants.HEADERS.WWW_AUTH)) {
    // they say it's required for 401, so...
    res.header(_constants.HEADERS.WWW_AUTH, `${_constants.TOKEN_BASIC}, ${_constants.TOKEN_BEARER}`);
  }

  try {
    if (_lodash.default.isString(body) || _lodash.default.isObject(body)) {
      if (!res.getHeader(_constants.HEADERS.CONTENT_TYPE)) {
        res.header(_constants.HEADERS.CONTENT_TYPE, _constants.HEADERS.JSON);
      }

      if (typeof body === 'object' && _lodash.default.isNil(body) === false) {
        if (typeof body.error === 'string') {
          res.locals._verdaccio_error = body.error;
        }

        body = JSON.stringify(body, undefined, '  ') + '\n';
      } // don't send etags with errors


      if (!res.statusCode || res.statusCode >= _constants.HTTP_STATUS.OK && res.statusCode < _constants.HTTP_STATUS.MULTIPLE_CHOICES) {
        res.header(_constants.HEADERS.ETAG, '"' + (0, _cryptoUtils.stringToMD5)(body) + '"');
      }
    } else {// send(null), send(204), etc.
    }
  } catch (err) {
    // if verdaccio sends headers first, and then calls res.send()
    // as an error handler, we can't report error properly,
    // and should just close socket
    if (err.message.match(/set headers after they are sent/)) {
      if (_lodash.default.isNil(res.socket) === false) {
        // @ts-ignore
        res.socket.destroy();
      }

      return;
    }

    throw err;
  }

  res.send(body);
}

const LOG_STATUS_MESSAGE = "@{status}, user: @{user}(@{remoteIP}), req: '@{request.method} @{request.url}'";
exports.LOG_STATUS_MESSAGE = LOG_STATUS_MESSAGE;
const LOG_VERDACCIO_ERROR = `${LOG_STATUS_MESSAGE}, error: @{!error}`;
exports.LOG_VERDACCIO_ERROR = LOG_VERDACCIO_ERROR;
const LOG_VERDACCIO_BYTES = `${LOG_STATUS_MESSAGE}, bytes: @{bytes.in}/@{bytes.out}`;
exports.LOG_VERDACCIO_BYTES = LOG_VERDACCIO_BYTES;

function log(config) {
  return function (req, res, next) {
    var _config$experiments;

    const _auth = req.headers.authorization;

    if (_lodash.default.isNil(_auth) === false) {
      req.headers.authorization = '<Classified>';
    }

    const _cookie = req.get('cookie');

    if (_lodash.default.isNil(_cookie) === false) {
      req.headers.cookie = '<Classified>';
    }

    req.url = req.originalUrl; // avoid log noise data from static content

    if (req.originalUrl.match(/static/) === null) {
      _logger.logger.http({
        req: req,
        ip: req.ip
      }, "@{ip} requested '@{req.method} @{req.url}'");
    }

    req.originalUrl = req.url;

    if (_lodash.default.isNil(_auth) === false) {
      req.headers.authorization = _auth;
    }

    if (_lodash.default.isNil(_cookie) === false) {
      req.headers.cookie = _cookie;
    }

    let bytesin = 0;

    if ((config === null || config === void 0 ? void 0 : (_config$experiments = config.experiments) === null || _config$experiments === void 0 ? void 0 : _config$experiments.bytesin_off) !== true) {
      req.on('data', function (chunk) {
        bytesin += chunk.length;
      });
    }

    let bytesout = 0;
    const _write = res.write; // FIXME: res.write should return boolean
    // @ts-ignore

    res.write = function (buf) {
      bytesout += buf.length;
      /* eslint prefer-rest-params: "off" */
      // @ts-ignore

      _write.apply(res, arguments);
    };

    let logHasBeenCalled = false;

    const log = function () {
      if (logHasBeenCalled) {
        return;
      }

      logHasBeenCalled = true;
      const forwardedFor = req.get('x-forwarded-for');
      const remoteAddress = req.connection.remoteAddress;
      const remoteIP = forwardedFor ? `${forwardedFor} via ${remoteAddress}` : remoteAddress;
      let message;

      if (res.locals._verdaccio_error) {
        message = LOG_VERDACCIO_ERROR;
      } else {
        message = LOG_VERDACCIO_BYTES;
      }

      req.url = req.originalUrl; // avoid log noise data from static content

      if (req.url.match(/static/) === null) {
        _logger.logger.http({
          request: {
            method: req.method,
            url: req.url
          },
          user: req.remote_user && req.remote_user.name || null,
          remoteIP,
          status: res.statusCode,
          error: res.locals._verdaccio_error,
          bytes: {
            in: bytesin,
            out: bytesout
          }
        }, message);

        req.originalUrl = req.url;
      }
    };

    req.on('close', function () {
      log();
    });
    const _end = res.end;

    res.end = function (buf) {
      if (buf) {
        bytesout += buf.length;
      }
      /* eslint prefer-rest-params: "off" */
      // @ts-ignore


      _end.apply(res, arguments);

      log();
    };

    next();
  };
} // Middleware


function errorReportingMiddleware(req, res, next) {
  res.locals.report_error = res.locals.report_error || function (err) {
    if (err.status && err.status >= _constants.HTTP_STATUS.BAD_REQUEST && err.status < 600) {
      if (!res.headersSent) {
        res.status(err.status);
        next({
          error: err.message || _constants.API_ERROR.UNKNOWN_ERROR
        });
      }
    } else {
      _logger.logger.error({
        err: err
      }, 'unexpected error: @{!err.message}\n@{err.stack}');

      if (!res.status || !res.send) {
        _logger.logger.error('this is an error in express.js, please report this');

        res.destroy();
      } else if (!res.headersSent) {
        res.status(_constants.HTTP_STATUS.INTERNAL_ERROR);
        next({
          error: _constants.API_ERROR.INTERNAL_SERVER_ERROR
        });
      } else {// socket should be already closed
      }
    }
  };

  next();
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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