"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _compression = _interopRequireDefault(require("compression"));

var _cors = _interopRequireDefault(require("cors"));

var _express = _interopRequireDefault(require("express"));

var _lodash = _interopRequireDefault(require("lodash"));

var _auth = _interopRequireDefault(require("../lib/auth"));

var _config = _interopRequireDefault(require("../lib/config"));

var _constants = require("../lib/constants");

var _logger = require("../lib/logger");

var _pluginLoader = _interopRequireDefault(require("../lib/plugin-loader"));

var _storage = _interopRequireDefault(require("../lib/storage"));

var _utils = require("../lib/utils");

var _debug = _interopRequireDefault(require("./debug"));

var _endpoint = _interopRequireDefault(require("./endpoint"));

var _middleware = require("./middleware");

var _web = _interopRequireDefault(require("./web"));

var _api = _interopRequireDefault(require("./web/api"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const defineAPI = function (config, storage) {
  var _config$server;

  const auth = new _auth.default(config);
  const app = (0, _express.default)(); // run in production mode by default, just in case
  // it shouldn't make any difference anyway

  app.set('env', process.env.NODE_ENV || 'production'); // Allow `req.ip` to resolve properly when Verdaccio is behind a proxy or load-balancer
  // See: https://expressjs.com/en/guide/behind-proxies.html

  if ((_config$server = config.server) !== null && _config$server !== void 0 && _config$server.trustProxy) {
    app.set('trust proxy', config.server.trustProxy);
  }

  app.use((0, _cors.default)()); // Router setup

  app.use((0, _middleware.log)(config));
  app.use(_middleware.errorReportingMiddleware);

  if (config.user_agent) {
    app.use(function (req, res, next) {
      res.setHeader('X-Powered-By', (0, _utils.getUserAgent)(config.user_agent));
      next();
    });
  } else {
    app.disable('x-powered-by');
  }

  app.use((0, _compression.default)());
  app.get('/-/static/favicon.ico', (0, _middleware.serveFavicon)(config)); // Hook for tests only

  if (config._debug) {
    (0, _debug.default)(app, config.self_path);
  } // register middleware plugins


  const plugin_params = {
    config: config,
    logger: _logger.logger
  };
  const plugins = (0, _pluginLoader.default)(config, config.middlewares, plugin_params, function (plugin) {
    return plugin.register_middlewares;
  });
  plugins.forEach(plugin => {
    plugin.register_middlewares(app, auth, storage);
  }); // For  npm request

  app.use((0, _endpoint.default)(config, auth, storage)); // For WebUI & WebUI API

  if (_lodash.default.get(config, 'web.enable', true)) {
    app.use('/', (0, _web.default)(config, auth, storage));
    app.use('/-/verdaccio/', (0, _api.default)(config, auth, storage));
  } else {
    app.get('/', function (req, res, next) {
      next(_utils.ErrorCode.getNotFound(_constants.API_ERROR.WEB_DISABLED));
    });
  } // Catch 404


  app.get('/*', function (req, res, next) {
    next(_utils.ErrorCode.getNotFound(_constants.API_ERROR.FILE_NOT_FOUND));
  });
  app.use(function (err, req, res, next) {
    if (_lodash.default.isError(err)) {
      if (err.code === 'ECONNABORT' && res.statusCode === _constants.HTTP_STATUS.NOT_MODIFIED) {
        return next();
      }

      if (_lodash.default.isFunction(res.locals.report_error) === false) {
        // in case of very early error this middleware may not be loaded before error is generated
        // fixing that
        (0, _middleware.errorReportingMiddleware)(req, res, _lodash.default.noop);
      }

      res.locals.report_error(err);
    } else {
      // Fall to Middleware.final
      return next(err);
    }
  });
  app.use(_middleware.final);
  return app;
};

var _default = async function _default(configHash) {
  (0, _logger.setup)(configHash.logs);
  const config = new _config.default(_lodash.default.cloneDeep(configHash)); // register middleware plugins

  const plugin_params = {
    config: config,
    logger: _logger.logger
  };
  const filters = (0, _pluginLoader.default)(config, config.filters || {}, plugin_params, plugin => plugin.filter_metadata);
  const storage = new _storage.default(config); // waits until init calls have been initialized

  await storage.init(config, filters);
  return defineAPI(config, storage);
};

exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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