"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = _default;

var _debug = _interopRequireDefault(require("debug"));

var _express = require("express");

var _lodash = _interopRequireDefault(require("lodash"));

var _authUtils = require("../../../../lib/auth-utils");

var _constants = require("../../../../lib/constants");

var _cryptoUtils = require("../../../../lib/crypto-utils");

var _logger = require("../../../../lib/logger");

var _utils = require("../../../../lib/utils");

var _rateLimiter = require("../../../rate-limiter");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function ownKeys(object, enumerableOnly) { var keys = Object.keys(object); if (Object.getOwnPropertySymbols) { var symbols = Object.getOwnPropertySymbols(object); enumerableOnly && (symbols = symbols.filter(function (sym) { return Object.getOwnPropertyDescriptor(object, sym).enumerable; })), keys.push.apply(keys, symbols); } return keys; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = null != arguments[i] ? arguments[i] : {}; i % 2 ? ownKeys(Object(source), !0).forEach(function (key) { _defineProperty(target, key, source[key]); }) : Object.getOwnPropertyDescriptors ? Object.defineProperties(target, Object.getOwnPropertyDescriptors(source)) : ownKeys(Object(source)).forEach(function (key) { Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key)); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const debug = (0, _debug.default)('verdaccio:token');

function normalizeToken(token) {
  return _objectSpread(_objectSpread({}, token), {}, {
    created: new Date(token.created).toISOString()
  });
} // https://github.com/npm/npm-profile/blob/latest/lib/index.js


function _default(auth, storage, config) {
  const tokenRoute = (0, _express.Router)();
  /* eslint new-cap: 0 */

  tokenRoute.get('/tokens', (0, _rateLimiter.limiter)(config === null || config === void 0 ? void 0 : config.userRateLimit), async function (req, res, next) {
    const {
      name
    } = req.remote_user;

    if (_lodash.default.isNil(name) === false) {
      try {
        const tokens = await storage.readTokens({
          user: name
        });
        const totalTokens = tokens.length;
        debug('token list retrieved: %o', totalTokens);
        res.status(_constants.HTTP_STATUS.OK);
        return next({
          objects: tokens.map(normalizeToken),
          urls: {
            next: '' // TODO: pagination?

          }
        });
      } catch (error) {
        _logger.logger.error({
          error: error.msg
        }, 'token list has failed: @{error}');

        return next(_utils.ErrorCode.getCode(_constants.HTTP_STATUS.INTERNAL_ERROR, error.message));
      }
    }

    return next(_utils.ErrorCode.getUnauthorized());
  });
  tokenRoute.post('/tokens', (0, _rateLimiter.limiter)(config === null || config === void 0 ? void 0 : config.userRateLimit), function (req, res, next) {
    const {
      password,
      readonly,
      cidr_whitelist
    } = req.body;
    const {
      name
    } = req.remote_user;

    if (!_lodash.default.isBoolean(readonly) || !_lodash.default.isArray(cidr_whitelist)) {
      return next(_utils.ErrorCode.getCode(_constants.HTTP_STATUS.BAD_DATA, _constants.SUPPORT_ERRORS.PARAMETERS_NOT_VALID));
    }

    auth.authenticate(name, password, async (err, user) => {
      if (err) {
        const errorCode = err.message ? _constants.HTTP_STATUS.UNAUTHORIZED : _constants.HTTP_STATUS.INTERNAL_ERROR;
        return next(_utils.ErrorCode.getCode(errorCode, err.message));
      }

      req.remote_user = user;

      if (!_lodash.default.isFunction(storage.saveToken)) {
        return next(_utils.ErrorCode.getCode(_constants.HTTP_STATUS.NOT_IMPLEMENTED, _constants.SUPPORT_ERRORS.STORAGE_NOT_IMPLEMENT));
      }

      try {
        const token = await (0, _authUtils.getApiToken)(auth, config, user, password);
        const key = (0, _cryptoUtils.stringToMD5)(token); // TODO: use a utility here

        const maskedToken = (0, _utils.mask)(token, 5);
        const created = new Date().getTime();
        /**
         * cidr_whitelist: is not being used, we pass it through
         * token: we do not store the real token (it is generated once and retrieved to the user), just a mask of it.
         */

        const saveToken = {
          user: name,
          token: maskedToken,
          key,
          cidr: cidr_whitelist,
          readonly,
          created
        };
        await storage.saveToken(saveToken);
        debug('token %o was created for user %o', key, name);
        res.set(_constants.HEADERS.CACHE_CONTROL, 'no-cache, no-store');
        return next(normalizeToken({
          token,
          user: name,
          key: saveToken.key,
          cidr: cidr_whitelist,
          readonly,
          created: saveToken.created
        }));
      } catch (error) {
        _logger.logger.error({
          error: error.msg
        }, 'token creation has failed: @{error}');

        return next(_utils.ErrorCode.getCode(_constants.HTTP_STATUS.INTERNAL_ERROR, error.message));
      }
    });
  });
  tokenRoute.delete('/tokens/token/:tokenKey', (0, _rateLimiter.limiter)(config === null || config === void 0 ? void 0 : config.userRateLimit), async (req, res, next) => {
    const {
      params: {
        tokenKey
      }
    } = req;
    const {
      name
    } = req.remote_user;

    if (_lodash.default.isNil(name) === false) {
      debug('%o has requested remove a token', name);

      try {
        await storage.deleteToken(name, tokenKey);

        _logger.logger.info({
          tokenKey,
          name
        }, 'token id @{tokenKey} was revoked for user @{name}');

        return next({});
      } catch (error) {
        _logger.logger.error({
          error: error.msg
        }, 'token creation has failed: @{error}');

        return next(_utils.ErrorCode.getCode(_constants.HTTP_STATUS.INTERNAL_ERROR, error.message));
      }
    }

    return next(_utils.ErrorCode.getUnauthorized());
  });
  return tokenRoute;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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