"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addVersion = addVersion;
exports.default = publish;
exports.publishPackage = publishPackage;
exports.removeTarball = removeTarball;
exports.unPublishPackage = unPublishPackage;
exports.uploadPackageTarball = uploadPackageTarball;

var _debug = _interopRequireDefault(require("debug"));

var _lodash = _interopRequireDefault(require("lodash"));

var _mime = _interopRequireDefault(require("mime"));

var _path = _interopRequireDefault(require("path"));

var _constants = require("../../../lib/constants");

var _logger = require("../../../lib/logger");

var _notify = require("../../../lib/notify");

var _storageUtils = require("../../../lib/storage-utils");

var _utils = require("../../../lib/utils");

var _middleware = require("../../middleware");

var _star = _interopRequireDefault(require("./star"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function ownKeys(object, enumerableOnly) { var keys = Object.keys(object); if (Object.getOwnPropertySymbols) { var symbols = Object.getOwnPropertySymbols(object); enumerableOnly && (symbols = symbols.filter(function (sym) { return Object.getOwnPropertyDescriptor(object, sym).enumerable; })), keys.push.apply(keys, symbols); } return keys; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = null != arguments[i] ? arguments[i] : {}; i % 2 ? ownKeys(Object(source), !0).forEach(function (key) { _defineProperty(target, key, source[key]); }) : Object.getOwnPropertyDescriptors ? Object.defineProperties(target, Object.getOwnPropertyDescriptors(source)) : ownKeys(Object(source)).forEach(function (key) { Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key)); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const debug = (0, _debug.default)('verdaccio:publish');

function publish(router, auth, storage, config) {
  const can = (0, _middleware.allow)(auth);
  /**
   * Publish a package / update package / un/start a package
   *
   * There are multiples scenarios here to be considered:
   *
   * 1. Publish scenario
   *
   * Publish a package consist of at least 1 step (PUT) with a metadata payload.
   * When a package is published, an _attachment property is present that contains the data
   * of the tarball.
   *
   * Example flow of publish.
   *
   *  npm http fetch PUT 201 http://localhost:4873/@scope%2ftest1 9627ms
      npm info lifecycle @scope/test1@1.0.1~publish: @scope/test1@1.0.1
      npm info lifecycle @scope/test1@1.0.1~postpublish: @scope/test1@1.0.1
      + @scope/test1@1.0.1
      npm verb exit [ 0, true ]
   *
   *
   * 2. Unpublish scenario
   *
   * Unpublish consist in 3 steps.
   *  1. Try to fetch  metadata -> if it fails, return 404
   *  2. Compute metadata locally (client side) and send a mutate payload excluding the version to be unpublished
   *    eg: if metadata reflects 1.0.1, 1.0.2 and 1.0.3, the computed metadata won't include 1.0.3.
   *  3. Once the second step has been successfully finished, delete the tarball.
   *
   *  All these steps are consecutive and required, there is no transacions here, if step 3 fails, metadata might
   *  get corrupted.
   *
   *  Note the unpublish call will suffix in the url a /-rev/14-5d500cfce92f90fd revision number, this not
   *  used internally.
   *
   *
   * Example flow of unpublish.
   *
   * npm http fetch GET 200 http://localhost:4873/@scope%2ftest1?write=true 1680ms
     npm http fetch PUT 201 http://localhost:4873/@scope%2ftest1/-rev/14-5d500cfce92f90fd 956606ms attempt #2
     npm http fetch GET 200 http://localhost:4873/@scope%2ftest1?write=true 1601ms
     npm http fetch DELETE 201 http://localhost:4873/@scope%2ftest1/-/test1-1.0.3.tgz/-rev/16-e11c8db282b2d992 19ms
   *
   * 3. Star a package
   *
   * Permissions: start a package depends of the publish and unpublish permissions, there is no specific flag for star or un start.
   * The URL for star is similar to the unpublish (change package format)
   *
   * npm has no enpoint for star a package, rather mutate the metadata and acts as, the difference is the
   * users property which is part of the payload and the body only includes
   *
   * {
    "_id": pkgName,
   	"_rev": "3-b0cdaefc9bdb77c8",
    "users": {
      [username]: boolean value (true, false)
    }
  }
   *
   */

  router.put('/:package/:_rev?/:revision?', can('publish'), (0, _middleware.media)(_mime.default.getType('json')), _middleware.expectJson, publishPackage(storage, config, auth));
  /**
   * Un-publishing an entire package.
   *
   * This scenario happens when the first call detect there is only one version remaining
   * in the metadata, then the client decides to DELETE the resource
   * npm http fetch GET 304 http://localhost:4873/@scope%2ftest1?write=true 1076ms (from cache)
     npm http fetch DELETE 201 http://localhost:4873/@scope%2ftest1/-rev/18-d8ebe3020bd4ac9c 22ms
   */

  router.delete('/:package/-rev/*', can('unpublish'), unPublishPackage(storage)); // removing a tarball

  router.delete('/:package/-/:filename/-rev/:revision', can('unpublish'), can('publish'), removeTarball(storage)); // uploading package tarball

  router.put('/:package/-/:filename/*', can('publish'), (0, _middleware.media)(_constants.HEADERS.OCTET_STREAM), uploadPackageTarball(storage)); // adding a version

  router.put('/:package/:version/-tag/:tag', can('publish'), (0, _middleware.media)(_mime.default.getType('json')), _middleware.expectJson, addVersion(storage));
}
/**
 * Publish a package
 */


function publishPackage(storage, config, auth) {
  const starApi = (0, _star.default)(storage);
  return function (req, res, next) {
    const packageName = req.params.package;
    debug('publishing or updating a new version for %o', packageName);
    /**
     * Write tarball of stream data from package clients.
     */

    const createTarball = function (filename, data, cb) {
      const stream = storage.addTarball(packageName, filename);
      stream.on('error', function (err) {
        cb(err);
      });
      stream.on('success', function () {
        cb();
      }); // this is dumb and memory-consuming, but what choices do we have?
      // flow: we need first refactor this file before decides which type use here

      stream.end(Buffer.from(data.data, 'base64'));
      stream.done();
    };
    /**
     * Add new package version in storage
     */


    const createVersion = function (version, metadata, cb) {
      storage.addVersion(packageName, version, metadata, null, cb);
    };
    /**
     * Add new tags in storage
     */


    const addTags = function (tags, cb) {
      storage.mergeTags(packageName, tags, cb);
    };

    const afterChange = function (error, okMessage, metadata) {
      const metadataCopy = _objectSpread({}, metadata);

      const {
        _attachments,
        versions
      } = metadataCopy; // `npm star` wouldn't have attachments
      // and `npm deprecate` would have attachments as a empty object, i.e {}

      if (_lodash.default.isNil(_attachments) || JSON.stringify(_attachments) === '{}') {
        if (error) {
          return next(error);
        }

        res.status(_constants.HTTP_STATUS.CREATED);
        return next({
          ok: okMessage,
          success: true
        });
      } // npm-registry-client 0.3+ embeds tarball into the json upload
      // https://github.com/isaacs/npm-registry-client/commit/e9fbeb8b67f249394f735c74ef11fe4720d46ca0
      // issue https://github.com/rlidwka/sinopia/issues/31, dealing with it here:


      const isInvalidBodyFormat = (0, _utils.isObject)(_attachments) === false || (0, _utils.hasDiffOneKey)(_attachments) || (0, _utils.isObject)(versions) === false || (0, _utils.hasDiffOneKey)(versions);

      if (isInvalidBodyFormat) {
        // npm is doing something strange again
        // if this happens in normal circumstances, report it as a bug
        _logger.logger.info({
          packageName
        }, `wrong package format on publish a package @{packageName}`);

        return next(_utils.ErrorCode.getBadRequest(_constants.API_ERROR.UNSUPORTED_REGISTRY_CALL));
      }

      if (error && error.status !== _constants.HTTP_STATUS.CONFLICT) {
        return next(error);
      } // at this point document is either created or existed before


      const [firstAttachmentKey] = Object.keys(_attachments);
      createTarball(_path.default.basename(firstAttachmentKey), _attachments[firstAttachmentKey], function (error) {
        if (error) {
          return next(error);
        }

        const versionToPublish = Object.keys(versions)[0];
        const versionMetadataToPublish = versions[versionToPublish];
        versionMetadataToPublish.readme = _lodash.default.isNil(versionMetadataToPublish.readme) === false ? String(versionMetadataToPublish.readme) : '';
        createVersion(versionToPublish, versionMetadataToPublish, function (error) {
          if (error) {
            return next(error);
          }

          addTags(metadataCopy[_constants.DIST_TAGS], async function (error) {
            if (error) {
              return next(error);
            }

            try {
              await (0, _notify.notify)(metadataCopy, config, req.remote_user, `${metadataCopy.name}@${versionToPublish}`);
            } catch (error) {
              _logger.logger.error({
                error
              }, 'notify batch service has failed: @{error}');
            }

            res.status(_constants.HTTP_STATUS.CREATED);
            return next({
              ok: okMessage,
              success: true
            });
          });
        });
      });
    };

    if ((0, _storageUtils.isPublishablePackage)(req.body) === false && (0, _utils.isObject)(req.body.users)) {
      return starApi(req, res, next);
    }

    try {
      const metadata = (0, _utils.validateMetadata)(req.body, packageName); // check _attachments to distinguish publish and deprecate

      if (req.params._rev || (0, _utils.isRelatedToDeprecation)(req.body) && _lodash.default.isEmpty(req.body._attachments)) {
        debug('updating a new version for %o', packageName); // we check unpublish permissions, an update is basically remove versions

        const remote = req.remote_user;
        auth.allow_unpublish({
          packageName
        }, remote, error => {
          if (error) {
            _logger.logger.error({
              packageName
            }, `not allowed to unpublish a version for @{packageName}`);

            return next(error);
          }

          storage.changePackage(packageName, metadata, req.params.revision, function (error) {
            afterChange(error, _constants.API_MESSAGE.PKG_CHANGED, metadata);
          });
        });
      } else {
        debug('adding a new version for %o', packageName);
        storage.addPackage(packageName, metadata, function (error) {
          afterChange(error, _constants.API_MESSAGE.PKG_CREATED, metadata);
        });
      }
    } catch (error) {
      _logger.logger.error({
        packageName
      }, 'error on publish, bad package data for @{packageName}');

      return next(_utils.ErrorCode.getBadData(_constants.API_ERROR.BAD_PACKAGE_DATA));
    }
  };
}
/**
 * un-publish a package
 */


function unPublishPackage(storage) {
  return function (req, res, next) {
    const packageName = req.params.package;
    debug('unpublishing %o', packageName);
    storage.removePackage(packageName, function (err) {
      if (err) {
        return next(err);
      }

      res.status(_constants.HTTP_STATUS.CREATED);
      return next({
        ok: _constants.API_MESSAGE.PKG_REMOVED
      });
    });
  };
}
/**
 * Delete tarball
 */


function removeTarball(storage) {
  return function (req, res, next) {
    const packageName = req.params.package;
    const {
      filename,
      revision
    } = req.params;
    debug('removing a tarball for %o-%o-%o', packageName, filename, revision);
    storage.removeTarball(packageName, filename, revision, function (err) {
      if (err) {
        return next(err);
      }

      res.status(_constants.HTTP_STATUS.CREATED);
      debug('success remove tarball for %o-%o-%o', packageName, filename, revision);
      return next({
        ok: _constants.API_MESSAGE.TARBALL_REMOVED
      });
    });
  };
}
/**
 * Adds a new version
 */


function addVersion(storage) {
  return function (req, res, next) {
    const {
      version,
      tag
    } = req.params;
    const packageName = req.params.package;
    storage.addVersion(packageName, version, req.body, tag, function (error) {
      if (error) {
        return next(error);
      }

      res.status(_constants.HTTP_STATUS.CREATED);
      return next({
        ok: _constants.API_MESSAGE.PKG_PUBLISHED
      });
    });
  };
}
/**
 * uploadPackageTarball
 */


function uploadPackageTarball(storage) {
  return function (req, res, next) {
    const packageName = req.params.package;
    const stream = storage.addTarball(packageName, req.params.filename);
    req.pipe(stream); // checking if end event came before closing

    let complete = false;
    req.on('end', function () {
      complete = true;
      stream.done();
    });
    req.on('close', function () {
      if (!complete) {
        stream.abort();
      }
    });
    stream.on('error', function (err) {
      return res.locals.report_error(err);
    });
    stream.on('success', function () {
      res.status(_constants.HTTP_STATUS.CREATED);
      return next({
        ok: _constants.API_MESSAGE.TARBALL_UPLOADED
      });
    });
  };
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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