"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = _default;

var _lodash = _interopRequireDefault(require("lodash"));

var _constants = require("../../../lib/constants");

var _utils = require("../../../lib/utils");

var _middleware = require("../../middleware");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const ABBREVIATED_HEADER = 'application/vnd.npm.install-v1+json; q=1.0, application/json; q=0.8, */*';

const downloadStream = (packageName, filename, storage, req, res) => {
  const stream = storage.getTarball(packageName, filename);
  stream.on('content-length', function (content) {
    res.header('Content-Length', content);
  });
  stream.on('error', function (err) {
    return res.locals.report_error(err);
  });
  res.header(_constants.HEADERS.CONTENT_TYPE, _constants.HEADERS.OCTET_STREAM);
  stream.pipe(res);
};

const redirectOrDownloadStream = (packageName, filename, storage, req, res, config) => {
  const tarballUrlRedirect = _lodash.default.get(config, 'experiments.tarball_url_redirect');

  storage.hasLocalTarball(packageName, filename).then(hasLocalTarball => {
    if (hasLocalTarball) {
      const context = {
        packageName,
        filename
      };
      const tarballUrl = typeof tarballUrlRedirect === 'function' ? tarballUrlRedirect(context) : _lodash.default.template(tarballUrlRedirect)(context);
      res.redirect(tarballUrl);
    } else {
      downloadStream(packageName, filename, storage, req, res);
    }
  }).catch(err => {
    res.locals.report_error(err);
  });
};

function _default(route, auth, storage, config) {
  const can = (0, _middleware.allow)(auth); // TODO: anonymous user?

  route.get('/:package/:version?', can('access'), function (req, res, next) {
    const getPackageMetaCallback = function (err, metadata) {
      if (err) {
        return next(err);
      }

      metadata = (0, _utils.convertDistRemoteToLocalTarballUrls)(metadata, req, config.url_prefix);
      let queryVersion = req.params.version;

      if (_lodash.default.isNil(queryVersion)) {
        return next(metadata);
      }

      let version = (0, _utils.getVersion)(metadata, queryVersion);

      if (_lodash.default.isNil(version) === false) {
        return next(version);
      }

      if (_lodash.default.isNil(metadata[_constants.DIST_TAGS]) === false) {
        if (_lodash.default.isNil(metadata[_constants.DIST_TAGS][queryVersion]) === false) {
          queryVersion = metadata[_constants.DIST_TAGS][queryVersion];
          version = (0, _utils.getVersion)(metadata, queryVersion);

          if (_lodash.default.isNil(version) === false) {
            return next(version);
          }
        }
      }

      return next(_utils.ErrorCode.getNotFound(`${_constants.API_ERROR.VERSION_NOT_EXIST}: ${req.params.version}`));
    };

    const abbreviated = req.get('Accept') === ABBREVIATED_HEADER;
    storage.getPackage({
      name: req.params.package,
      uplinksLook: true,
      req,
      abbreviated,
      callback: getPackageMetaCallback
    });
  });
  route.get('/:scopedPackage/-/:scope/:filename', can('access'), function (req, res) {
    const {
      scopedPackage,
      filename
    } = req.params;

    if (_lodash.default.get(config, 'experiments.tarball_url_redirect') === undefined) {
      downloadStream(scopedPackage, filename, storage, req, res);
    } else {
      redirectOrDownloadStream(scopedPackage, filename, storage, req, res, config);
    }
  });
  route.get('/:package/-/:filename', can('access'), function (req, res) {
    if (_lodash.default.get(config, 'experiments.tarball_url_redirect') === undefined) {
      downloadStream(req.params.package, req.params.filename, storage, req, res);
    } else {
      redirectOrDownloadStream(req.params.package, req.params.filename, storage, req, res, config);
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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