"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MetricAggregationType = exports.AdjustmentType = exports.StepScalingAction = void 0;
const cdk = require("@aws-cdk/core");
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
/**
 * Define a step scaling action.
 *
 * This kind of scaling policy adjusts the target capacity in configurable
 * steps. The size of the step is configurable based on the metric's distance
 * to its alarm threshold.
 *
 * This Action must be used as the target of a CloudWatch alarm to take effect.
 */
class StepScalingAction extends cdk.Construct {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.adjustments = new Array();
        // Cloudformation requires either the ResourceId, ScalableDimension, and ServiceNamespace
        // properties, or the ScalingTargetId property, but not both.
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-applicationautoscaling-scalingpolicy.html
        const resource = new applicationautoscaling_generated_1.CfnScalingPolicy(this, 'Resource', {
            policyName: props.policyName || this.node.uniqueId,
            policyType: 'StepScaling',
            scalingTargetId: props.scalingTarget.scalableTargetId,
            stepScalingPolicyConfiguration: {
                adjustmentType: props.adjustmentType,
                cooldown: props.cooldown && props.cooldown.toSeconds(),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                metricAggregationType: props.metricAggregationType,
                stepAdjustments: cdk.Lazy.anyValue({ produce: () => this.adjustments }),
            },
        });
        this.scalingPolicyArn = resource.ref;
    }
    /**
     * Add an adjusment interval to the ScalingAction.
     */
    addAdjustment(adjustment) {
        if (adjustment.lowerBound === undefined && adjustment.upperBound === undefined) {
            throw new Error('At least one of lowerBound or upperBound is required');
        }
        this.adjustments.push({
            metricIntervalLowerBound: adjustment.lowerBound,
            metricIntervalUpperBound: adjustment.upperBound,
            scalingAdjustment: adjustment.adjustment,
        });
    }
}
exports.StepScalingAction = StepScalingAction;
/**
 * How adjustment numbers are interpreted.
 */
var AdjustmentType;
(function (AdjustmentType) {
    AdjustmentType["CHANGE_IN_CAPACITY"] = "ChangeInCapacity";
    AdjustmentType["PERCENT_CHANGE_IN_CAPACITY"] = "PercentChangeInCapacity";
    AdjustmentType["EXACT_CAPACITY"] = "ExactCapacity";
})(AdjustmentType = exports.AdjustmentType || (exports.AdjustmentType = {}));
/**
 * How the scaling metric is going to be aggregated.
 */
var MetricAggregationType;
(function (MetricAggregationType) {
    MetricAggregationType["AVERAGE"] = "Average";
    MetricAggregationType["MINIMUM"] = "Minimum";
    MetricAggregationType["MAXIMUM"] = "Maximum";
})(MetricAggregationType = exports.MetricAggregationType || (exports.MetricAggregationType = {}));
//# sourceMappingURL=data:application/json;base64,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