"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServiceNamespace = exports.ScalableTarget = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
const step_scaling_policy_1 = require("./step-scaling-policy");
const target_tracking_scaling_policy_1 = require("./target-tracking-scaling-policy");
/**
 * Define a scalable target.
 */
class ScalableTarget extends core_1.Resource {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.actions = new Array();
        core_1.withResolved(props.maxCapacity, max => {
            if (max < 0) {
                throw new RangeError(`maxCapacity cannot be negative, got: ${props.maxCapacity}`);
            }
        });
        core_1.withResolved(props.minCapacity, min => {
            if (min < 0) {
                throw new RangeError(`minCapacity cannot be negative, got: ${props.minCapacity}`);
            }
        });
        core_1.withResolved(props.minCapacity, props.maxCapacity, (min, max) => {
            if (max < min) {
                throw new RangeError(`minCapacity (${props.minCapacity}) should be lower than maxCapacity (${props.maxCapacity})`);
            }
        });
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('application-autoscaling.amazonaws.com'),
        });
        const resource = new applicationautoscaling_generated_1.CfnScalableTarget(this, 'Resource', {
            maxCapacity: props.maxCapacity,
            minCapacity: props.minCapacity,
            resourceId: props.resourceId,
            roleArn: this.role.roleArn,
            scalableDimension: props.scalableDimension,
            scheduledActions: core_1.Lazy.anyValue({ produce: () => this.actions }, { omitEmptyArray: true }),
            serviceNamespace: props.serviceNamespace,
        });
        this.scalableTargetId = resource.ref;
    }
    /**
     *
     */
    static fromScalableTargetId(scope, id, scalableTargetId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.scalableTargetId = scalableTargetId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add a policy statement to the role's policy.
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Scale out or in based on time.
     */
    scaleOnSchedule(id, action) {
        if (action.minCapacity === undefined && action.maxCapacity === undefined) {
            throw new Error(`You must supply at least one of minCapacity or maxCapacity, got ${JSON.stringify(action)}`);
        }
        this.actions.push({
            scheduledActionName: id,
            schedule: action.schedule.expressionString,
            startTime: action.startTime,
            endTime: action.endTime,
            scalableTargetAction: {
                maxCapacity: action.maxCapacity,
                minCapacity: action.minCapacity,
            },
        });
    }
    /**
     * Scale out or in, in response to a metric.
     */
    scaleOnMetric(id, props) {
        return new step_scaling_policy_1.StepScalingPolicy(this, id, { ...props, scalingTarget: this });
    }
    /**
     * Scale out or in in order to keep a metric around a target value.
     */
    scaleToTrackMetric(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, id, { ...props, scalingTarget: this });
    }
}
exports.ScalableTarget = ScalableTarget;
/**
 * The service that supports Application AutoScaling.
 */
var ServiceNamespace;
(function (ServiceNamespace) {
    ServiceNamespace["ECS"] = "ecs";
    ServiceNamespace["ELASTIC_MAP_REDUCE"] = "elasticmapreduce";
    ServiceNamespace["EC2"] = "ec2";
    ServiceNamespace["APPSTREAM"] = "appstream";
    ServiceNamespace["DYNAMODB"] = "dynamodb";
    ServiceNamespace["RDS"] = "rds";
    ServiceNamespace["SAGEMAKER"] = "sagemaker";
    ServiceNamespace["CUSTOM_RESOURCE"] = "custom-resource";
    ServiceNamespace["LAMBDA"] = "lambda";
    ServiceNamespace["COMPREHEND"] = "comprehend";
})(ServiceNamespace = exports.ServiceNamespace || (exports.ServiceNamespace = {}));
//# sourceMappingURL=data:application/json;base64,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