"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const aws_codepipeline_1 = require("@aws-cdk/aws-codepipeline");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_codepipeline_actions_1 = require("@aws-cdk/aws-codepipeline-actions");
const aws_codebuild_1 = require("@aws-cdk/aws-codebuild");
const aws_cloudfront_1 = require("@aws-cdk/aws-cloudfront");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_route53_targets_1 = require("@aws-cdk/aws-route53-targets");
let standardBuildSpec = {
    version: '0.2',
    phases: {
        install: {
            "runtime-versions": {
                "nodejs": "12"
            }
        },
        build: {
            commands: [
                'npm install',
                `npm run build`
            ],
        },
    },
    artifacts: {
        files: ["**/*"],
        "base-directory": "dist"
    }
};
exports.DEFAULT_BUILD_SPEC = standardBuildSpec;
class CicdSpaWebsite extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.setupBucket();
        this.setupCloudFront();
        this.setupRoute53();
        this.setupCodePipeline();
    }
    setupBucket() {
        this.buildArtifactBucket = new aws_s3_1.Bucket(this, 'build-artifact-bucket', {
            encryption: aws_s3_1.BucketEncryption.KMS_MANAGED,
            bucketName: `${this.props.url.replace(/\./gi, '-')}-artifacts`,
            publicReadAccess: false,
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL
        });
        this.websiteBucket = new aws_s3_1.Bucket(this, 'website-bucket', {
            bucketName: `${this.props.url}`,
            ...(!this.props.certificate ? {
                websiteIndexDocument: 'index.html',
                websiteErrorDocument: 'index.html',
                publicReadAccess: true
            } : {})
        });
        if (this.props.certificate) {
            this.oai = new aws_cloudfront_1.OriginAccessIdentity(this, 'oai', {});
            this.websiteBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ["s3:GetObject"],
                principals: [this.oai.grantPrincipal]
            }));
        }
    }
    setupRoute53() {
        if (this.props.hostedZone) {
            const hostedZone = aws_route53_1.HostedZone.fromHostedZoneAttributes(this, 'zone', this.props.hostedZone);
            const recordName = this.props.url.split('.')[0];
            let target = aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.CloudFrontTarget(this.distribution));
            if (!this.props.certificate) {
                target = aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.BucketWebsiteTarget(this.websiteBucket));
            }
            new aws_route53_1.RecordSet(this, 'website-dns', {
                recordType: aws_route53_1.RecordType.A,
                recordName,
                zone: hostedZone,
                target: target
            });
        }
    }
    setupCodePipeline() {
        var _a;
        let sourceArtifact = new aws_codepipeline_1.Artifact('source-code');
        let compiledSite = new aws_codepipeline_1.Artifact("built-site");
        let buildSpec = aws_codebuild_1.BuildSpec.fromObject(exports.DEFAULT_BUILD_SPEC);
        if (this.props.buildSpec) {
            if (typeof (this.props.buildSpec) === "object") {
                buildSpec = aws_codebuild_1.BuildSpec.fromObject(this.props.buildSpec);
            }
            else if (typeof (this.props.buildSpec) === "string") {
                buildSpec = aws_codebuild_1.BuildSpec.fromSourceFilename(this.props.buildSpec);
            }
        }
        const project = new aws_codebuild_1.PipelineProject(this, `build-project`, {
            buildSpec,
            environment: {
                buildImage: aws_codebuild_1.LinuxBuildImage.AMAZON_LINUX_2_2,
                computeType: aws_codebuild_1.ComputeType.SMALL,
                privileged: true
            }
        });
        let accessControl = "public-read";
        if (this.props.certificate) {
            accessControl = "private";
        }
        const s3DeployAction = new aws_codepipeline_actions_1.S3DeployAction({
            actionName: 'copy-files',
            bucket: this.websiteBucket,
            input: compiledSite,
            runOrder: 1,
            // @ts-ignore
            accessControl
        });
        const invalidateLambda = new aws_lambda_1.Function(this, 'invalidate-function', {
            code: aws_lambda_1.Code.fromAsset('./lib/handlers/invalidate-cache'),
            environment: {},
            handler: "index.handler",
            runtime: aws_lambda_1.Runtime.NODEJS_10_X
        });
        invalidateLambda.addToRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                "codepipeline:PutJobSuccessResult",
                "cloudfront:CreateInvalidation"
            ],
            resources: ["*"]
        }));
        const invalidateCache = new aws_codepipeline_actions_1.LambdaInvokeAction({
            actionName: "invalidate-cache",
            lambda: invalidateLambda,
            // @ts-ignore
            userParameters: (_a = this.distribution) === null || _a === void 0 ? void 0 : _a.distributionId,
            runOrder: 2
        });
        let deployActions = [
            s3DeployAction
        ];
        if (this.props.certificate) {
            deployActions.push(invalidateCache);
        }
        const pipeline = new aws_codepipeline_1.Pipeline(this, "build-pipeline", {
            artifactBucket: this.buildArtifactBucket,
            pipelineName: `${this.props.url.replace(/\./gi, '-')}-build-pipeline`,
            stages: [
                {
                    stageName: "pull", actions: [
                        new aws_codepipeline_actions_1.GitHubSourceAction({
                            ...this.props.githubSource,
                            output: sourceArtifact,
                            actionName: "pull-from-github"
                        })
                    ]
                },
                {
                    stageName: "build", actions: [
                        new aws_codepipeline_actions_1.CodeBuildAction({
                            actionName: 'build',
                            input: sourceArtifact,
                            outputs: [compiledSite],
                            project
                        })
                    ]
                },
                {
                    stageName: "deploy", actions: deployActions
                },
            ]
        });
        pipeline.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                "s3:DeleteObject*",
                "s3:PutObject*",
                "s3:Abort*"
            ],
            resources: [
                this.websiteBucket.bucketArn,
                `${this.websiteBucket.bucketArn}/*`
            ],
            effect: aws_iam_1.Effect.ALLOW
        }));
    }
    setupCloudFront() {
        let certificateToUse = this.props.certificate;
        if (certificateToUse) {
            let certificate;
            if (typeof (certificateToUse) === "boolean") {
                if (!this.props.hostedZone) {
                    throw new Error("If you'd like a certificate then you must provide a `hostedZone`.");
                }
                certificate = new aws_certificatemanager_1.DnsValidatedCertificate(this, 'certificate', {
                    domainName: this.props.url,
                    hostedZone: aws_route53_1.HostedZone.fromHostedZoneAttributes(this, 'hosted-zone', this.props.hostedZone)
                });
            }
            else {
                certificate = certificateToUse;
            }
            this.distribution = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'site-distribution', {
                viewerCertificate: aws_cloudfront_1.ViewerCertificate.fromAcmCertificate(certificate),
                originConfigs: [
                    {
                        s3OriginSource: {
                            s3BucketSource: this.websiteBucket,
                            originAccessIdentity: this.oai
                        },
                        behaviors: [{ isDefaultBehavior: true }]
                    }
                ]
            });
        }
    }
}
exports.CicdSpaWebsite = CicdSpaWebsite;
//# sourceMappingURL=data:application/json;base64,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