"use strict";
// Copyright (c) Samuel Gratzl
Object.defineProperty(exports, "__esModule", { value: true });
const base_1 = require("@jupyter-widgets/base");
const version_1 = require("./version");
const bundle_1 = require("@upsetjs/bundle");
const utils_1 = require("./utils");
class UpSetModel extends base_1.DOMWidgetModel {
    defaults() {
        return Object.assign(Object.assign({}, super.defaults()), { _model_name: UpSetModel.model_name, _model_module: UpSetModel.model_module, _model_module_version: UpSetModel.model_module_version, _view_name: UpSetModel.view_name, _view_module: UpSetModel.view_module, _view_module_version: UpSetModel.view_module_version });
    }
}
exports.UpSetModel = UpSetModel;
UpSetModel.serializers = Object.assign({}, base_1.DOMWidgetModel.serializers);
UpSetModel.model_name = 'UpSetModel';
UpSetModel.model_module = version_1.MODULE_NAME;
UpSetModel.model_module_version = version_1.MODULE_VERSION;
UpSetModel.view_name = 'UpSetView'; // Set to null if no view
UpSetModel.view_module = version_1.MODULE_NAME; // Set to null if no view
UpSetModel.view_module_version = version_1.MODULE_VERSION;
class UpSetView extends base_1.DOMWidgetView {
    constructor() {
        super(...arguments);
        this.props = {
            sets: [],
            width: 300,
            height: 300,
        };
        this.elems = [];
        this.elemToIndex = new Map();
        this.changeSelection = (s) => {
            this.model.off('change', this.changed_prop, null);
            if (!s) {
                this.model.set('value', null);
            }
            else {
                const setish = {
                    name: s.name,
                    type: s.type,
                    cardinality: s.cardinality,
                    elems: s.elems.map((e) => this.elemToIndex.get(e)),
                };
                if (s.type !== 'set') {
                    setish.degree = s.degree;
                    setish.set_names = Array.from(s.sets).map((s) => s.name);
                }
                this.model.set('value', setish);
            }
            this.props.selection = s;
            this.renderImpl();
            this.touch();
            this.model.on('change', this.changed_prop, this);
        };
    }
    render() {
        this.model.on('change', this.changed_prop, this);
        this.updateProps(this.stateToProps());
    }
    stateToProps() {
        const props = {};
        const state = this.model.get_state(true);
        const toCamelCase = (v) => v.replace(/([-_]\w)/g, (g) => g[1].toUpperCase());
        const toPropName = (key) => {
            if (key === 'value') {
                return 'selection';
            }
            if (key.startsWith('_')) {
                return toCamelCase(key.slice(1));
            }
            return toCamelCase(key);
        };
        Object.keys(state).forEach((key) => {
            let v = state[key];
            if (v == null ||
                (Array.isArray(v) && v.length === 0) ||
                (key.startsWith('_') && key !== '_queries' && key !== '_sets') ||
                key === 'layout') {
                return;
            }
            const propName = toPropName(key);
            if (propName === 'fontSizes' || propName === 'combinations') {
                const converted = {};
                Object.keys(v).forEach((key) => {
                    converted[toCamelCase(key)] = v[key];
                });
                v = converted;
            }
            props[propName] = v;
        });
        return props;
    }
    updateProps(delta) {
        if (delta) {
            Object.assign(this.props, delta);
        }
        this.fixProps(delta);
        this.renderImpl();
    }
    fixProps(delta) {
        const props = this.props;
        if (delta.elems != null) {
            this.elems = delta.elems;
            this.elemToIndex.clear();
            this.elems.forEach((e, i) => this.elemToIndex.set(e, i));
        }
        if (delta.sets != null) {
            props.sets = utils_1.fixSets(props.sets, this.elems);
        }
        if (delta.combinations != null) {
            const c = utils_1.fixCombinations(props.combinations, props.sets, this.elems);
            if (c == null) {
                delete props.combinations;
            }
            else {
                props.combinations = c;
            }
        }
        if (delta.selection) {
            props.selection = utils_1.resolveSet(delta.selection, props.sets, props.combinations, this.elems);
        }
        if (delta.queries) {
            props.queries = delta.queries.map((query) => {
                if (bundle_1.isSetQuery(query)) {
                    return Object.assign({}, query, {
                        set: utils_1.resolveSet(query.set, props.sets, props.combinations, this.elems),
                    });
                }
                else if (bundle_1.isElemQuery(query)) {
                    return Object.assign({}, query, {
                        elems: Array.from(query.elems).map((i) => this.elems[i]),
                    });
                }
                return query;
            });
        }
        if (this.model.get('mode') === 'click') {
            props.onClick = this.changeSelection;
            delete props.onHover;
        }
        else {
            props.onHover = this.changeSelection;
            delete props.onClick;
        }
    }
    changed_prop(model, options) {
        console.log(model, options);
    }
    renderImpl() {
        const bb = this.el.getBoundingClientRect();
        const p = Object.assign({}, this.props);
        if (!bb.width || !bb.height) {
            requestAnimationFrame(() => {
                const bb2 = this.el.getBoundingClientRect();
                p.width = bb2.width || 600;
                p.height = bb2.height || 400;
                bundle_1.renderUpSet(this.el, p);
            });
            return;
        }
        p.width = bb.width;
        p.height = bb.height;
        bundle_1.renderUpSet(this.el, p);
    }
}
exports.UpSetView = UpSetView;
//# sourceMappingURL=widget.js.map