# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['django_object_actions', 'django_object_actions.tests']

package_data = \
{'': ['*'], 'django_object_actions': ['templates/django_object_actions/*']}

setup_kwargs = {
    'name': 'django-object-actions',
    'version': '4.1.0',
    'description': 'A Django app for adding object tools for models in the admin',
    'long_description': 'Django Object Actions\n=====================\n\n[![CI](https://github.com/crccheck/django-object-actions/actions/workflows/ci.yml/badge.svg?branch=master)](https://github.com/crccheck/django-object-actions/actions/workflows/ci.yml?query=branch%3Amaster)\n\nIf you\'ve ever tried making admin object tools you may have thought, "why can\'t\nthis be as easy as making Django Admin Actions?" Well now they can be.\n\n\nQuick-Start Guide\n-----------------\n\nInstall Django Object Actions:\n\n```shell\n$ pip install django-object-actions\n```\n\nAdd `django_object_actions` to your `INSTALLED_APPS` so Django can find\nour templates.\n\nIn your admin.py:\n\n```python\nfrom django_object_actions import DjangoObjectActions, action\n\nclass ArticleAdmin(DjangoObjectActions, admin.ModelAdmin):\n    @action(label="Publish", description="Submit this article") # optional\n    def publish_this(self, request, obj):\n        publish_obj(obj)\n\n    change_actions = (\'publish_this\', )\n```\n\n\nUsage\n-----\n\nDefining new &*tool actions* is just like defining regular [admin actions]. The\nmajor difference is the functions for `django-object-actions` will take an\nobject instance instead of a queryset (see *Re-using Admin Actions* below).\n\n*Tool actions* are exposed by putting them in a `change_actions` attribute in\nyour `admin.ModelAdmin`. You can also add *tool actions* to the main changelist\nviews too. There, you\'ll get a queryset like a regular [admin action][admin actions]:\n\n```python\nfrom django_object_actions import DjangoObjectActions\n\nclass MyModelAdmin(DjangoObjectActions, admin.ModelAdmin):\n    @action(\n        label="This will be the label of the button",  # optional\n        description="This will be the tooltip of the button" # optional\n    )\n    def toolfunc(self, request, obj):\n        pass\n\n    def make_published(modeladmin, request, queryset):\n        queryset.update(status=\'p\')\n\n    change_actions = (\'toolfunc\', )\n    changelist_actions = (\'make_published\', )\n```\n\nJust like admin actions, you can send a message with `self.message_user`.\nNormally, you would do something to the object and return to the same url, but\nif you return a `HttpResponse`, it will follow it (hey, just like [admin\nactions]!).\n\nIf your admin modifies `get_urls`, `change_view`, or `changelist_view`,\nyou\'ll need to take extra care because `django-object-actions` uses them too.\n\n### Re-using Admin Actions\n\nIf you would like a preexisting admin action to also be an *object action*, add\nthe `takes_instance_or_queryset` decorator to convert object instances into a\nqueryset and pass querysets:\n\n```python\nfrom django_object_actions import DjangoObjectActions, takes_instance_or_queryset\n\nclass RobotAdmin(DjangoObjectActions, admin.ModelAdmin):\n    # ... snip ...\n\n    @takes_instance_or_queryset\n    def tighten_lug_nuts(self, request, queryset):\n        queryset.update(lugnuts=F(\'lugnuts\') - 1)\n\n    change_actions = [\'tighten_lug_nuts\']\n    actions = [\'tighten_lug_nuts\']\n```\n\n[admin actions]: https://docs.djangoproject.com/en/stable/ref/contrib/admin/actions/\n\n### Customizing *Object Actions*\n\nTo give the action some a helpful title tooltip, you can use the `action` decorator\nand set the description argument.\n\n```python\n@action(description="Increment the vote count by one")\ndef increment_vote(self, request, obj):\n    obj.votes = obj.votes + 1\n    obj.save()\n```\n\nAlternatively, you can also add a `short_description` attribute,\nsimilar to how admin actions work:\n\n```python\ndef increment_vote(self, request, obj):\n    obj.votes = obj.votes + 1\n    obj.save()\nincrement_vote.short_description = "Increment the vote count by one"\n```\n\nBy default, Django Object Actions will guess what to label the button\nbased on the name of the function. You can override this with a `label`\nattribute:\n\n```python\n@action(label="Vote++")\ndef increment_vote(self, request, obj):\n    obj.votes = obj.votes + 1\n    obj.save()\n```\n\nor\n\n```python\ndef increment_vote(self, request, obj):\n    obj.votes = obj.votes + 1\n    obj.save()\nincrement_vote.label = "Vote++"\n```\n\nIf you need even more control, you can add arbitrary attributes to the buttons\nby adding a Django widget style\n[attrs](https://docs.djangoproject.com/en/stable/ref/forms/widgets/#django.forms.Widget.attrs)\nattribute:\n\n```python\n@action(attrs = {\'class\': \'addlink\'})\ndef increment_vote(self, request, obj):\n    obj.votes = obj.votes + 1\n    obj.save()\n```\n\nor\n\n```python\ndef increment_vote(self, request, obj):\n    obj.votes = obj.votes + 1\n    obj.save()\nincrement_vote.attrs = {\n    \'class\': \'addlink\',\n}\n```\n\n### Programmatically Disabling Actions\n\nYou can programmatically disable registered actions by defining your own\ncustom `get_change_actions()` method. In this example, certain actions\nonly apply to certain object states (e.g. You should not be able to\nclose an company account if the account is already closed):\n\n```python\ndef get_change_actions(self, request, object_id, form_url):\n    actions = super(PollAdmin, self).get_change_actions(request, object_id, form_url)\n    actions = list(actions)\n    if not request.user.is_superuser:\n        return []\n\n    obj = self.model.objects.get(pk=object_id)\n    if obj.question.endswith(\'?\'):\n        actions.remove(\'question_mark\')\n\n    return actions\n```\n\nThe same is true for changelist actions with `get_changelist_actions`.\n\n### Alternate Installation\n\nYou don\'t have to add this to `INSTALLED_APPS`, all you need to to do\nis copy the template `django_object_actions/change_form.html` some place\nDjango\'s template loader [will find\nit](https://docs.djangoproject.com/en/stable/ref/settings/#template-dirs).\n\nIf you don\'t intend to use the template customizations at all, don\'t\nadd `django_object_actions` to your `INSTALLED_APPS` at all and use\n`BaseDjangoObjectActions` instead of `DjangoObjectActions`.\n\n\nMore Examples\n-------------\n\nMaking an action that links off-site:\n\n```python\ndef external_link(self, request, obj):\n    from django.http import HttpResponseRedirect\n    return HttpResponseRedirect(f\'https://example.com/{obj.id}\')\n```\n\n\nLimitations\n-----------\n\n1.  `django-object-actions` expects functions to be methods of the model\n    admin. While Django gives you a lot more options for their admin\n    actions.\n2.  If you provide your own custom `change_form.html`, you\'ll also need\n    to manually copy in the relevant bits of [our change form\n    ](./django_object_actions/templates/django_object_actions/change_form.html).\n3.  Security. This has been written with the assumption that everyone in\n    the Django admin belongs there. Permissions should be enforced in\n    your own actions irregardless of what this provides. Better default\n    security is planned for the future.\n\n\nPython and Django compatibility\n-------------------------------\n\nSee [`ci.yml`](./.github/workflows/ci.yml) for which Python and Django versions this supports.\n\n\nDemo Admin & Docker images\n--------------------------\n\nYou can try the demo admin against several versions of Django with these Docker\nimages: https://hub.docker.com/r/crccheck/django-object-actions/tags\n\nThis runs the example Django project in `./example_project` based on the "polls"\ntutorial. `admin.py` demos what you can do with this app.\n\n\nDevelopment\n-----------\n\nGetting started:\n\n```shell\n# get a copy of the code\ngit clone git@github.com:crccheck/django-object-actions.git\ncd django-object-actions\n# Install requirements\nmake install\nmake test  # run test suite\nmake quickstart  # runs \'make resetdb\' and some extra steps\n```\n\nVarious helpers are available as make commands. Type `make help` and\nview the `Makefile` to see what other things you can do.\n\nSome commands assume you are in the virtualenv. If you see\n"ModuleNotFoundError"s, try running `poetry shell` first.\n\n\nSimilar Packages\n----------------\n\nIf you want an actions menu for each row of your changelist, check out [Django\nAdmin Row Actions](https://github.com/DjangoAdminHackers/django-admin-row-actions).\n\nDjango Object Actions is very similar to\n[django-object-tools](https://github.com/praekelt/django-object-tools), but does\nnot require messing with your urls.py, does not do anything special with\npermissions, and uses the same patterns as making [admin actions].\n',
    'author': 'crccheck',
    'author_email': 'c@crccheck.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/crccheck/django-object-actions',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
