from copy import deepcopy

from django.db.models import Model
from functools import partial
from typing import Dict, Type

from channels.db import database_sync_to_async
from rest_framework import status

from djangochannelsrestframework.consumers import APIConsumerMetaclass
from djangochannelsrestframework.decorators import action
from djangochannelsrestframework.generics import GenericAsyncAPIConsumer
from djangochannelsrestframework.mixins import RetrieveModelMixin
from djangochannelsrestframework.observer import ModelObserver


class _GenericModelObserver:
    def __init__(self, func, **kwargs):
        self.func = func
        self._group_names = None
        self._serializer = None

    def bind_to_model(self, model_cls: Type[Model]) -> ModelObserver:
        observer = ModelObserver(func=self.func, model_cls=model_cls)
        observer.groups(self._group_names)
        observer.serializer(self._serializer)
        return observer

    def groups(self, func):
        self._group_names = func
        return self

    def serializer(self, func):
        self._serializer = func
        return self


class ObserverAPIConsumerMetaclass(APIConsumerMetaclass):
    def __new__(mcs, name, bases, body) -> Type[GenericAsyncAPIConsumer]:

        queryset = body.get("queryset", None)
        if queryset is not None:
            for attr_name, attr in body.items():
                if isinstance(attr, _GenericModelObserver):
                    body[attr_name] = attr.bind_to_model(model_cls=queryset.model)
            for base in bases:
                for attr_name in dir(base):
                    attr = getattr(base, attr_name)
                    if isinstance(attr, _GenericModelObserver):
                        body[attr_name] = attr.bind_to_model(model_cls=queryset.model)

        return super().__new__(mcs, name, bases, body)


class ObserverConsumerMixin(metaclass=ObserverAPIConsumerMetaclass):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.subscribed_requests = {}  # type: Dict[function, str]


class ObserverModelInstanceMixin(ObserverConsumerMixin, RetrieveModelMixin):
    @action()
    async def subscribe_instance(self, request_id=None, **kwargs):
        if request_id is None:
            raise ValueError("request_id must have a value set")
        # subscribe!
        instance = await database_sync_to_async(self.get_object)(**kwargs)
        await self.handle_instance_change.subscribe(instance=instance)
        self.subscribed_requests[self.__class__.handle_instance_change] = request_id

        return None, status.HTTP_201_CREATED

    @action()
    async def unsubscribe_instance(self, request_id=None, **kwargs):
        if request_id is None:
            raise ValueError("request_id must have a value set")
        # subscribe!
        instance = await database_sync_to_async(self.get_object)(**kwargs)
        await self.handle_instance_change.unsubscribe(instance=instance)
        del self.subscribed_requests[self.__class__.handle_instance_change]

        return None, status.HTTP_204_NO_CONTENT

    @_GenericModelObserver
    async def handle_instance_change(self, message: Dict, **kwargs):
        message = deepcopy(message)
        action = message.pop("action")
        message.pop("type")

        await self.handle_observed_action(
            action=action,
            request_id=self.subscribed_requests.get(
                self.__class__.handle_instance_change
            ),
            **message
        )

    @handle_instance_change.groups
    def handle_instance_change(self: ModelObserver, instance, *args, **kwargs):
        # one channel for all updates.
        yield "{}-model-{}-pk-{}".format(
            self.func.__name__.replace("_", "."), self.model_label, instance.pk
        )

    async def handle_observed_action(self, action: str, request_id: str, **kwargs):
        """
        run the action.
        """
        try:
            await self.check_permissions(action, **kwargs)

            reply = partial(self.reply, action=action, request_id=request_id)

            if action == "delete":
                await reply(data=kwargs, status=204)
                # send the delete
                return

            # the @action decorator will wrap non-async action into async ones.

            response = await self.retrieve(
                request_id=request_id, action=action, **kwargs
            )

            if isinstance(response, tuple):
                data, status = response
                await reply(data=data, status=status)

        except Exception as exc:
            await self.handle_exception(exc, action=action, request_id=request_id)
