from collections import namedtuple
import os
import pytest
import random
import subprocess

from filelock import FileLock
import pyslim
import msprime

from .recipe_specs import recipe_specs

class Outfiles:
    """
    Simple wrapper for a list of Outfile items: these specify a file path that SLiM can
    use to output a file, the SLiM variable name containing that path, for use in the
    recipe, a function to apply to the path to create an object for the unit test to
    access and a "name" (key) in the dictionary to access that object in the unit test.
    """
    Outfile = namedtuple("Outfile", "path, slim_name, post_process, key")

    @staticmethod
    def parse_pedigree_info(fname):
        # returns a dictionary whose keys are SLiM individual IDs, and whose values
        # are dictionaries with two entries:
        # - 'parents' is the SLiM IDs of the parents
        # - 'age' is a dictionary whose keys are tuples (SLiM generation, stage)
        #   and whose values are ages (keys not present are ones the indivdiual was
        #   not alive for)
        assert os.path.isfile(fname)
        out = {}
        with open(fname, 'r') as f:
            header = f.readline().split()
            assert header == ['generation', 'stage', 'individual', 'age', 'parent1', 'parent2']
            for line in f:
                gen, stage, ind, age, p1, p2 = line.split()
                gen = int(gen)
                ind = int(ind)
                age = int(age)
                parents = tuple([int(p) for p in (p1, p2) if p != "-1"])
                if ind not in out:
                    out[ind] = {
                            "parents" : parents,
                            "age" : {}
                            }
                else:
                    for p in parents:
                        assert p in out[ind]['parents']
                out[ind]['age'][(gen, stage)] = age
        return out

    def __init__(self, out_dir):
        self._outfiles = [
            self.Outfile(
                path=os.path.join(out_dir, "out.trees"),
                slim_name="TREES_FILE",  # The var containing the path name for SLiM output
                post_process=pyslim.load,  # function applied on path to create returned obj
                key="ts",  # The key to use for the post-processed item in the returned dict
            ),
            self.Outfile(
                path=os.path.join(out_dir, "out.pedigree"),
                slim_name="PEDIGREE_FILE",
                post_process=self.parse_pedigree_info,
                key="info",
            ),
            self.Outfile(
                path=os.path.join(out_dir, "out.trees"),
                slim_name="",  # Empty, therefore no need to pass to SLiM
                post_process=lambda path: pyslim.load(path, legacy_metadata=True),
                key="ts_legacy_metadata",
            ),
        ]
    def __getitem__(self, index):
        return self._outfiles[index]

    def __len__(self, index):
        return len(self._outfiles)
    
    def results(self):
        res = {"path": {}}
        for o in self._outfiles:
            res["path"][o.key] = o.path
            if os.path.isfile(o.path):
                res[o.key] = o.post_process(o.path) 
        return res
    

def run_slim(recipe, out_dir, recipe_dir="test_recipes", **kwargs):
    """
    Run the recipe, present in the recipe_dir, outputting to files in out_dir
    kwargs are passed as variables to SLiM (in addition to the outfiles)
    """
    script_dir = os.path.dirname(os.path.realpath(__file__))
    full_recipe = os.path.abspath(os.path.join(script_dir, recipe_dir, recipe))
    if not os.path.isfile(full_recipe):
        raise RuntimeError(f"{full_recipe} cannot be found")
    assert os.path.isdir(out_dir)  # should have been generated by the calling function
    outfiles = Outfiles(out_dir)
    slim_vars = []
    for o in outfiles:
        if o.slim_name != "":
            slim_vars += ["-d", f"{o.slim_name}=\"{o.path}\""]
    for k in kwargs:
        x = kwargs[k]
        if x is not None:
            if isinstance(x, str):
                x = f"'{x}'"
            if isinstance(x, bool):
                x = 'T' if x else 'F'
            slim_vars += ["-d", f"{k}={x}"]
    cmd = ["slim", "-s", "23"] + slim_vars + [full_recipe]
    print(f"Running {cmd}, outputting errors etc. to '{out_dir}/SLiM_run_output.log'")
    with open(os.path.join(out_dir, "SLiM_run_output.log"), "w") as out:
        retval = subprocess.call(cmd, stderr=subprocess.STDOUT, stdout=out)
        if retval != 0:
            raise RuntimeError(f"Could not run {' '.join(cmd)}")
    return outfiles.results()


class HelperFunctions:
    @classmethod
    def run_slim_restart(cls, in_ts, recipe, out_dir, **kwargs):
        # Saves out the tree sequence to a trees file and run the SLiM recipe
        # on it, saving to files in out_dir.
        infile = os.path.join(out_dir, "in.trees")
        in_ts.dump(infile)
        kwargs['TREES_IN'] = infile
        if 'STAGE' not in kwargs:
            kwargs['STAGE'] = in_ts.metadata['SLiM']['stage']
        results = run_slim(recipe, out_dir, **kwargs)
        return results["ts"]

    @classmethod
    def run_msprime_restart(cls, in_ts, out_dir, sex=None, WF=False):
        out_ts = cls.run_slim_restart(
            in_ts,
            "restart_msprime.slim",  # This is standard script defined in test_recipes
            out_dir,
            WF=WF, SEX=sex, L=int(in_ts.sequence_length))
        return out_ts

    @staticmethod
    def get_msprime_examples():
        # NOTE: we use DTWF below to avoid rounding of floating-point times
        # that occur with a continuous-time simulator
        demographic_events = [
            msprime.MassMigration(
            time=5, source=1, destination=0, proportion=1.0)
        ]
        seed = 6
        for n in [2, 10, 20]:
            for mutrate in [0.0]:
                for recrate in [0.0, 0.01]:
                    yield msprime.simulate(n, mutation_rate=mutrate,
                                           recombination_rate=recrate,
                                           length=200, random_seed=seed,
                                           model="dtwf")
                    seed += 1
                    population_configurations =[
                        msprime.PopulationConfiguration(
                        sample_size=n, initial_size=100),
                        msprime.PopulationConfiguration(
                        sample_size=n, initial_size=100)
                    ]
                    yield msprime.simulate(
                        population_configurations=population_configurations,
                        demographic_events=demographic_events,
                        recombination_rate=recrate,
                        mutation_rate=mutrate,
                        length=250, random_seed=seed,
                        model="dtwf")
                    seed += 1

@pytest.fixture
def helper_functions():
    return HelperFunctions

@pytest.fixture(scope="session", params=recipe_specs.keys())
def recipe(request, tmp_path_factory, worker_id):
    """
    Most logic in this fixture is to check whether we are running as a single proc, or as a worker.
    If a worker, and this is the first run, we need to lock the process until the simulation has finished.
    """
    recipe_name = request.param
    if worker_id == "master":
        out_dir = tmp_path_factory.getbasetemp() / recipe_name
    else:
        # get the temp directory shared by all workers, so that simulation files
        # are shared between workers
        out_dir = tmp_path_factory.getbasetemp().parent / recipe_name
    with FileLock(str(out_dir) + ".lock"):
        if out_dir.is_dir():
            ret = Outfiles(out_dir).results()
        else:
            os.makedirs(out_dir, exist_ok=True)
            ret = run_slim(recipe=recipe_name, out_dir=out_dir)
    
    ret.update(recipe_specs[recipe_name])
    return ret
