"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('Origin Access Identity', () => {
    test('With automatic comment', () => {
        const stack = new cdk.Stack();
        new lib_1.OriginAccessIdentity(stack, 'OAI');
        expect(stack).toMatchTemplate({
            Resources: {
                OAIE1EFC67F: {
                    Type: 'AWS::CloudFront::CloudFrontOriginAccessIdentity',
                    Properties: {
                        CloudFrontOriginAccessIdentityConfig: {
                            Comment: 'Allows CloudFront to reach the bucket',
                        },
                    },
                },
            },
        });
    });
    test('With provided comment', () => {
        const stack = new cdk.Stack();
        new lib_1.OriginAccessIdentity(stack, 'OAI', {
            comment: 'test comment',
        });
        expect(stack).toMatchTemplate({
            Resources: {
                OAIE1EFC67F: {
                    Type: 'AWS::CloudFront::CloudFrontOriginAccessIdentity',
                    Properties: {
                        CloudFrontOriginAccessIdentityConfig: {
                            Comment: 'test comment',
                        },
                    },
                },
            },
        });
    });
    test('Truncates long comments', () => {
        const stack = new cdk.Stack();
        new lib_1.OriginAccessIdentity(stack, 'OAI', {
            comment: 'This is a really long comment. Auto-generated comments based on ids of origins might sometimes be this long or even longer and that will break',
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::CloudFrontOriginAccessIdentity', {
            CloudFrontOriginAccessIdentityConfig: {
                Comment: 'This is a really long comment. Auto-generated comments based on ids of origins might sometimes be this long or even longer and t',
            },
        });
    });
    test('Builds ARN of CloudFront user', () => {
        const stack = new cdk.Stack();
        const oai = lib_1.OriginAccessIdentity.fromOriginAccessIdentityName(stack, 'OAI', 'OAITest');
        expect(oai.grantPrincipal.policyFragment.principalJson.AWS[0]).toMatch(/:iam::cloudfront:user\/CloudFront Origin Access Identity OAITest$/);
    });
});
//# sourceMappingURL=data:application/json;base64,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