"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const acm = require("@aws-cdk/aws-certificatemanager");
const lambda = require("@aws-cdk/aws-lambda");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const test_origin_1 = require("./test-origin");
let app;
let stack;
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'Stack', {
        env: { account: '1234', region: 'testregion' },
    });
});
test('minimal example renders correctly', () => {
    const origin = test_origin_1.defaultOrigin();
    new lib_1.Distribution(stack, 'MyDist', { defaultBehavior: { origin } });
    expect(stack).toHaveResource('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                ForwardedValues: { QueryString: false },
                TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                ViewerProtocolPolicy: 'allow-all',
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Origins: [{
                    DomainName: 'www.example.com',
                    Id: 'StackMyDistOrigin1D6D5E535',
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only',
                    },
                }],
        },
    });
});
test('exhaustive example of props renders correctly', () => {
    const origin = test_origin_1.defaultOrigin();
    const certificate = acm.Certificate.fromCertificateArn(stack, 'Cert', 'arn:aws:acm:us-east-1:123456789012:certificate/12345678-1234-1234-1234-123456789012');
    new lib_1.Distribution(stack, 'MyDist', {
        defaultBehavior: { origin },
        certificate,
        comment: 'a test',
        defaultRootObject: 'index.html',
        domainNames: ['example.com'],
        enabled: false,
        enableIpv6: false,
        enableLogging: true,
        geoRestriction: lib_1.GeoRestriction.blacklist('US', 'GB'),
        httpVersion: lib_1.HttpVersion.HTTP1_1,
        logFilePrefix: 'logs/',
        logIncludesCookies: true,
        priceClass: lib_1.PriceClass.PRICE_CLASS_100,
        webAclId: '473e64fd-f30b-4765-81a0-62ad96dd167a',
    });
    expect(stack).toHaveResource('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            Aliases: ['example.com'],
            DefaultCacheBehavior: {
                ForwardedValues: { QueryString: false },
                TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                ViewerProtocolPolicy: 'allow-all',
            },
            Comment: 'a test',
            DefaultRootObject: 'index.html',
            Enabled: false,
            HttpVersion: 'http1.1',
            IPV6Enabled: false,
            Logging: {
                Bucket: { 'Fn::GetAtt': ['MyDistLoggingBucket9B8976BC', 'DomainName'] },
                IncludeCookies: true,
                Prefix: 'logs/',
            },
            Origins: [{
                    DomainName: 'www.example.com',
                    Id: 'StackMyDistOrigin1D6D5E535',
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only',
                    },
                }],
            PriceClass: 'PriceClass_100',
            Restrictions: {
                GeoRestriction: {
                    Locations: ['US', 'GB'],
                    RestrictionType: 'blacklist',
                },
            },
            ViewerCertificate: {
                AcmCertificateArn: 'arn:aws:acm:us-east-1:123456789012:certificate/12345678-1234-1234-1234-123456789012',
                SslSupportMethod: 'sni-only',
                MinimumProtocolVersion: 'TLSv1.2_2019',
            },
            WebACLId: '473e64fd-f30b-4765-81a0-62ad96dd167a',
        },
    });
});
describe('multiple behaviors', () => {
    test('a second behavior can\'t be specified with the catch-all path pattern', () => {
        const origin = test_origin_1.defaultOrigin();
        expect(() => {
            new lib_1.Distribution(stack, 'MyDist', {
                defaultBehavior: { origin },
                additionalBehaviors: {
                    '*': { origin },
                },
            });
        }).toThrow(/Only the default behavior can have a path pattern of \'*\'/);
    });
    test('a second behavior can be added to the original origin', () => {
        const origin = test_origin_1.defaultOrigin();
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            additionalBehaviors: {
                'api/*': { origin },
            },
        });
        expect(stack).toHaveResource('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                DefaultCacheBehavior: {
                    ForwardedValues: { QueryString: false },
                    TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                    ViewerProtocolPolicy: 'allow-all',
                },
                CacheBehaviors: [{
                        PathPattern: 'api/*',
                        ForwardedValues: { QueryString: false },
                        TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                        ViewerProtocolPolicy: 'allow-all',
                    }],
                Enabled: true,
                HttpVersion: 'http2',
                IPV6Enabled: true,
                Origins: [{
                        DomainName: 'www.example.com',
                        Id: 'StackMyDistOrigin1D6D5E535',
                        CustomOriginConfig: {
                            OriginProtocolPolicy: 'https-only',
                        },
                    }],
            },
        });
    });
    test('a second behavior can be added to a secondary origin', () => {
        const origin = test_origin_1.defaultOrigin();
        const origin2 = test_origin_1.defaultOrigin('origin2.example.com');
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            additionalBehaviors: {
                'api/*': { origin: origin2 },
            },
        });
        expect(stack).toHaveResource('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                DefaultCacheBehavior: {
                    ForwardedValues: { QueryString: false },
                    TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                    ViewerProtocolPolicy: 'allow-all',
                },
                CacheBehaviors: [{
                        PathPattern: 'api/*',
                        ForwardedValues: { QueryString: false },
                        TargetOriginId: 'StackMyDistOrigin20B96F3AD',
                        ViewerProtocolPolicy: 'allow-all',
                    }],
                Enabled: true,
                HttpVersion: 'http2',
                IPV6Enabled: true,
                Origins: [{
                        DomainName: 'www.example.com',
                        Id: 'StackMyDistOrigin1D6D5E535',
                        CustomOriginConfig: {
                            OriginProtocolPolicy: 'https-only',
                        },
                    },
                    {
                        DomainName: 'origin2.example.com',
                        Id: 'StackMyDistOrigin20B96F3AD',
                        CustomOriginConfig: {
                            OriginProtocolPolicy: 'https-only',
                        },
                    }],
            },
        });
    });
    test('behavior creation order is preserved', () => {
        const origin = test_origin_1.defaultOrigin();
        const origin2 = test_origin_1.defaultOrigin('origin2.example.com');
        const dist = new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            additionalBehaviors: {
                'api/1*': { origin: origin2 },
            },
        });
        dist.addBehavior('api/2*', origin);
        expect(stack).toHaveResource('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                DefaultCacheBehavior: {
                    ForwardedValues: { QueryString: false },
                    TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                    ViewerProtocolPolicy: 'allow-all',
                },
                CacheBehaviors: [{
                        PathPattern: 'api/1*',
                        ForwardedValues: { QueryString: false },
                        TargetOriginId: 'StackMyDistOrigin20B96F3AD',
                        ViewerProtocolPolicy: 'allow-all',
                    },
                    {
                        PathPattern: 'api/2*',
                        ForwardedValues: { QueryString: false },
                        TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                        ViewerProtocolPolicy: 'allow-all',
                    }],
                Enabled: true,
                HttpVersion: 'http2',
                IPV6Enabled: true,
                Origins: [{
                        DomainName: 'www.example.com',
                        Id: 'StackMyDistOrigin1D6D5E535',
                        CustomOriginConfig: {
                            OriginProtocolPolicy: 'https-only',
                        },
                    },
                    {
                        DomainName: 'origin2.example.com',
                        Id: 'StackMyDistOrigin20B96F3AD',
                        CustomOriginConfig: {
                            OriginProtocolPolicy: 'https-only',
                        },
                    }],
            },
        });
    });
});
describe('certificates', () => {
    test('should fail if using an imported certificate from outside of us-east-1', () => {
        const origin = test_origin_1.defaultOrigin();
        const certificate = acm.Certificate.fromCertificateArn(stack, 'Cert', 'arn:aws:acm:eu-west-1:123456789012:certificate/12345678-1234-1234-1234-123456789012');
        expect(() => {
            new lib_1.Distribution(stack, 'Dist', {
                defaultBehavior: { origin },
                certificate,
            });
        }).toThrow(/Distribution certificates must be in the us-east-1 region and the certificate you provided is in eu-west-1./);
    });
    test('adding a certificate without a domain name throws', () => {
        const certificate = acm.Certificate.fromCertificateArn(stack, 'Cert', 'arn:aws:acm:us-east-1:123456789012:certificate/12345678-1234-1234-1234-123456789012');
        expect(() => {
            new lib_1.Distribution(stack, 'Dist1', {
                defaultBehavior: { origin: test_origin_1.defaultOrigin() },
                certificate,
            });
        }).toThrow(/Must specify at least one domain name/);
        expect(() => {
            new lib_1.Distribution(stack, 'Dist2', {
                defaultBehavior: { origin: test_origin_1.defaultOrigin() },
                domainNames: [],
                certificate,
            });
        }).toThrow(/Must specify at least one domain name/);
    });
    test('adding a certificate and domain renders the correct ViewerCertificate and Aliases property', () => {
        const certificate = acm.Certificate.fromCertificateArn(stack, 'Cert', 'arn:aws:acm:us-east-1:123456789012:certificate/12345678-1234-1234-1234-123456789012');
        new lib_1.Distribution(stack, 'Dist', {
            defaultBehavior: { origin: test_origin_1.defaultOrigin() },
            domainNames: ['example.com', 'www.example.com'],
            certificate,
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                Aliases: ['example.com', 'www.example.com'],
                ViewerCertificate: {
                    AcmCertificateArn: 'arn:aws:acm:us-east-1:123456789012:certificate/12345678-1234-1234-1234-123456789012',
                    SslSupportMethod: 'sni-only',
                    MinimumProtocolVersion: 'TLSv1.2_2019',
                },
            },
        });
    });
});
describe('custom error responses', () => {
    test('should fail if responsePagePath is defined but responseCode is not', () => {
        const origin = test_origin_1.defaultOrigin();
        expect(() => {
            new lib_1.Distribution(stack, 'Dist', {
                defaultBehavior: { origin },
                errorResponses: [{
                        httpStatus: 404,
                        responsePagePath: '/errors/404.html',
                    }],
            });
        }).toThrow(/\'responseCode\' must be provided if \'responsePagePath\' is defined/);
    });
    test('should fail if only the error code is provided', () => {
        const origin = test_origin_1.defaultOrigin();
        expect(() => {
            new lib_1.Distribution(stack, 'Dist', {
                defaultBehavior: { origin },
                errorResponses: [{ httpStatus: 404 }],
            });
        }).toThrow(/A custom error response without either a \'responseCode\' or \'errorCachingMinTtl\' is not valid./);
    });
    test('should render the array of error configs if provided', () => {
        const origin = test_origin_1.defaultOrigin();
        new lib_1.Distribution(stack, 'Dist', {
            defaultBehavior: { origin },
            errorResponses: [{
                    httpStatus: 404,
                    responseHttpStatus: 404,
                    responsePagePath: '/errors/404.html',
                },
                {
                    httpStatus: 500,
                    ttl: core_1.Duration.seconds(2),
                }],
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                CustomErrorResponses: [
                    {
                        ErrorCode: 404,
                        ResponseCode: 404,
                        ResponsePagePath: '/errors/404.html',
                    },
                    {
                        ErrorCachingMinTTL: 2,
                        ErrorCode: 500,
                    },
                ],
            },
        });
    });
});
describe('logging', () => {
    test('does not include logging if disabled and no bucket provided', () => {
        const origin = test_origin_1.defaultOrigin();
        new lib_1.Distribution(stack, 'MyDist', { defaultBehavior: { origin } });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                Logging: assert_1.ABSENT,
            },
        });
    });
    test('throws error if logging disabled but bucket provided', () => {
        const origin = test_origin_1.defaultOrigin();
        expect(() => {
            new lib_1.Distribution(stack, 'MyDist', {
                defaultBehavior: { origin },
                enableLogging: false,
                logBucket: new s3.Bucket(stack, 'Bucket'),
            });
        }).toThrow(/Explicitly disabled logging but provided a logging bucket./);
    });
    test('creates bucket if none is provided', () => {
        const origin = test_origin_1.defaultOrigin();
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            enableLogging: true,
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                Logging: {
                    Bucket: { 'Fn::GetAtt': ['MyDistLoggingBucket9B8976BC', 'DomainName'] },
                },
            },
        });
    });
    test('uses existing bucket if provided', () => {
        const origin = test_origin_1.defaultOrigin();
        const loggingBucket = new s3.Bucket(stack, 'MyLoggingBucket');
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            logBucket: loggingBucket,
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                Logging: {
                    Bucket: { 'Fn::GetAtt': ['MyLoggingBucket4382CD04', 'DomainName'] },
                },
            },
        });
    });
    test('can set prefix and cookies', () => {
        const origin = test_origin_1.defaultOrigin();
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            enableLogging: true,
            logFilePrefix: 'logs/',
            logIncludesCookies: true,
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                Logging: {
                    Bucket: { 'Fn::GetAtt': ['MyDistLoggingBucket9B8976BC', 'DomainName'] },
                    IncludeCookies: true,
                    Prefix: 'logs/',
                },
            },
        });
    });
});
describe('with Lambda@Edge functions', () => {
    let lambdaFunction;
    let origin;
    beforeEach(() => {
        lambdaFunction = new lambda.Function(stack, 'Function', {
            runtime: lambda.Runtime.NODEJS,
            code: lambda.Code.fromInline('whatever'),
            handler: 'index.handler',
        });
        origin = test_origin_1.defaultOrigin();
    });
    test('can add an edge lambdas to the default behavior', () => {
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: {
                origin,
                edgeLambdas: [
                    {
                        functionVersion: lambdaFunction.currentVersion,
                        eventType: lib_1.LambdaEdgeEventType.ORIGIN_REQUEST,
                        includeBody: true,
                    },
                ],
            },
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                DefaultCacheBehavior: {
                    LambdaFunctionAssociations: [
                        {
                            EventType: 'origin-request',
                            IncludeBody: true,
                            LambdaFunctionARN: {
                                Ref: 'FunctionCurrentVersion4E2B2261477a5ae8059bbaa7813f752292c0f65e',
                            },
                        },
                    ],
                },
            },
        });
    });
    test('edgelambda.amazonaws.com is added to the trust policy of lambda', () => {
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: {
                origin,
                edgeLambdas: [
                    {
                        functionVersion: lambdaFunction.currentVersion,
                        eventType: lib_1.LambdaEdgeEventType.ORIGIN_REQUEST,
                    },
                ],
            },
        });
        expect(stack).toHaveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: 'sts:AssumeRole',
                        Effect: 'Allow',
                        Principal: {
                            Service: 'lambda.amazonaws.com',
                        },
                    },
                    {
                        Action: 'sts:AssumeRole',
                        Effect: 'Allow',
                        Principal: {
                            Service: 'edgelambda.amazonaws.com',
                        },
                    },
                ],
                Version: '2012-10-17',
            },
        });
    });
    test('can add an edge lambdas to additional behaviors', () => {
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            additionalBehaviors: {
                'images/*': {
                    origin,
                    edgeLambdas: [
                        {
                            functionVersion: lambdaFunction.currentVersion,
                            eventType: lib_1.LambdaEdgeEventType.VIEWER_REQUEST,
                        },
                    ],
                },
            },
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                CacheBehaviors: [
                    {
                        PathPattern: 'images/*',
                        LambdaFunctionAssociations: [
                            {
                                EventType: 'viewer-request',
                                LambdaFunctionARN: {
                                    Ref: 'FunctionCurrentVersion4E2B2261477a5ae8059bbaa7813f752292c0f65e',
                                },
                            },
                        ],
                    },
                ],
            },
        });
    });
    test('fails creation when attempting to add the $LATEST function version as an edge Lambda to the default behavior', () => {
        expect(() => {
            new lib_1.Distribution(stack, 'MyDist', {
                defaultBehavior: {
                    origin,
                    edgeLambdas: [
                        {
                            functionVersion: lambdaFunction.latestVersion,
                            eventType: lib_1.LambdaEdgeEventType.ORIGIN_RESPONSE,
                        },
                    ],
                },
            });
        }).toThrow(/\$LATEST function version cannot be used for Lambda@Edge/);
    });
    test('with removable env vars', () => {
        const envLambdaFunction = new lambda.Function(stack, 'EnvFunction', {
            runtime: lambda.Runtime.NODEJS,
            code: lambda.Code.fromInline('whateverwithenv'),
            handler: 'index.handler',
        });
        envLambdaFunction.addEnvironment('KEY', 'value', { removeInEdge: true });
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: {
                origin,
                edgeLambdas: [
                    {
                        functionVersion: envLambdaFunction.currentVersion,
                        eventType: lib_1.LambdaEdgeEventType.ORIGIN_REQUEST,
                    },
                ],
            },
        });
        expect(stack).toHaveResource('AWS::Lambda::Function', {
            Environment: assert_1.ABSENT,
            Code: {
                ZipFile: 'whateverwithenv',
            },
        });
    });
    test('with incompatible env vars', () => {
        const envLambdaFunction = new lambda.Function(stack, 'EnvFunction', {
            runtime: lambda.Runtime.NODEJS,
            code: lambda.Code.fromInline('whateverwithenv'),
            handler: 'index.handler',
            environment: {
                KEY: 'value',
            },
        });
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: {
                origin,
                edgeLambdas: [
                    {
                        functionVersion: envLambdaFunction.currentVersion,
                        eventType: lib_1.LambdaEdgeEventType.ORIGIN_REQUEST,
                    },
                ],
            },
        });
        expect(() => app.synth()).toThrow(/KEY/);
    });
    test('with singleton function', () => {
        const singleton = new lambda.SingletonFunction(stack, 'Singleton', {
            uuid: 'singleton-for-cloudfront',
            runtime: lambda.Runtime.NODEJS_12_X,
            code: lambda.Code.fromInline('code'),
            handler: 'index.handler',
        });
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: {
                origin,
                edgeLambdas: [
                    {
                        functionVersion: singleton.currentVersion,
                        eventType: lib_1.LambdaEdgeEventType.ORIGIN_REQUEST,
                    },
                ],
            },
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                DefaultCacheBehavior: {
                    LambdaFunctionAssociations: [
                        {
                            EventType: 'origin-request',
                            LambdaFunctionARN: {
                                Ref: 'SingletonLambdasingletonforcloudfrontCurrentVersion0078406348a0962a52448a200cd0dbc0e22edb2a',
                            },
                        },
                    ],
                },
            },
        });
    });
});
test('price class is included if provided', () => {
    const origin = test_origin_1.defaultOrigin();
    new lib_1.Distribution(stack, 'Dist', {
        defaultBehavior: { origin },
        priceClass: lib_1.PriceClass.PRICE_CLASS_200,
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            PriceClass: 'PriceClass_200',
        },
    });
});
test('escape hatches are supported', () => {
    const dist = new lib_1.Distribution(stack, 'Dist', {
        defaultBehavior: { origin: test_origin_1.defaultOrigin },
    });
    const cfnDist = dist.node.defaultChild;
    cfnDist.addPropertyOverride('DistributionConfig.DefaultCacheBehavior.ForwardedValues.Headers', ['*']);
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                ForwardedValues: {
                    Headers: ['*'],
                },
            },
        },
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGlzdHJpYnV0aW9uLnRlc3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJkaXN0cmlidXRpb24udGVzdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLDRDQUF5QztBQUN6QyxnQ0FBOEI7QUFDOUIsdURBQXVEO0FBQ3ZELDhDQUE4QztBQUM5QyxzQ0FBc0M7QUFDdEMsd0NBQXFEO0FBQ3JELGdDQUE4SDtBQUM5SCwrQ0FBOEM7QUFFOUMsSUFBSSxHQUFRLENBQUM7QUFDYixJQUFJLEtBQVksQ0FBQztBQUVqQixVQUFVLENBQUMsR0FBRyxFQUFFO0lBQ2QsR0FBRyxHQUFHLElBQUksVUFBRyxFQUFFLENBQUM7SUFDaEIsS0FBSyxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxPQUFPLEVBQUU7UUFDOUIsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLE1BQU0sRUFBRSxNQUFNLEVBQUUsWUFBWSxFQUFFO0tBQy9DLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLG1DQUFtQyxFQUFFLEdBQUcsRUFBRTtJQUM3QyxNQUFNLE1BQU0sR0FBRywyQkFBYSxFQUFFLENBQUM7SUFDL0IsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUUsRUFBRSxlQUFlLEVBQUUsRUFBRSxNQUFNLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFFbkUsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQywrQkFBK0IsRUFBRTtRQUM1RCxrQkFBa0IsRUFBRTtZQUNsQixvQkFBb0IsRUFBRTtnQkFDcEIsZUFBZSxFQUFFLEVBQUUsV0FBVyxFQUFFLEtBQUssRUFBRTtnQkFDdkMsY0FBYyxFQUFFLDRCQUE0QjtnQkFDNUMsb0JBQW9CLEVBQUUsV0FBVzthQUNsQztZQUNELE9BQU8sRUFBRSxJQUFJO1lBQ2IsV0FBVyxFQUFFLE9BQU87WUFDcEIsV0FBVyxFQUFFLElBQUk7WUFDakIsT0FBTyxFQUFFLENBQUM7b0JBQ1IsVUFBVSxFQUFFLGlCQUFpQjtvQkFDN0IsRUFBRSxFQUFFLDRCQUE0QjtvQkFDaEMsa0JBQWtCLEVBQUU7d0JBQ2xCLG9CQUFvQixFQUFFLFlBQVk7cUJBQ25DO2lCQUNGLENBQUM7U0FDSDtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLCtDQUErQyxFQUFFLEdBQUcsRUFBRTtJQUN6RCxNQUFNLE1BQU0sR0FBRywyQkFBYSxFQUFFLENBQUM7SUFDL0IsTUFBTSxXQUFXLEdBQUcsR0FBRyxDQUFDLFdBQVcsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFLHFGQUFxRixDQUFDLENBQUM7SUFFN0osSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7UUFDaEMsZUFBZSxFQUFFLEVBQUUsTUFBTSxFQUFFO1FBQzNCLFdBQVc7UUFDWCxPQUFPLEVBQUUsUUFBUTtRQUNqQixpQkFBaUIsRUFBRSxZQUFZO1FBQy9CLFdBQVcsRUFBRSxDQUFDLGFBQWEsQ0FBQztRQUM1QixPQUFPLEVBQUUsS0FBSztRQUNkLFVBQVUsRUFBRSxLQUFLO1FBQ2pCLGFBQWEsRUFBRSxJQUFJO1FBQ25CLGNBQWMsRUFBRSxvQkFBYyxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDO1FBQ3BELFdBQVcsRUFBRSxpQkFBVyxDQUFDLE9BQU87UUFDaEMsYUFBYSxFQUFFLE9BQU87UUFDdEIsa0JBQWtCLEVBQUUsSUFBSTtRQUN4QixVQUFVLEVBQUUsZ0JBQVUsQ0FBQyxlQUFlO1FBQ3RDLFFBQVEsRUFBRSxzQ0FBc0M7S0FDakQsQ0FBQyxDQUFDO0lBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQywrQkFBK0IsRUFBRTtRQUM1RCxrQkFBa0IsRUFBRTtZQUNsQixPQUFPLEVBQUUsQ0FBQyxhQUFhLENBQUM7WUFDeEIsb0JBQW9CLEVBQUU7Z0JBQ3BCLGVBQWUsRUFBRSxFQUFFLFdBQVcsRUFBRSxLQUFLLEVBQUU7Z0JBQ3ZDLGNBQWMsRUFBRSw0QkFBNEI7Z0JBQzVDLG9CQUFvQixFQUFFLFdBQVc7YUFDbEM7WUFDRCxPQUFPLEVBQUUsUUFBUTtZQUNqQixpQkFBaUIsRUFBRSxZQUFZO1lBQy9CLE9BQU8sRUFBRSxLQUFLO1lBQ2QsV0FBVyxFQUFFLFNBQVM7WUFDdEIsV0FBVyxFQUFFLEtBQUs7WUFDbEIsT0FBTyxFQUFFO2dCQUNQLE1BQU0sRUFBRSxFQUFFLFlBQVksRUFBRSxDQUFDLDZCQUE2QixFQUFFLFlBQVksQ0FBQyxFQUFFO2dCQUN2RSxjQUFjLEVBQUUsSUFBSTtnQkFDcEIsTUFBTSxFQUFFLE9BQU87YUFDaEI7WUFDRCxPQUFPLEVBQUUsQ0FBQztvQkFDUixVQUFVLEVBQUUsaUJBQWlCO29CQUM3QixFQUFFLEVBQUUsNEJBQTRCO29CQUNoQyxrQkFBa0IsRUFBRTt3QkFDbEIsb0JBQW9CLEVBQUUsWUFBWTtxQkFDbkM7aUJBQ0YsQ0FBQztZQUNGLFVBQVUsRUFBRSxnQkFBZ0I7WUFDNUIsWUFBWSxFQUFFO2dCQUNaLGNBQWMsRUFBRTtvQkFDZCxTQUFTLEVBQUUsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDO29CQUN2QixlQUFlLEVBQUUsV0FBVztpQkFDN0I7YUFDRjtZQUNELGlCQUFpQixFQUFFO2dCQUNqQixpQkFBaUIsRUFBRSxxRkFBcUY7Z0JBQ3hHLGdCQUFnQixFQUFFLFVBQVU7Z0JBQzVCLHNCQUFzQixFQUFFLGNBQWM7YUFDdkM7WUFDRCxRQUFRLEVBQUUsc0NBQXNDO1NBQ2pEO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxRQUFRLENBQUMsb0JBQW9CLEVBQUUsR0FBRyxFQUFFO0lBRWxDLElBQUksQ0FBQyx1RUFBdUUsRUFBRSxHQUFHLEVBQUU7UUFDakYsTUFBTSxNQUFNLEdBQUcsMkJBQWEsRUFBRSxDQUFDO1FBRS9CLE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDVixJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtnQkFDaEMsZUFBZSxFQUFFLEVBQUUsTUFBTSxFQUFFO2dCQUMzQixtQkFBbUIsRUFBRTtvQkFDbkIsR0FBRyxFQUFFLEVBQUUsTUFBTSxFQUFFO2lCQUNoQjthQUNGLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyw0REFBNEQsQ0FBQyxDQUFDO0lBQzNFLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHVEQUF1RCxFQUFFLEdBQUcsRUFBRTtRQUNqRSxNQUFNLE1BQU0sR0FBRywyQkFBYSxFQUFFLENBQUM7UUFDL0IsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7WUFDaEMsZUFBZSxFQUFFLEVBQUUsTUFBTSxFQUFFO1lBQzNCLG1CQUFtQixFQUFFO2dCQUNuQixPQUFPLEVBQUUsRUFBRSxNQUFNLEVBQUU7YUFDcEI7U0FDRixDQUFDLENBQUM7UUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLCtCQUErQixFQUFFO1lBQzVELGtCQUFrQixFQUFFO2dCQUNsQixvQkFBb0IsRUFBRTtvQkFDcEIsZUFBZSxFQUFFLEVBQUUsV0FBVyxFQUFFLEtBQUssRUFBRTtvQkFDdkMsY0FBYyxFQUFFLDRCQUE0QjtvQkFDNUMsb0JBQW9CLEVBQUUsV0FBVztpQkFDbEM7Z0JBQ0QsY0FBYyxFQUFFLENBQUM7d0JBQ2YsV0FBVyxFQUFFLE9BQU87d0JBQ3BCLGVBQWUsRUFBRSxFQUFFLFdBQVcsRUFBRSxLQUFLLEVBQUU7d0JBQ3ZDLGNBQWMsRUFBRSw0QkFBNEI7d0JBQzVDLG9CQUFvQixFQUFFLFdBQVc7cUJBQ2xDLENBQUM7Z0JBQ0YsT0FBTyxFQUFFLElBQUk7Z0JBQ2IsV0FBVyxFQUFFLE9BQU87Z0JBQ3BCLFdBQVcsRUFBRSxJQUFJO2dCQUNqQixPQUFPLEVBQUUsQ0FBQzt3QkFDUixVQUFVLEVBQUUsaUJBQWlCO3dCQUM3QixFQUFFLEVBQUUsNEJBQTRCO3dCQUNoQyxrQkFBa0IsRUFBRTs0QkFDbEIsb0JBQW9CLEVBQUUsWUFBWTt5QkFDbkM7cUJBQ0YsQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsc0RBQXNELEVBQUUsR0FBRyxFQUFFO1FBQ2hFLE1BQU0sTUFBTSxHQUFHLDJCQUFhLEVBQUUsQ0FBQztRQUMvQixNQUFNLE9BQU8sR0FBRywyQkFBYSxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDckQsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7WUFDaEMsZUFBZSxFQUFFLEVBQUUsTUFBTSxFQUFFO1lBQzNCLG1CQUFtQixFQUFFO2dCQUNuQixPQUFPLEVBQUUsRUFBRSxNQUFNLEVBQUUsT0FBTyxFQUFFO2FBQzdCO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQywrQkFBK0IsRUFBRTtZQUM1RCxrQkFBa0IsRUFBRTtnQkFDbEIsb0JBQW9CLEVBQUU7b0JBQ3BCLGVBQWUsRUFBRSxFQUFFLFdBQVcsRUFBRSxLQUFLLEVBQUU7b0JBQ3ZDLGNBQWMsRUFBRSw0QkFBNEI7b0JBQzVDLG9CQUFvQixFQUFFLFdBQVc7aUJBQ2xDO2dCQUNELGNBQWMsRUFBRSxDQUFDO3dCQUNmLFdBQVcsRUFBRSxPQUFPO3dCQUNwQixlQUFlLEVBQUUsRUFBRSxXQUFXLEVBQUUsS0FBSyxFQUFFO3dCQUN2QyxjQUFjLEVBQUUsNEJBQTRCO3dCQUM1QyxvQkFBb0IsRUFBRSxXQUFXO3FCQUNsQyxDQUFDO2dCQUNGLE9BQU8sRUFBRSxJQUFJO2dCQUNiLFdBQVcsRUFBRSxPQUFPO2dCQUNwQixXQUFXLEVBQUUsSUFBSTtnQkFDakIsT0FBTyxFQUFFLENBQUM7d0JBQ1IsVUFBVSxFQUFFLGlCQUFpQjt3QkFDN0IsRUFBRSxFQUFFLDRCQUE0Qjt3QkFDaEMsa0JBQWtCLEVBQUU7NEJBQ2xCLG9CQUFvQixFQUFFLFlBQVk7eUJBQ25DO3FCQUNGO29CQUNEO3dCQUNFLFVBQVUsRUFBRSxxQkFBcUI7d0JBQ2pDLEVBQUUsRUFBRSw0QkFBNEI7d0JBQ2hDLGtCQUFrQixFQUFFOzRCQUNsQixvQkFBb0IsRUFBRSxZQUFZO3lCQUNuQztxQkFDRixDQUFDO2FBQ0g7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxzQ0FBc0MsRUFBRSxHQUFHLEVBQUU7UUFDaEQsTUFBTSxNQUFNLEdBQUcsMkJBQWEsRUFBRSxDQUFDO1FBQy9CLE1BQU0sT0FBTyxHQUFHLDJCQUFhLENBQUMscUJBQXFCLENBQUMsQ0FBQztRQUNyRCxNQUFNLElBQUksR0FBRyxJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUM3QyxlQUFlLEVBQUUsRUFBRSxNQUFNLEVBQUU7WUFDM0IsbUJBQW1CLEVBQUU7Z0JBQ25CLFFBQVEsRUFBRSxFQUFFLE1BQU0sRUFBRSxPQUFPLEVBQUU7YUFDOUI7U0FDRixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsV0FBVyxDQUFDLFFBQVEsRUFBRSxNQUFNLENBQUMsQ0FBQztRQUVuQyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLCtCQUErQixFQUFFO1lBQzVELGtCQUFrQixFQUFFO2dCQUNsQixvQkFBb0IsRUFBRTtvQkFDcEIsZUFBZSxFQUFFLEVBQUUsV0FBVyxFQUFFLEtBQUssRUFBRTtvQkFDdkMsY0FBYyxFQUFFLDRCQUE0QjtvQkFDNUMsb0JBQW9CLEVBQUUsV0FBVztpQkFDbEM7Z0JBQ0QsY0FBYyxFQUFFLENBQUM7d0JBQ2YsV0FBVyxFQUFFLFFBQVE7d0JBQ3JCLGVBQWUsRUFBRSxFQUFFLFdBQVcsRUFBRSxLQUFLLEVBQUU7d0JBQ3ZDLGNBQWMsRUFBRSw0QkFBNEI7d0JBQzVDLG9CQUFvQixFQUFFLFdBQVc7cUJBQ2xDO29CQUNEO3dCQUNFLFdBQVcsRUFBRSxRQUFRO3dCQUNyQixlQUFlLEVBQUUsRUFBRSxXQUFXLEVBQUUsS0FBSyxFQUFFO3dCQUN2QyxjQUFjLEVBQUUsNEJBQTRCO3dCQUM1QyxvQkFBb0IsRUFBRSxXQUFXO3FCQUNsQyxDQUFDO2dCQUNGLE9BQU8sRUFBRSxJQUFJO2dCQUNiLFdBQVcsRUFBRSxPQUFPO2dCQUNwQixXQUFXLEVBQUUsSUFBSTtnQkFDakIsT0FBTyxFQUFFLENBQUM7d0JBQ1IsVUFBVSxFQUFFLGlCQUFpQjt3QkFDN0IsRUFBRSxFQUFFLDRCQUE0Qjt3QkFDaEMsa0JBQWtCLEVBQUU7NEJBQ2xCLG9CQUFvQixFQUFFLFlBQVk7eUJBQ25DO3FCQUNGO29CQUNEO3dCQUNFLFVBQVUsRUFBRSxxQkFBcUI7d0JBQ2pDLEVBQUUsRUFBRSw0QkFBNEI7d0JBQ2hDLGtCQUFrQixFQUFFOzRCQUNsQixvQkFBb0IsRUFBRSxZQUFZO3lCQUNuQztxQkFDRixDQUFDO2FBQ0g7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsUUFBUSxDQUFDLGNBQWMsRUFBRSxHQUFHLEVBQUU7SUFFNUIsSUFBSSxDQUFDLHdFQUF3RSxFQUFFLEdBQUcsRUFBRTtRQUNsRixNQUFNLE1BQU0sR0FBRywyQkFBYSxFQUFFLENBQUM7UUFDL0IsTUFBTSxXQUFXLEdBQUcsR0FBRyxDQUFDLFdBQVcsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFLHFGQUFxRixDQUFDLENBQUM7UUFFN0osTUFBTSxDQUFDLEdBQUcsRUFBRTtZQUNWLElBQUksa0JBQVksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUM5QixlQUFlLEVBQUUsRUFBRSxNQUFNLEVBQUU7Z0JBQzNCLFdBQVc7YUFDWixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsNkdBQTZHLENBQUMsQ0FBQztJQUM1SCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxtREFBbUQsRUFBRSxHQUFHLEVBQUU7UUFDN0QsTUFBTSxXQUFXLEdBQUcsR0FBRyxDQUFDLFdBQVcsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFLHFGQUFxRixDQUFDLENBQUM7UUFFN0osTUFBTSxDQUFDLEdBQUcsRUFBRTtZQUNWLElBQUksa0JBQVksQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFO2dCQUMvQixlQUFlLEVBQUUsRUFBRSxNQUFNLEVBQUUsMkJBQWEsRUFBRSxFQUFFO2dCQUM1QyxXQUFXO2FBQ1osQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLHVDQUF1QyxDQUFDLENBQUM7UUFFcEQsTUFBTSxDQUFDLEdBQUcsRUFBRTtZQUNWLElBQUksa0JBQVksQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFO2dCQUMvQixlQUFlLEVBQUUsRUFBRSxNQUFNLEVBQUUsMkJBQWEsRUFBRSxFQUFFO2dCQUM1QyxXQUFXLEVBQUUsRUFBRTtnQkFDZixXQUFXO2FBQ1osQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLHVDQUF1QyxDQUFDLENBQUM7SUFDdEQsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsNEZBQTRGLEVBQUUsR0FBRyxFQUFFO1FBQ3RHLE1BQU0sV0FBVyxHQUFHLEdBQUcsQ0FBQyxXQUFXLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRSxxRkFBcUYsQ0FBQyxDQUFDO1FBRTdKLElBQUksa0JBQVksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO1lBQzlCLGVBQWUsRUFBRSxFQUFFLE1BQU0sRUFBRSwyQkFBYSxFQUFFLEVBQUU7WUFDNUMsV0FBVyxFQUFFLENBQUMsYUFBYSxFQUFFLGlCQUFpQixDQUFDO1lBQy9DLFdBQVc7U0FDWixDQUFDLENBQUM7UUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsK0JBQStCLEVBQUU7WUFDaEUsa0JBQWtCLEVBQUU7Z0JBQ2xCLE9BQU8sRUFBRSxDQUFDLGFBQWEsRUFBRSxpQkFBaUIsQ0FBQztnQkFDM0MsaUJBQWlCLEVBQUU7b0JBQ2pCLGlCQUFpQixFQUFFLHFGQUFxRjtvQkFDeEcsZ0JBQWdCLEVBQUUsVUFBVTtvQkFDNUIsc0JBQXNCLEVBQUUsY0FBYztpQkFDdkM7YUFDRjtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxRQUFRLENBQUMsd0JBQXdCLEVBQUUsR0FBRyxFQUFFO0lBRXRDLElBQUksQ0FBQyxvRUFBb0UsRUFBRSxHQUFHLEVBQUU7UUFDOUUsTUFBTSxNQUFNLEdBQUcsMkJBQWEsRUFBRSxDQUFDO1FBRS9CLE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDVixJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDOUIsZUFBZSxFQUFFLEVBQUUsTUFBTSxFQUFFO2dCQUMzQixjQUFjLEVBQUUsQ0FBQzt3QkFDZixVQUFVLEVBQUUsR0FBRzt3QkFDZixnQkFBZ0IsRUFBRSxrQkFBa0I7cUJBQ3JDLENBQUM7YUFDSCxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsc0VBQXNFLENBQUMsQ0FBQztJQUNyRixDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxnREFBZ0QsRUFBRSxHQUFHLEVBQUU7UUFDMUQsTUFBTSxNQUFNLEdBQUcsMkJBQWEsRUFBRSxDQUFDO1FBRS9CLE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDVixJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDOUIsZUFBZSxFQUFFLEVBQUUsTUFBTSxFQUFFO2dCQUMzQixjQUFjLEVBQUUsQ0FBQyxFQUFFLFVBQVUsRUFBRSxHQUFHLEVBQUUsQ0FBQzthQUN0QyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsbUdBQW1HLENBQUMsQ0FBQztJQUNsSCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxzREFBc0QsRUFBRSxHQUFHLEVBQUU7UUFDaEUsTUFBTSxNQUFNLEdBQUcsMkJBQWEsRUFBRSxDQUFDO1FBQy9CLElBQUksa0JBQVksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO1lBQzlCLGVBQWUsRUFBRSxFQUFFLE1BQU0sRUFBRTtZQUMzQixjQUFjLEVBQUUsQ0FBQztvQkFDZixVQUFVLEVBQUUsR0FBRztvQkFDZixrQkFBa0IsRUFBRSxHQUFHO29CQUN2QixnQkFBZ0IsRUFBRSxrQkFBa0I7aUJBQ3JDO2dCQUNEO29CQUNFLFVBQVUsRUFBRSxHQUFHO29CQUNmLEdBQUcsRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztpQkFDekIsQ0FBQztTQUNILENBQUMsQ0FBQztRQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQywrQkFBK0IsRUFBRTtZQUNoRSxrQkFBa0IsRUFBRTtnQkFDbEIsb0JBQW9CLEVBQUU7b0JBQ3BCO3dCQUNFLFNBQVMsRUFBRSxHQUFHO3dCQUNkLFlBQVksRUFBRSxHQUFHO3dCQUNqQixnQkFBZ0IsRUFBRSxrQkFBa0I7cUJBQ3JDO29CQUNEO3dCQUNFLGtCQUFrQixFQUFFLENBQUM7d0JBQ3JCLFNBQVMsRUFBRSxHQUFHO3FCQUNmO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztBQUVMLENBQUMsQ0FBQyxDQUFDO0FBRUgsUUFBUSxDQUFDLFNBQVMsRUFBRSxHQUFHLEVBQUU7SUFDdkIsSUFBSSxDQUFDLDZEQUE2RCxFQUFFLEdBQUcsRUFBRTtRQUN2RSxNQUFNLE1BQU0sR0FBRywyQkFBYSxFQUFFLENBQUM7UUFDL0IsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUUsRUFBRSxlQUFlLEVBQUUsRUFBRSxNQUFNLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFbkUsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLCtCQUErQixFQUFFO1lBQ2hFLGtCQUFrQixFQUFFO2dCQUNsQixPQUFPLEVBQUUsZUFBTTthQUNoQjtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHNEQUFzRCxFQUFFLEdBQUcsRUFBRTtRQUNoRSxNQUFNLE1BQU0sR0FBRywyQkFBYSxFQUFFLENBQUM7UUFFL0IsTUFBTSxDQUFDLEdBQUcsRUFBRTtZQUNWLElBQUksa0JBQVksQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO2dCQUNoQyxlQUFlLEVBQUUsRUFBRSxNQUFNLEVBQUU7Z0JBQzNCLGFBQWEsRUFBRSxLQUFLO2dCQUNwQixTQUFTLEVBQUUsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxRQUFRLENBQUM7YUFDMUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLDREQUE0RCxDQUFDLENBQUM7SUFDM0UsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsb0NBQW9DLEVBQUUsR0FBRyxFQUFFO1FBQzlDLE1BQU0sTUFBTSxHQUFHLDJCQUFhLEVBQUUsQ0FBQztRQUMvQixJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUNoQyxlQUFlLEVBQUUsRUFBRSxNQUFNLEVBQUU7WUFDM0IsYUFBYSxFQUFFLElBQUk7U0FDcEIsQ0FBQyxDQUFDO1FBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLCtCQUErQixFQUFFO1lBQ2hFLGtCQUFrQixFQUFFO2dCQUNsQixPQUFPLEVBQUU7b0JBQ1AsTUFBTSxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUMsNkJBQTZCLEVBQUUsWUFBWSxDQUFDLEVBQUU7aUJBQ3hFO2FBQ0Y7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxrQ0FBa0MsRUFBRSxHQUFHLEVBQUU7UUFDNUMsTUFBTSxNQUFNLEdBQUcsMkJBQWEsRUFBRSxDQUFDO1FBQy9CLE1BQU0sYUFBYSxHQUFHLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsaUJBQWlCLENBQUMsQ0FBQztRQUM5RCxJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUNoQyxlQUFlLEVBQUUsRUFBRSxNQUFNLEVBQUU7WUFDM0IsU0FBUyxFQUFFLGFBQWE7U0FDekIsQ0FBQyxDQUFDO1FBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLCtCQUErQixFQUFFO1lBQ2hFLGtCQUFrQixFQUFFO2dCQUNsQixPQUFPLEVBQUU7b0JBQ1AsTUFBTSxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUMseUJBQXlCLEVBQUUsWUFBWSxDQUFDLEVBQUU7aUJBQ3BFO2FBQ0Y7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyw0QkFBNEIsRUFBRSxHQUFHLEVBQUU7UUFDdEMsTUFBTSxNQUFNLEdBQUcsMkJBQWEsRUFBRSxDQUFDO1FBQy9CLElBQUksa0JBQVksQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO1lBQ2hDLGVBQWUsRUFBRSxFQUFFLE1BQU0sRUFBRTtZQUMzQixhQUFhLEVBQUUsSUFBSTtZQUNuQixhQUFhLEVBQUUsT0FBTztZQUN0QixrQkFBa0IsRUFBRSxJQUFJO1NBQ3pCLENBQUMsQ0FBQztRQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQywrQkFBK0IsRUFBRTtZQUNoRSxrQkFBa0IsRUFBRTtnQkFDbEIsT0FBTyxFQUFFO29CQUNQLE1BQU0sRUFBRSxFQUFFLFlBQVksRUFBRSxDQUFDLDZCQUE2QixFQUFFLFlBQVksQ0FBQyxFQUFFO29CQUN2RSxjQUFjLEVBQUUsSUFBSTtvQkFDcEIsTUFBTSxFQUFFLE9BQU87aUJBQ2hCO2FBQ0Y7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsUUFBUSxDQUFDLDRCQUE0QixFQUFFLEdBQUcsRUFBRTtJQUMxQyxJQUFJLGNBQStCLENBQUM7SUFDcEMsSUFBSSxNQUFlLENBQUM7SUFFcEIsVUFBVSxDQUFDLEdBQUcsRUFBRTtRQUNkLGNBQWMsR0FBRyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTtZQUN0RCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxNQUFNO1lBQzlCLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxVQUFVLENBQUM7WUFDeEMsT0FBTyxFQUFFLGVBQWU7U0FDekIsQ0FBQyxDQUFDO1FBRUgsTUFBTSxHQUFHLDJCQUFhLEVBQUUsQ0FBQztJQUMzQixDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxpREFBaUQsRUFBRSxHQUFHLEVBQUU7UUFDM0QsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7WUFDaEMsZUFBZSxFQUFFO2dCQUNmLE1BQU07Z0JBQ04sV0FBVyxFQUFFO29CQUNYO3dCQUNFLGVBQWUsRUFBRSxjQUFjLENBQUMsY0FBYzt3QkFDOUMsU0FBUyxFQUFFLHlCQUFtQixDQUFDLGNBQWM7d0JBQzdDLFdBQVcsRUFBRSxJQUFJO3FCQUNsQjtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLCtCQUErQixFQUFFO1lBQ2hFLGtCQUFrQixFQUFFO2dCQUNsQixvQkFBb0IsRUFBRTtvQkFDcEIsMEJBQTBCLEVBQUU7d0JBQzFCOzRCQUNFLFNBQVMsRUFBRSxnQkFBZ0I7NEJBQzNCLFdBQVcsRUFBRSxJQUFJOzRCQUNqQixpQkFBaUIsRUFBRTtnQ0FDakIsR0FBRyxFQUFFLGdFQUFnRTs2QkFDdEU7eUJBQ0Y7cUJBQ0Y7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLGlFQUFpRSxFQUFFLEdBQUcsRUFBRTtRQUMzRSxJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUNoQyxlQUFlLEVBQUU7Z0JBQ2YsTUFBTTtnQkFDTixXQUFXLEVBQUU7b0JBQ1g7d0JBQ0UsZUFBZSxFQUFFLGNBQWMsQ0FBQyxjQUFjO3dCQUM5QyxTQUFTLEVBQUUseUJBQW1CLENBQUMsY0FBYztxQkFDOUM7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsZ0JBQWdCLEVBQUU7WUFDN0Msd0JBQXdCLEVBQUU7Z0JBQ3hCLFNBQVMsRUFBRTtvQkFDVDt3QkFDRSxNQUFNLEVBQUUsZ0JBQWdCO3dCQUN4QixNQUFNLEVBQUUsT0FBTzt3QkFDZixTQUFTLEVBQUU7NEJBQ1QsT0FBTyxFQUFFLHNCQUFzQjt5QkFDaEM7cUJBQ0Y7b0JBQ0Q7d0JBQ0UsTUFBTSxFQUFFLGdCQUFnQjt3QkFDeEIsTUFBTSxFQUFFLE9BQU87d0JBQ2YsU0FBUyxFQUFFOzRCQUNULE9BQU8sRUFBRSwwQkFBMEI7eUJBQ3BDO3FCQUNGO2lCQUNGO2dCQUNELE9BQU8sRUFBRSxZQUFZO2FBQ3RCO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsaURBQWlELEVBQUUsR0FBRyxFQUFFO1FBQzNELElBQUksa0JBQVksQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO1lBQ2hDLGVBQWUsRUFBRSxFQUFFLE1BQU0sRUFBRTtZQUMzQixtQkFBbUIsRUFBRTtnQkFDbkIsVUFBVSxFQUFFO29CQUNWLE1BQU07b0JBQ04sV0FBVyxFQUFFO3dCQUNYOzRCQUNFLGVBQWUsRUFBRSxjQUFjLENBQUMsY0FBYzs0QkFDOUMsU0FBUyxFQUFFLHlCQUFtQixDQUFDLGNBQWM7eUJBQzlDO3FCQUNGO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsK0JBQStCLEVBQUU7WUFDaEUsa0JBQWtCLEVBQUU7Z0JBQ2xCLGNBQWMsRUFBRTtvQkFDZDt3QkFDRSxXQUFXLEVBQUUsVUFBVTt3QkFDdkIsMEJBQTBCLEVBQUU7NEJBQzFCO2dDQUNFLFNBQVMsRUFBRSxnQkFBZ0I7Z0NBQzNCLGlCQUFpQixFQUFFO29DQUNqQixHQUFHLEVBQUUsZ0VBQWdFO2lDQUN0RTs2QkFDRjt5QkFDRjtxQkFDRjtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsOEdBQThHLEVBQUUsR0FBRyxFQUFFO1FBQ3hILE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDVixJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtnQkFDaEMsZUFBZSxFQUFFO29CQUNmLE1BQU07b0JBQ04sV0FBVyxFQUFFO3dCQUNYOzRCQUNFLGVBQWUsRUFBRSxjQUFjLENBQUMsYUFBYTs0QkFDN0MsU0FBUyxFQUFFLHlCQUFtQixDQUFDLGVBQWU7eUJBQy9DO3FCQUNGO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLDBEQUEwRCxDQUFDLENBQUM7SUFDekUsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMseUJBQXlCLEVBQUUsR0FBRyxFQUFFO1FBQ25DLE1BQU0saUJBQWlCLEdBQUcsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxhQUFhLEVBQUU7WUFDbEUsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsTUFBTTtZQUM5QixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsaUJBQWlCLENBQUM7WUFDL0MsT0FBTyxFQUFFLGVBQWU7U0FDekIsQ0FBQyxDQUFDO1FBQ0gsaUJBQWlCLENBQUMsY0FBYyxDQUFDLEtBQUssRUFBRSxPQUFPLEVBQUUsRUFBRSxZQUFZLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztRQUV6RSxJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUNoQyxlQUFlLEVBQUU7Z0JBQ2YsTUFBTTtnQkFDTixXQUFXLEVBQUU7b0JBQ1g7d0JBQ0UsZUFBZSxFQUFFLGlCQUFpQixDQUFDLGNBQWM7d0JBQ2pELFNBQVMsRUFBRSx5QkFBbUIsQ0FBQyxjQUFjO3FCQUM5QztpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyx1QkFBdUIsRUFBRTtZQUNwRCxXQUFXLEVBQUUsZUFBTTtZQUNuQixJQUFJLEVBQUU7Z0JBQ0osT0FBTyxFQUFFLGlCQUFpQjthQUMzQjtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLDRCQUE0QixFQUFFLEdBQUcsRUFBRTtRQUN0QyxNQUFNLGlCQUFpQixHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsYUFBYSxFQUFFO1lBQ2xFLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLE1BQU07WUFDOUIsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLGlCQUFpQixDQUFDO1lBQy9DLE9BQU8sRUFBRSxlQUFlO1lBQ3hCLFdBQVcsRUFBRTtnQkFDWCxHQUFHLEVBQUUsT0FBTzthQUNiO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7WUFDaEMsZUFBZSxFQUFFO2dCQUNmLE1BQU07Z0JBQ04sV0FBVyxFQUFFO29CQUNYO3dCQUNFLGVBQWUsRUFBRSxpQkFBaUIsQ0FBQyxjQUFjO3dCQUNqRCxTQUFTLEVBQUUseUJBQW1CLENBQUMsY0FBYztxQkFDOUM7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUMsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDM0MsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMseUJBQXlCLEVBQUUsR0FBRyxFQUFFO1FBQ25DLE1BQU0sU0FBUyxHQUFHLElBQUksTUFBTSxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxXQUFXLEVBQUU7WUFDakUsSUFBSSxFQUFFLDBCQUEwQjtZQUNoQyxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1lBQ25DLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxNQUFNLENBQUM7WUFDcEMsT0FBTyxFQUFFLGVBQWU7U0FDekIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7WUFDaEMsZUFBZSxFQUFFO2dCQUNmLE1BQU07Z0JBQ04sV0FBVyxFQUFFO29CQUNYO3dCQUNFLGVBQWUsRUFBRSxTQUFTLENBQUMsY0FBYzt3QkFDekMsU0FBUyxFQUFFLHlCQUFtQixDQUFDLGNBQWM7cUJBQzlDO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsK0JBQStCLEVBQUU7WUFDaEUsa0JBQWtCLEVBQUU7Z0JBQ2xCLG9CQUFvQixFQUFFO29CQUNwQiwwQkFBMEIsRUFBRTt3QkFDMUI7NEJBQ0UsU0FBUyxFQUFFLGdCQUFnQjs0QkFDM0IsaUJBQWlCLEVBQUU7Z0NBQ2pCLEdBQUcsRUFBRSw2RkFBNkY7NkJBQ25HO3lCQUNGO3FCQUNGO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHFDQUFxQyxFQUFFLEdBQUcsRUFBRTtJQUMvQyxNQUFNLE1BQU0sR0FBRywyQkFBYSxFQUFFLENBQUM7SUFDL0IsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7UUFDOUIsZUFBZSxFQUFFLEVBQUUsTUFBTSxFQUFFO1FBQzNCLFVBQVUsRUFBRSxnQkFBVSxDQUFDLGVBQWU7S0FDdkMsQ0FBQyxDQUFDO0lBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLCtCQUErQixFQUFFO1FBQ2hFLGtCQUFrQixFQUFFO1lBQ2xCLFVBQVUsRUFBRSxnQkFBZ0I7U0FDN0I7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyw4QkFBOEIsRUFBRSxHQUFHLEVBQUU7SUFDeEMsTUFBTSxJQUFJLEdBQUcsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7UUFDM0MsZUFBZSxFQUFFLEVBQUUsTUFBTSxFQUFFLDJCQUFhLEVBQUU7S0FDM0MsQ0FBQyxDQUFDO0lBQ0gsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUErQixDQUFDO0lBQzFELE9BQU8sQ0FBQyxtQkFBbUIsQ0FBQyxpRUFBaUUsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7SUFFdEcsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLCtCQUErQixFQUFFO1FBQ2hFLGtCQUFrQixFQUFFO1lBQ2xCLG9CQUFvQixFQUFFO2dCQUNwQixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFLENBQUMsR0FBRyxDQUFDO2lCQUNmO2FBQ0Y7U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQUJTRU5UIH0gZnJvbSAnQGF3cy1jZGsvYXNzZXJ0JztcbmltcG9ydCAnQGF3cy1jZGsvYXNzZXJ0L2plc3QnO1xuaW1wb3J0ICogYXMgYWNtIGZyb20gJ0Bhd3MtY2RrL2F3cy1jZXJ0aWZpY2F0ZW1hbmFnZXInO1xuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnO1xuaW1wb3J0ICogYXMgczMgZnJvbSAnQGF3cy1jZGsvYXdzLXMzJztcbmltcG9ydCB7IEFwcCwgRHVyYXRpb24sIFN0YWNrIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBDZm5EaXN0cmlidXRpb24sIERpc3RyaWJ1dGlvbiwgR2VvUmVzdHJpY3Rpb24sIEh0dHBWZXJzaW9uLCBJT3JpZ2luLCBMYW1iZGFFZGdlRXZlbnRUeXBlLCBQcmljZUNsYXNzIH0gZnJvbSAnLi4vbGliJztcbmltcG9ydCB7IGRlZmF1bHRPcmlnaW4gfSBmcm9tICcuL3Rlc3Qtb3JpZ2luJztcblxubGV0IGFwcDogQXBwO1xubGV0IHN0YWNrOiBTdGFjaztcblxuYmVmb3JlRWFjaCgoKSA9PiB7XG4gIGFwcCA9IG5ldyBBcHAoKTtcbiAgc3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnU3RhY2snLCB7XG4gICAgZW52OiB7IGFjY291bnQ6ICcxMjM0JywgcmVnaW9uOiAndGVzdHJlZ2lvbicgfSxcbiAgfSk7XG59KTtcblxudGVzdCgnbWluaW1hbCBleGFtcGxlIHJlbmRlcnMgY29ycmVjdGx5JywgKCkgPT4ge1xuICBjb25zdCBvcmlnaW4gPSBkZWZhdWx0T3JpZ2luKCk7XG4gIG5ldyBEaXN0cmlidXRpb24oc3RhY2ssICdNeURpc3QnLCB7IGRlZmF1bHRCZWhhdmlvcjogeyBvcmlnaW4gfSB9KTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OkNsb3VkRnJvbnQ6OkRpc3RyaWJ1dGlvbicsIHtcbiAgICBEaXN0cmlidXRpb25Db25maWc6IHtcbiAgICAgIERlZmF1bHRDYWNoZUJlaGF2aW9yOiB7XG4gICAgICAgIEZvcndhcmRlZFZhbHVlczogeyBRdWVyeVN0cmluZzogZmFsc2UgfSxcbiAgICAgICAgVGFyZ2V0T3JpZ2luSWQ6ICdTdGFja015RGlzdE9yaWdpbjFENkQ1RTUzNScsXG4gICAgICAgIFZpZXdlclByb3RvY29sUG9saWN5OiAnYWxsb3ctYWxsJyxcbiAgICAgIH0sXG4gICAgICBFbmFibGVkOiB0cnVlLFxuICAgICAgSHR0cFZlcnNpb246ICdodHRwMicsXG4gICAgICBJUFY2RW5hYmxlZDogdHJ1ZSxcbiAgICAgIE9yaWdpbnM6IFt7XG4gICAgICAgIERvbWFpbk5hbWU6ICd3d3cuZXhhbXBsZS5jb20nLFxuICAgICAgICBJZDogJ1N0YWNrTXlEaXN0T3JpZ2luMUQ2RDVFNTM1JyxcbiAgICAgICAgQ3VzdG9tT3JpZ2luQ29uZmlnOiB7XG4gICAgICAgICAgT3JpZ2luUHJvdG9jb2xQb2xpY3k6ICdodHRwcy1vbmx5JyxcbiAgICAgICAgfSxcbiAgICAgIH1dLFxuICAgIH0sXG4gIH0pO1xufSk7XG5cbnRlc3QoJ2V4aGF1c3RpdmUgZXhhbXBsZSBvZiBwcm9wcyByZW5kZXJzIGNvcnJlY3RseScsICgpID0+IHtcbiAgY29uc3Qgb3JpZ2luID0gZGVmYXVsdE9yaWdpbigpO1xuICBjb25zdCBjZXJ0aWZpY2F0ZSA9IGFjbS5DZXJ0aWZpY2F0ZS5mcm9tQ2VydGlmaWNhdGVBcm4oc3RhY2ssICdDZXJ0JywgJ2Fybjphd3M6YWNtOnVzLWVhc3QtMToxMjM0NTY3ODkwMTI6Y2VydGlmaWNhdGUvMTIzNDU2NzgtMTIzNC0xMjM0LTEyMzQtMTIzNDU2Nzg5MDEyJyk7XG5cbiAgbmV3IERpc3RyaWJ1dGlvbihzdGFjaywgJ015RGlzdCcsIHtcbiAgICBkZWZhdWx0QmVoYXZpb3I6IHsgb3JpZ2luIH0sXG4gICAgY2VydGlmaWNhdGUsXG4gICAgY29tbWVudDogJ2EgdGVzdCcsXG4gICAgZGVmYXVsdFJvb3RPYmplY3Q6ICdpbmRleC5odG1sJyxcbiAgICBkb21haW5OYW1lczogWydleGFtcGxlLmNvbSddLFxuICAgIGVuYWJsZWQ6IGZhbHNlLFxuICAgIGVuYWJsZUlwdjY6IGZhbHNlLFxuICAgIGVuYWJsZUxvZ2dpbmc6IHRydWUsXG4gICAgZ2VvUmVzdHJpY3Rpb246IEdlb1Jlc3RyaWN0aW9uLmJsYWNrbGlzdCgnVVMnLCAnR0InKSxcbiAgICBodHRwVmVyc2lvbjogSHR0cFZlcnNpb24uSFRUUDFfMSxcbiAgICBsb2dGaWxlUHJlZml4OiAnbG9ncy8nLFxuICAgIGxvZ0luY2x1ZGVzQ29va2llczogdHJ1ZSxcbiAgICBwcmljZUNsYXNzOiBQcmljZUNsYXNzLlBSSUNFX0NMQVNTXzEwMCxcbiAgICB3ZWJBY2xJZDogJzQ3M2U2NGZkLWYzMGItNDc2NS04MWEwLTYyYWQ5NmRkMTY3YScsXG4gIH0pO1xuXG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6Q2xvdWRGcm9udDo6RGlzdHJpYnV0aW9uJywge1xuICAgIERpc3RyaWJ1dGlvbkNvbmZpZzoge1xuICAgICAgQWxpYXNlczogWydleGFtcGxlLmNvbSddLFxuICAgICAgRGVmYXVsdENhY2hlQmVoYXZpb3I6IHtcbiAgICAgICAgRm9yd2FyZGVkVmFsdWVzOiB7IFF1ZXJ5U3RyaW5nOiBmYWxzZSB9LFxuICAgICAgICBUYXJnZXRPcmlnaW5JZDogJ1N0YWNrTXlEaXN0T3JpZ2luMUQ2RDVFNTM1JyxcbiAgICAgICAgVmlld2VyUHJvdG9jb2xQb2xpY3k6ICdhbGxvdy1hbGwnLFxuICAgICAgfSxcbiAgICAgIENvbW1lbnQ6ICdhIHRlc3QnLFxuICAgICAgRGVmYXVsdFJvb3RPYmplY3Q6ICdpbmRleC5odG1sJyxcbiAgICAgIEVuYWJsZWQ6IGZhbHNlLFxuICAgICAgSHR0cFZlcnNpb246ICdodHRwMS4xJyxcbiAgICAgIElQVjZFbmFibGVkOiBmYWxzZSxcbiAgICAgIExvZ2dpbmc6IHtcbiAgICAgICAgQnVja2V0OiB7ICdGbjo6R2V0QXR0JzogWydNeURpc3RMb2dnaW5nQnVja2V0OUI4OTc2QkMnLCAnRG9tYWluTmFtZSddIH0sXG4gICAgICAgIEluY2x1ZGVDb29raWVzOiB0cnVlLFxuICAgICAgICBQcmVmaXg6ICdsb2dzLycsXG4gICAgICB9LFxuICAgICAgT3JpZ2luczogW3tcbiAgICAgICAgRG9tYWluTmFtZTogJ3d3dy5leGFtcGxlLmNvbScsXG4gICAgICAgIElkOiAnU3RhY2tNeURpc3RPcmlnaW4xRDZENUU1MzUnLFxuICAgICAgICBDdXN0b21PcmlnaW5Db25maWc6IHtcbiAgICAgICAgICBPcmlnaW5Qcm90b2NvbFBvbGljeTogJ2h0dHBzLW9ubHknLFxuICAgICAgICB9LFxuICAgICAgfV0sXG4gICAgICBQcmljZUNsYXNzOiAnUHJpY2VDbGFzc18xMDAnLFxuICAgICAgUmVzdHJpY3Rpb25zOiB7XG4gICAgICAgIEdlb1Jlc3RyaWN0aW9uOiB7XG4gICAgICAgICAgTG9jYXRpb25zOiBbJ1VTJywgJ0dCJ10sXG4gICAgICAgICAgUmVzdHJpY3Rpb25UeXBlOiAnYmxhY2tsaXN0JyxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgICBWaWV3ZXJDZXJ0aWZpY2F0ZToge1xuICAgICAgICBBY21DZXJ0aWZpY2F0ZUFybjogJ2Fybjphd3M6YWNtOnVzLWVhc3QtMToxMjM0NTY3ODkwMTI6Y2VydGlmaWNhdGUvMTIzNDU2NzgtMTIzNC0xMjM0LTEyMzQtMTIzNDU2Nzg5MDEyJyxcbiAgICAgICAgU3NsU3VwcG9ydE1ldGhvZDogJ3NuaS1vbmx5JyxcbiAgICAgICAgTWluaW11bVByb3RvY29sVmVyc2lvbjogJ1RMU3YxLjJfMjAxOScsXG4gICAgICB9LFxuICAgICAgV2ViQUNMSWQ6ICc0NzNlNjRmZC1mMzBiLTQ3NjUtODFhMC02MmFkOTZkZDE2N2EnLFxuICAgIH0sXG4gIH0pO1xufSk7XG5cbmRlc2NyaWJlKCdtdWx0aXBsZSBiZWhhdmlvcnMnLCAoKSA9PiB7XG5cbiAgdGVzdCgnYSBzZWNvbmQgYmVoYXZpb3IgY2FuXFwndCBiZSBzcGVjaWZpZWQgd2l0aCB0aGUgY2F0Y2gtYWxsIHBhdGggcGF0dGVybicsICgpID0+IHtcbiAgICBjb25zdCBvcmlnaW4gPSBkZWZhdWx0T3JpZ2luKCk7XG5cbiAgICBleHBlY3QoKCkgPT4ge1xuICAgICAgbmV3IERpc3RyaWJ1dGlvbihzdGFjaywgJ015RGlzdCcsIHtcbiAgICAgICAgZGVmYXVsdEJlaGF2aW9yOiB7IG9yaWdpbiB9LFxuICAgICAgICBhZGRpdGlvbmFsQmVoYXZpb3JzOiB7XG4gICAgICAgICAgJyonOiB7IG9yaWdpbiB9LFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfSkudG9UaHJvdygvT25seSB0aGUgZGVmYXVsdCBiZWhhdmlvciBjYW4gaGF2ZSBhIHBhdGggcGF0dGVybiBvZiBcXCcqXFwnLyk7XG4gIH0pO1xuXG4gIHRlc3QoJ2Egc2Vjb25kIGJlaGF2aW9yIGNhbiBiZSBhZGRlZCB0byB0aGUgb3JpZ2luYWwgb3JpZ2luJywgKCkgPT4ge1xuICAgIGNvbnN0IG9yaWdpbiA9IGRlZmF1bHRPcmlnaW4oKTtcbiAgICBuZXcgRGlzdHJpYnV0aW9uKHN0YWNrLCAnTXlEaXN0Jywge1xuICAgICAgZGVmYXVsdEJlaGF2aW9yOiB7IG9yaWdpbiB9LFxuICAgICAgYWRkaXRpb25hbEJlaGF2aW9yczoge1xuICAgICAgICAnYXBpLyonOiB7IG9yaWdpbiB9LFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6Q2xvdWRGcm9udDo6RGlzdHJpYnV0aW9uJywge1xuICAgICAgRGlzdHJpYnV0aW9uQ29uZmlnOiB7XG4gICAgICAgIERlZmF1bHRDYWNoZUJlaGF2aW9yOiB7XG4gICAgICAgICAgRm9yd2FyZGVkVmFsdWVzOiB7IFF1ZXJ5U3RyaW5nOiBmYWxzZSB9LFxuICAgICAgICAgIFRhcmdldE9yaWdpbklkOiAnU3RhY2tNeURpc3RPcmlnaW4xRDZENUU1MzUnLFxuICAgICAgICAgIFZpZXdlclByb3RvY29sUG9saWN5OiAnYWxsb3ctYWxsJyxcbiAgICAgICAgfSxcbiAgICAgICAgQ2FjaGVCZWhhdmlvcnM6IFt7XG4gICAgICAgICAgUGF0aFBhdHRlcm46ICdhcGkvKicsXG4gICAgICAgICAgRm9yd2FyZGVkVmFsdWVzOiB7IFF1ZXJ5U3RyaW5nOiBmYWxzZSB9LFxuICAgICAgICAgIFRhcmdldE9yaWdpbklkOiAnU3RhY2tNeURpc3RPcmlnaW4xRDZENUU1MzUnLFxuICAgICAgICAgIFZpZXdlclByb3RvY29sUG9saWN5OiAnYWxsb3ctYWxsJyxcbiAgICAgICAgfV0sXG4gICAgICAgIEVuYWJsZWQ6IHRydWUsXG4gICAgICAgIEh0dHBWZXJzaW9uOiAnaHR0cDInLFxuICAgICAgICBJUFY2RW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgT3JpZ2luczogW3tcbiAgICAgICAgICBEb21haW5OYW1lOiAnd3d3LmV4YW1wbGUuY29tJyxcbiAgICAgICAgICBJZDogJ1N0YWNrTXlEaXN0T3JpZ2luMUQ2RDVFNTM1JyxcbiAgICAgICAgICBDdXN0b21PcmlnaW5Db25maWc6IHtcbiAgICAgICAgICAgIE9yaWdpblByb3RvY29sUG9saWN5OiAnaHR0cHMtb25seScsXG4gICAgICAgICAgfSxcbiAgICAgICAgfV0sXG4gICAgICB9LFxuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCdhIHNlY29uZCBiZWhhdmlvciBjYW4gYmUgYWRkZWQgdG8gYSBzZWNvbmRhcnkgb3JpZ2luJywgKCkgPT4ge1xuICAgIGNvbnN0IG9yaWdpbiA9IGRlZmF1bHRPcmlnaW4oKTtcbiAgICBjb25zdCBvcmlnaW4yID0gZGVmYXVsdE9yaWdpbignb3JpZ2luMi5leGFtcGxlLmNvbScpO1xuICAgIG5ldyBEaXN0cmlidXRpb24oc3RhY2ssICdNeURpc3QnLCB7XG4gICAgICBkZWZhdWx0QmVoYXZpb3I6IHsgb3JpZ2luIH0sXG4gICAgICBhZGRpdGlvbmFsQmVoYXZpb3JzOiB7XG4gICAgICAgICdhcGkvKic6IHsgb3JpZ2luOiBvcmlnaW4yIH0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpDbG91ZEZyb250OjpEaXN0cmlidXRpb24nLCB7XG4gICAgICBEaXN0cmlidXRpb25Db25maWc6IHtcbiAgICAgICAgRGVmYXVsdENhY2hlQmVoYXZpb3I6IHtcbiAgICAgICAgICBGb3J3YXJkZWRWYWx1ZXM6IHsgUXVlcnlTdHJpbmc6IGZhbHNlIH0sXG4gICAgICAgICAgVGFyZ2V0T3JpZ2luSWQ6ICdTdGFja015RGlzdE9yaWdpbjFENkQ1RTUzNScsXG4gICAgICAgICAgVmlld2VyUHJvdG9jb2xQb2xpY3k6ICdhbGxvdy1hbGwnLFxuICAgICAgICB9LFxuICAgICAgICBDYWNoZUJlaGF2aW9yczogW3tcbiAgICAgICAgICBQYXRoUGF0dGVybjogJ2FwaS8qJyxcbiAgICAgICAgICBGb3J3YXJkZWRWYWx1ZXM6IHsgUXVlcnlTdHJpbmc6IGZhbHNlIH0sXG4gICAgICAgICAgVGFyZ2V0T3JpZ2luSWQ6ICdTdGFja015RGlzdE9yaWdpbjIwQjk2RjNBRCcsXG4gICAgICAgICAgVmlld2VyUHJvdG9jb2xQb2xpY3k6ICdhbGxvdy1hbGwnLFxuICAgICAgICB9XSxcbiAgICAgICAgRW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgSHR0cFZlcnNpb246ICdodHRwMicsXG4gICAgICAgIElQVjZFbmFibGVkOiB0cnVlLFxuICAgICAgICBPcmlnaW5zOiBbe1xuICAgICAgICAgIERvbWFpbk5hbWU6ICd3d3cuZXhhbXBsZS5jb20nLFxuICAgICAgICAgIElkOiAnU3RhY2tNeURpc3RPcmlnaW4xRDZENUU1MzUnLFxuICAgICAgICAgIEN1c3RvbU9yaWdpbkNvbmZpZzoge1xuICAgICAgICAgICAgT3JpZ2luUHJvdG9jb2xQb2xpY3k6ICdodHRwcy1vbmx5JyxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgRG9tYWluTmFtZTogJ29yaWdpbjIuZXhhbXBsZS5jb20nLFxuICAgICAgICAgIElkOiAnU3RhY2tNeURpc3RPcmlnaW4yMEI5NkYzQUQnLFxuICAgICAgICAgIEN1c3RvbU9yaWdpbkNvbmZpZzoge1xuICAgICAgICAgICAgT3JpZ2luUHJvdG9jb2xQb2xpY3k6ICdodHRwcy1vbmx5JyxcbiAgICAgICAgICB9LFxuICAgICAgICB9XSxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2JlaGF2aW9yIGNyZWF0aW9uIG9yZGVyIGlzIHByZXNlcnZlZCcsICgpID0+IHtcbiAgICBjb25zdCBvcmlnaW4gPSBkZWZhdWx0T3JpZ2luKCk7XG4gICAgY29uc3Qgb3JpZ2luMiA9IGRlZmF1bHRPcmlnaW4oJ29yaWdpbjIuZXhhbXBsZS5jb20nKTtcbiAgICBjb25zdCBkaXN0ID0gbmV3IERpc3RyaWJ1dGlvbihzdGFjaywgJ015RGlzdCcsIHtcbiAgICAgIGRlZmF1bHRCZWhhdmlvcjogeyBvcmlnaW4gfSxcbiAgICAgIGFkZGl0aW9uYWxCZWhhdmlvcnM6IHtcbiAgICAgICAgJ2FwaS8xKic6IHsgb3JpZ2luOiBvcmlnaW4yIH0sXG4gICAgICB9LFxuICAgIH0pO1xuICAgIGRpc3QuYWRkQmVoYXZpb3IoJ2FwaS8yKicsIG9yaWdpbik7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OkNsb3VkRnJvbnQ6OkRpc3RyaWJ1dGlvbicsIHtcbiAgICAgIERpc3RyaWJ1dGlvbkNvbmZpZzoge1xuICAgICAgICBEZWZhdWx0Q2FjaGVCZWhhdmlvcjoge1xuICAgICAgICAgIEZvcndhcmRlZFZhbHVlczogeyBRdWVyeVN0cmluZzogZmFsc2UgfSxcbiAgICAgICAgICBUYXJnZXRPcmlnaW5JZDogJ1N0YWNrTXlEaXN0T3JpZ2luMUQ2RDVFNTM1JyxcbiAgICAgICAgICBWaWV3ZXJQcm90b2NvbFBvbGljeTogJ2FsbG93LWFsbCcsXG4gICAgICAgIH0sXG4gICAgICAgIENhY2hlQmVoYXZpb3JzOiBbe1xuICAgICAgICAgIFBhdGhQYXR0ZXJuOiAnYXBpLzEqJyxcbiAgICAgICAgICBGb3J3YXJkZWRWYWx1ZXM6IHsgUXVlcnlTdHJpbmc6IGZhbHNlIH0sXG4gICAgICAgICAgVGFyZ2V0T3JpZ2luSWQ6ICdTdGFja015RGlzdE9yaWdpbjIwQjk2RjNBRCcsXG4gICAgICAgICAgVmlld2VyUHJvdG9jb2xQb2xpY3k6ICdhbGxvdy1hbGwnLFxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgUGF0aFBhdHRlcm46ICdhcGkvMionLFxuICAgICAgICAgIEZvcndhcmRlZFZhbHVlczogeyBRdWVyeVN0cmluZzogZmFsc2UgfSxcbiAgICAgICAgICBUYXJnZXRPcmlnaW5JZDogJ1N0YWNrTXlEaXN0T3JpZ2luMUQ2RDVFNTM1JyxcbiAgICAgICAgICBWaWV3ZXJQcm90b2NvbFBvbGljeTogJ2FsbG93LWFsbCcsXG4gICAgICAgIH1dLFxuICAgICAgICBFbmFibGVkOiB0cnVlLFxuICAgICAgICBIdHRwVmVyc2lvbjogJ2h0dHAyJyxcbiAgICAgICAgSVBWNkVuYWJsZWQ6IHRydWUsXG4gICAgICAgIE9yaWdpbnM6IFt7XG4gICAgICAgICAgRG9tYWluTmFtZTogJ3d3dy5leGFtcGxlLmNvbScsXG4gICAgICAgICAgSWQ6ICdTdGFja015RGlzdE9yaWdpbjFENkQ1RTUzNScsXG4gICAgICAgICAgQ3VzdG9tT3JpZ2luQ29uZmlnOiB7XG4gICAgICAgICAgICBPcmlnaW5Qcm90b2NvbFBvbGljeTogJ2h0dHBzLW9ubHknLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBEb21haW5OYW1lOiAnb3JpZ2luMi5leGFtcGxlLmNvbScsXG4gICAgICAgICAgSWQ6ICdTdGFja015RGlzdE9yaWdpbjIwQjk2RjNBRCcsXG4gICAgICAgICAgQ3VzdG9tT3JpZ2luQ29uZmlnOiB7XG4gICAgICAgICAgICBPcmlnaW5Qcm90b2NvbFBvbGljeTogJ2h0dHBzLW9ubHknLFxuICAgICAgICAgIH0sXG4gICAgICAgIH1dLFxuICAgICAgfSxcbiAgICB9KTtcbiAgfSk7XG59KTtcblxuZGVzY3JpYmUoJ2NlcnRpZmljYXRlcycsICgpID0+IHtcblxuICB0ZXN0KCdzaG91bGQgZmFpbCBpZiB1c2luZyBhbiBpbXBvcnRlZCBjZXJ0aWZpY2F0ZSBmcm9tIG91dHNpZGUgb2YgdXMtZWFzdC0xJywgKCkgPT4ge1xuICAgIGNvbnN0IG9yaWdpbiA9IGRlZmF1bHRPcmlnaW4oKTtcbiAgICBjb25zdCBjZXJ0aWZpY2F0ZSA9IGFjbS5DZXJ0aWZpY2F0ZS5mcm9tQ2VydGlmaWNhdGVBcm4oc3RhY2ssICdDZXJ0JywgJ2Fybjphd3M6YWNtOmV1LXdlc3QtMToxMjM0NTY3ODkwMTI6Y2VydGlmaWNhdGUvMTIzNDU2NzgtMTIzNC0xMjM0LTEyMzQtMTIzNDU2Nzg5MDEyJyk7XG5cbiAgICBleHBlY3QoKCkgPT4ge1xuICAgICAgbmV3IERpc3RyaWJ1dGlvbihzdGFjaywgJ0Rpc3QnLCB7XG4gICAgICAgIGRlZmF1bHRCZWhhdmlvcjogeyBvcmlnaW4gfSxcbiAgICAgICAgY2VydGlmaWNhdGUsXG4gICAgICB9KTtcbiAgICB9KS50b1Rocm93KC9EaXN0cmlidXRpb24gY2VydGlmaWNhdGVzIG11c3QgYmUgaW4gdGhlIHVzLWVhc3QtMSByZWdpb24gYW5kIHRoZSBjZXJ0aWZpY2F0ZSB5b3UgcHJvdmlkZWQgaXMgaW4gZXUtd2VzdC0xLi8pO1xuICB9KTtcblxuICB0ZXN0KCdhZGRpbmcgYSBjZXJ0aWZpY2F0ZSB3aXRob3V0IGEgZG9tYWluIG5hbWUgdGhyb3dzJywgKCkgPT4ge1xuICAgIGNvbnN0IGNlcnRpZmljYXRlID0gYWNtLkNlcnRpZmljYXRlLmZyb21DZXJ0aWZpY2F0ZUFybihzdGFjaywgJ0NlcnQnLCAnYXJuOmF3czphY206dXMtZWFzdC0xOjEyMzQ1Njc4OTAxMjpjZXJ0aWZpY2F0ZS8xMjM0NTY3OC0xMjM0LTEyMzQtMTIzNC0xMjM0NTY3ODkwMTInKTtcblxuICAgIGV4cGVjdCgoKSA9PiB7XG4gICAgICBuZXcgRGlzdHJpYnV0aW9uKHN0YWNrLCAnRGlzdDEnLCB7XG4gICAgICAgIGRlZmF1bHRCZWhhdmlvcjogeyBvcmlnaW46IGRlZmF1bHRPcmlnaW4oKSB9LFxuICAgICAgICBjZXJ0aWZpY2F0ZSxcbiAgICAgIH0pO1xuICAgIH0pLnRvVGhyb3coL011c3Qgc3BlY2lmeSBhdCBsZWFzdCBvbmUgZG9tYWluIG5hbWUvKTtcblxuICAgIGV4cGVjdCgoKSA9PiB7XG4gICAgICBuZXcgRGlzdHJpYnV0aW9uKHN0YWNrLCAnRGlzdDInLCB7XG4gICAgICAgIGRlZmF1bHRCZWhhdmlvcjogeyBvcmlnaW46IGRlZmF1bHRPcmlnaW4oKSB9LFxuICAgICAgICBkb21haW5OYW1lczogW10sXG4gICAgICAgIGNlcnRpZmljYXRlLFxuICAgICAgfSk7XG4gICAgfSkudG9UaHJvdygvTXVzdCBzcGVjaWZ5IGF0IGxlYXN0IG9uZSBkb21haW4gbmFtZS8pO1xuICB9KTtcblxuICB0ZXN0KCdhZGRpbmcgYSBjZXJ0aWZpY2F0ZSBhbmQgZG9tYWluIHJlbmRlcnMgdGhlIGNvcnJlY3QgVmlld2VyQ2VydGlmaWNhdGUgYW5kIEFsaWFzZXMgcHJvcGVydHknLCAoKSA9PiB7XG4gICAgY29uc3QgY2VydGlmaWNhdGUgPSBhY20uQ2VydGlmaWNhdGUuZnJvbUNlcnRpZmljYXRlQXJuKHN0YWNrLCAnQ2VydCcsICdhcm46YXdzOmFjbTp1cy1lYXN0LTE6MTIzNDU2Nzg5MDEyOmNlcnRpZmljYXRlLzEyMzQ1Njc4LTEyMzQtMTIzNC0xMjM0LTEyMzQ1Njc4OTAxMicpO1xuXG4gICAgbmV3IERpc3RyaWJ1dGlvbihzdGFjaywgJ0Rpc3QnLCB7XG4gICAgICBkZWZhdWx0QmVoYXZpb3I6IHsgb3JpZ2luOiBkZWZhdWx0T3JpZ2luKCkgfSxcbiAgICAgIGRvbWFpbk5hbWVzOiBbJ2V4YW1wbGUuY29tJywgJ3d3dy5leGFtcGxlLmNvbSddLFxuICAgICAgY2VydGlmaWNhdGUsXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpDbG91ZEZyb250OjpEaXN0cmlidXRpb24nLCB7XG4gICAgICBEaXN0cmlidXRpb25Db25maWc6IHtcbiAgICAgICAgQWxpYXNlczogWydleGFtcGxlLmNvbScsICd3d3cuZXhhbXBsZS5jb20nXSxcbiAgICAgICAgVmlld2VyQ2VydGlmaWNhdGU6IHtcbiAgICAgICAgICBBY21DZXJ0aWZpY2F0ZUFybjogJ2Fybjphd3M6YWNtOnVzLWVhc3QtMToxMjM0NTY3ODkwMTI6Y2VydGlmaWNhdGUvMTIzNDU2NzgtMTIzNC0xMjM0LTEyMzQtMTIzNDU2Nzg5MDEyJyxcbiAgICAgICAgICBTc2xTdXBwb3J0TWV0aG9kOiAnc25pLW9ubHknLFxuICAgICAgICAgIE1pbmltdW1Qcm90b2NvbFZlcnNpb246ICdUTFN2MS4yXzIwMTknLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICB9KTtcbiAgfSk7XG59KTtcblxuZGVzY3JpYmUoJ2N1c3RvbSBlcnJvciByZXNwb25zZXMnLCAoKSA9PiB7XG5cbiAgdGVzdCgnc2hvdWxkIGZhaWwgaWYgcmVzcG9uc2VQYWdlUGF0aCBpcyBkZWZpbmVkIGJ1dCByZXNwb25zZUNvZGUgaXMgbm90JywgKCkgPT4ge1xuICAgIGNvbnN0IG9yaWdpbiA9IGRlZmF1bHRPcmlnaW4oKTtcblxuICAgIGV4cGVjdCgoKSA9PiB7XG4gICAgICBuZXcgRGlzdHJpYnV0aW9uKHN0YWNrLCAnRGlzdCcsIHtcbiAgICAgICAgZGVmYXVsdEJlaGF2aW9yOiB7IG9yaWdpbiB9LFxuICAgICAgICBlcnJvclJlc3BvbnNlczogW3tcbiAgICAgICAgICBodHRwU3RhdHVzOiA0MDQsXG4gICAgICAgICAgcmVzcG9uc2VQYWdlUGF0aDogJy9lcnJvcnMvNDA0Lmh0bWwnLFxuICAgICAgICB9XSxcbiAgICAgIH0pO1xuICAgIH0pLnRvVGhyb3coL1xcJ3Jlc3BvbnNlQ29kZVxcJyBtdXN0IGJlIHByb3ZpZGVkIGlmIFxcJ3Jlc3BvbnNlUGFnZVBhdGhcXCcgaXMgZGVmaW5lZC8pO1xuICB9KTtcblxuICB0ZXN0KCdzaG91bGQgZmFpbCBpZiBvbmx5IHRoZSBlcnJvciBjb2RlIGlzIHByb3ZpZGVkJywgKCkgPT4ge1xuICAgIGNvbnN0IG9yaWdpbiA9IGRlZmF1bHRPcmlnaW4oKTtcblxuICAgIGV4cGVjdCgoKSA9PiB7XG4gICAgICBuZXcgRGlzdHJpYnV0aW9uKHN0YWNrLCAnRGlzdCcsIHtcbiAgICAgICAgZGVmYXVsdEJlaGF2aW9yOiB7IG9yaWdpbiB9LFxuICAgICAgICBlcnJvclJlc3BvbnNlczogW3sgaHR0cFN0YXR1czogNDA0IH1dLFxuICAgICAgfSk7XG4gICAgfSkudG9UaHJvdygvQSBjdXN0b20gZXJyb3IgcmVzcG9uc2Ugd2l0aG91dCBlaXRoZXIgYSBcXCdyZXNwb25zZUNvZGVcXCcgb3IgXFwnZXJyb3JDYWNoaW5nTWluVHRsXFwnIGlzIG5vdCB2YWxpZC4vKTtcbiAgfSk7XG5cbiAgdGVzdCgnc2hvdWxkIHJlbmRlciB0aGUgYXJyYXkgb2YgZXJyb3IgY29uZmlncyBpZiBwcm92aWRlZCcsICgpID0+IHtcbiAgICBjb25zdCBvcmlnaW4gPSBkZWZhdWx0T3JpZ2luKCk7XG4gICAgbmV3IERpc3RyaWJ1dGlvbihzdGFjaywgJ0Rpc3QnLCB7XG4gICAgICBkZWZhdWx0QmVoYXZpb3I6IHsgb3JpZ2luIH0sXG4gICAgICBlcnJvclJlc3BvbnNlczogW3tcbiAgICAgICAgaHR0cFN0YXR1czogNDA0LFxuICAgICAgICByZXNwb25zZUh0dHBTdGF0dXM6IDQwNCxcbiAgICAgICAgcmVzcG9uc2VQYWdlUGF0aDogJy9lcnJvcnMvNDA0Lmh0bWwnLFxuICAgICAgfSxcbiAgICAgIHtcbiAgICAgICAgaHR0cFN0YXR1czogNTAwLFxuICAgICAgICB0dGw6IER1cmF0aW9uLnNlY29uZHMoMiksXG4gICAgICB9XSxcbiAgICB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNsb3VkRnJvbnQ6OkRpc3RyaWJ1dGlvbicsIHtcbiAgICAgIERpc3RyaWJ1dGlvbkNvbmZpZzoge1xuICAgICAgICBDdXN0b21FcnJvclJlc3BvbnNlczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEVycm9yQ29kZTogNDA0LFxuICAgICAgICAgICAgUmVzcG9uc2VDb2RlOiA0MDQsXG4gICAgICAgICAgICBSZXNwb25zZVBhZ2VQYXRoOiAnL2Vycm9ycy80MDQuaHRtbCcsXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBFcnJvckNhY2hpbmdNaW5UVEw6IDIsXG4gICAgICAgICAgICBFcnJvckNvZGU6IDUwMCxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9KTtcbiAgfSk7XG5cbn0pO1xuXG5kZXNjcmliZSgnbG9nZ2luZycsICgpID0+IHtcbiAgdGVzdCgnZG9lcyBub3QgaW5jbHVkZSBsb2dnaW5nIGlmIGRpc2FibGVkIGFuZCBubyBidWNrZXQgcHJvdmlkZWQnLCAoKSA9PiB7XG4gICAgY29uc3Qgb3JpZ2luID0gZGVmYXVsdE9yaWdpbigpO1xuICAgIG5ldyBEaXN0cmlidXRpb24oc3RhY2ssICdNeURpc3QnLCB7IGRlZmF1bHRCZWhhdmlvcjogeyBvcmlnaW4gfSB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNsb3VkRnJvbnQ6OkRpc3RyaWJ1dGlvbicsIHtcbiAgICAgIERpc3RyaWJ1dGlvbkNvbmZpZzoge1xuICAgICAgICBMb2dnaW5nOiBBQlNFTlQsXG4gICAgICB9LFxuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCd0aHJvd3MgZXJyb3IgaWYgbG9nZ2luZyBkaXNhYmxlZCBidXQgYnVja2V0IHByb3ZpZGVkJywgKCkgPT4ge1xuICAgIGNvbnN0IG9yaWdpbiA9IGRlZmF1bHRPcmlnaW4oKTtcblxuICAgIGV4cGVjdCgoKSA9PiB7XG4gICAgICBuZXcgRGlzdHJpYnV0aW9uKHN0YWNrLCAnTXlEaXN0Jywge1xuICAgICAgICBkZWZhdWx0QmVoYXZpb3I6IHsgb3JpZ2luIH0sXG4gICAgICAgIGVuYWJsZUxvZ2dpbmc6IGZhbHNlLFxuICAgICAgICBsb2dCdWNrZXQ6IG5ldyBzMy5CdWNrZXQoc3RhY2ssICdCdWNrZXQnKSxcbiAgICAgIH0pO1xuICAgIH0pLnRvVGhyb3coL0V4cGxpY2l0bHkgZGlzYWJsZWQgbG9nZ2luZyBidXQgcHJvdmlkZWQgYSBsb2dnaW5nIGJ1Y2tldC4vKTtcbiAgfSk7XG5cbiAgdGVzdCgnY3JlYXRlcyBidWNrZXQgaWYgbm9uZSBpcyBwcm92aWRlZCcsICgpID0+IHtcbiAgICBjb25zdCBvcmlnaW4gPSBkZWZhdWx0T3JpZ2luKCk7XG4gICAgbmV3IERpc3RyaWJ1dGlvbihzdGFjaywgJ015RGlzdCcsIHtcbiAgICAgIGRlZmF1bHRCZWhhdmlvcjogeyBvcmlnaW4gfSxcbiAgICAgIGVuYWJsZUxvZ2dpbmc6IHRydWUsXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpDbG91ZEZyb250OjpEaXN0cmlidXRpb24nLCB7XG4gICAgICBEaXN0cmlidXRpb25Db25maWc6IHtcbiAgICAgICAgTG9nZ2luZzoge1xuICAgICAgICAgIEJ1Y2tldDogeyAnRm46OkdldEF0dCc6IFsnTXlEaXN0TG9nZ2luZ0J1Y2tldDlCODk3NkJDJywgJ0RvbWFpbk5hbWUnXSB9LFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgndXNlcyBleGlzdGluZyBidWNrZXQgaWYgcHJvdmlkZWQnLCAoKSA9PiB7XG4gICAgY29uc3Qgb3JpZ2luID0gZGVmYXVsdE9yaWdpbigpO1xuICAgIGNvbnN0IGxvZ2dpbmdCdWNrZXQgPSBuZXcgczMuQnVja2V0KHN0YWNrLCAnTXlMb2dnaW5nQnVja2V0Jyk7XG4gICAgbmV3IERpc3RyaWJ1dGlvbihzdGFjaywgJ015RGlzdCcsIHtcbiAgICAgIGRlZmF1bHRCZWhhdmlvcjogeyBvcmlnaW4gfSxcbiAgICAgIGxvZ0J1Y2tldDogbG9nZ2luZ0J1Y2tldCxcbiAgICB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNsb3VkRnJvbnQ6OkRpc3RyaWJ1dGlvbicsIHtcbiAgICAgIERpc3RyaWJ1dGlvbkNvbmZpZzoge1xuICAgICAgICBMb2dnaW5nOiB7XG4gICAgICAgICAgQnVja2V0OiB7ICdGbjo6R2V0QXR0JzogWydNeUxvZ2dpbmdCdWNrZXQ0MzgyQ0QwNCcsICdEb21haW5OYW1lJ10gfSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2NhbiBzZXQgcHJlZml4IGFuZCBjb29raWVzJywgKCkgPT4ge1xuICAgIGNvbnN0IG9yaWdpbiA9IGRlZmF1bHRPcmlnaW4oKTtcbiAgICBuZXcgRGlzdHJpYnV0aW9uKHN0YWNrLCAnTXlEaXN0Jywge1xuICAgICAgZGVmYXVsdEJlaGF2aW9yOiB7IG9yaWdpbiB9LFxuICAgICAgZW5hYmxlTG9nZ2luZzogdHJ1ZSxcbiAgICAgIGxvZ0ZpbGVQcmVmaXg6ICdsb2dzLycsXG4gICAgICBsb2dJbmNsdWRlc0Nvb2tpZXM6IHRydWUsXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpDbG91ZEZyb250OjpEaXN0cmlidXRpb24nLCB7XG4gICAgICBEaXN0cmlidXRpb25Db25maWc6IHtcbiAgICAgICAgTG9nZ2luZzoge1xuICAgICAgICAgIEJ1Y2tldDogeyAnRm46OkdldEF0dCc6IFsnTXlEaXN0TG9nZ2luZ0J1Y2tldDlCODk3NkJDJywgJ0RvbWFpbk5hbWUnXSB9LFxuICAgICAgICAgIEluY2x1ZGVDb29raWVzOiB0cnVlLFxuICAgICAgICAgIFByZWZpeDogJ2xvZ3MvJyxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH0pO1xufSk7XG5cbmRlc2NyaWJlKCd3aXRoIExhbWJkYUBFZGdlIGZ1bmN0aW9ucycsICgpID0+IHtcbiAgbGV0IGxhbWJkYUZ1bmN0aW9uOiBsYW1iZGEuRnVuY3Rpb247XG4gIGxldCBvcmlnaW46IElPcmlnaW47XG5cbiAgYmVmb3JlRWFjaCgoKSA9PiB7XG4gICAgbGFtYmRhRnVuY3Rpb24gPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnRnVuY3Rpb24nLCB7XG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlMsXG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tSW5saW5lKCd3aGF0ZXZlcicpLFxuICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgIH0pO1xuXG4gICAgb3JpZ2luID0gZGVmYXVsdE9yaWdpbigpO1xuICB9KTtcblxuICB0ZXN0KCdjYW4gYWRkIGFuIGVkZ2UgbGFtYmRhcyB0byB0aGUgZGVmYXVsdCBiZWhhdmlvcicsICgpID0+IHtcbiAgICBuZXcgRGlzdHJpYnV0aW9uKHN0YWNrLCAnTXlEaXN0Jywge1xuICAgICAgZGVmYXVsdEJlaGF2aW9yOiB7XG4gICAgICAgIG9yaWdpbixcbiAgICAgICAgZWRnZUxhbWJkYXM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBmdW5jdGlvblZlcnNpb246IGxhbWJkYUZ1bmN0aW9uLmN1cnJlbnRWZXJzaW9uLFxuICAgICAgICAgICAgZXZlbnRUeXBlOiBMYW1iZGFFZGdlRXZlbnRUeXBlLk9SSUdJTl9SRVFVRVNULFxuICAgICAgICAgICAgaW5jbHVkZUJvZHk6IHRydWUsXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpDbG91ZEZyb250OjpEaXN0cmlidXRpb24nLCB7XG4gICAgICBEaXN0cmlidXRpb25Db25maWc6IHtcbiAgICAgICAgRGVmYXVsdENhY2hlQmVoYXZpb3I6IHtcbiAgICAgICAgICBMYW1iZGFGdW5jdGlvbkFzc29jaWF0aW9uczogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBFdmVudFR5cGU6ICdvcmlnaW4tcmVxdWVzdCcsXG4gICAgICAgICAgICAgIEluY2x1ZGVCb2R5OiB0cnVlLFxuICAgICAgICAgICAgICBMYW1iZGFGdW5jdGlvbkFSTjoge1xuICAgICAgICAgICAgICAgIFJlZjogJ0Z1bmN0aW9uQ3VycmVudFZlcnNpb240RTJCMjI2MTQ3N2E1YWU4MDU5YmJhYTc4MTNmNzUyMjkyYzBmNjVlJyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2VkZ2VsYW1iZGEuYW1hem9uYXdzLmNvbSBpcyBhZGRlZCB0byB0aGUgdHJ1c3QgcG9saWN5IG9mIGxhbWJkYScsICgpID0+IHtcbiAgICBuZXcgRGlzdHJpYnV0aW9uKHN0YWNrLCAnTXlEaXN0Jywge1xuICAgICAgZGVmYXVsdEJlaGF2aW9yOiB7XG4gICAgICAgIG9yaWdpbixcbiAgICAgICAgZWRnZUxhbWJkYXM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBmdW5jdGlvblZlcnNpb246IGxhbWJkYUZ1bmN0aW9uLmN1cnJlbnRWZXJzaW9uLFxuICAgICAgICAgICAgZXZlbnRUeXBlOiBMYW1iZGFFZGdlRXZlbnRUeXBlLk9SSUdJTl9SRVFVRVNULFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpJQU06OlJvbGUnLCB7XG4gICAgICBBc3N1bWVSb2xlUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgQWN0aW9uOiAnc3RzOkFzc3VtZVJvbGUnLFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgUHJpbmNpcGFsOiB7XG4gICAgICAgICAgICAgIFNlcnZpY2U6ICdsYW1iZGEuYW1hem9uYXdzLmNvbScsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgQWN0aW9uOiAnc3RzOkFzc3VtZVJvbGUnLFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgUHJpbmNpcGFsOiB7XG4gICAgICAgICAgICAgIFNlcnZpY2U6ICdlZGdlbGFtYmRhLmFtYXpvbmF3cy5jb20nLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgICBWZXJzaW9uOiAnMjAxMi0xMC0xNycsXG4gICAgICB9LFxuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCdjYW4gYWRkIGFuIGVkZ2UgbGFtYmRhcyB0byBhZGRpdGlvbmFsIGJlaGF2aW9ycycsICgpID0+IHtcbiAgICBuZXcgRGlzdHJpYnV0aW9uKHN0YWNrLCAnTXlEaXN0Jywge1xuICAgICAgZGVmYXVsdEJlaGF2aW9yOiB7IG9yaWdpbiB9LFxuICAgICAgYWRkaXRpb25hbEJlaGF2aW9yczoge1xuICAgICAgICAnaW1hZ2VzLyonOiB7XG4gICAgICAgICAgb3JpZ2luLFxuICAgICAgICAgIGVkZ2VMYW1iZGFzOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIGZ1bmN0aW9uVmVyc2lvbjogbGFtYmRhRnVuY3Rpb24uY3VycmVudFZlcnNpb24sXG4gICAgICAgICAgICAgIGV2ZW50VHlwZTogTGFtYmRhRWRnZUV2ZW50VHlwZS5WSUVXRVJfUkVRVUVTVCxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpDbG91ZEZyb250OjpEaXN0cmlidXRpb24nLCB7XG4gICAgICBEaXN0cmlidXRpb25Db25maWc6IHtcbiAgICAgICAgQ2FjaGVCZWhhdmlvcnM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBQYXRoUGF0dGVybjogJ2ltYWdlcy8qJyxcbiAgICAgICAgICAgIExhbWJkYUZ1bmN0aW9uQXNzb2NpYXRpb25zOiBbXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBFdmVudFR5cGU6ICd2aWV3ZXItcmVxdWVzdCcsXG4gICAgICAgICAgICAgICAgTGFtYmRhRnVuY3Rpb25BUk46IHtcbiAgICAgICAgICAgICAgICAgIFJlZjogJ0Z1bmN0aW9uQ3VycmVudFZlcnNpb240RTJCMjI2MTQ3N2E1YWU4MDU5YmJhYTc4MTNmNzUyMjkyYzBmNjVlJyxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnZmFpbHMgY3JlYXRpb24gd2hlbiBhdHRlbXB0aW5nIHRvIGFkZCB0aGUgJExBVEVTVCBmdW5jdGlvbiB2ZXJzaW9uIGFzIGFuIGVkZ2UgTGFtYmRhIHRvIHRoZSBkZWZhdWx0IGJlaGF2aW9yJywgKCkgPT4ge1xuICAgIGV4cGVjdCgoKSA9PiB7XG4gICAgICBuZXcgRGlzdHJpYnV0aW9uKHN0YWNrLCAnTXlEaXN0Jywge1xuICAgICAgICBkZWZhdWx0QmVoYXZpb3I6IHtcbiAgICAgICAgICBvcmlnaW4sXG4gICAgICAgICAgZWRnZUxhbWJkYXM6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgZnVuY3Rpb25WZXJzaW9uOiBsYW1iZGFGdW5jdGlvbi5sYXRlc3RWZXJzaW9uLFxuICAgICAgICAgICAgICBldmVudFR5cGU6IExhbWJkYUVkZ2VFdmVudFR5cGUuT1JJR0lOX1JFU1BPTlNFLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfSkudG9UaHJvdygvXFwkTEFURVNUIGZ1bmN0aW9uIHZlcnNpb24gY2Fubm90IGJlIHVzZWQgZm9yIExhbWJkYUBFZGdlLyk7XG4gIH0pO1xuXG4gIHRlc3QoJ3dpdGggcmVtb3ZhYmxlIGVudiB2YXJzJywgKCkgPT4ge1xuICAgIGNvbnN0IGVudkxhbWJkYUZ1bmN0aW9uID0gbmV3IGxhbWJkYS5GdW5jdGlvbihzdGFjaywgJ0VudkZ1bmN0aW9uJywge1xuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTLFxuICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUlubGluZSgnd2hhdGV2ZXJ3aXRoZW52JyksXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgfSk7XG4gICAgZW52TGFtYmRhRnVuY3Rpb24uYWRkRW52aXJvbm1lbnQoJ0tFWScsICd2YWx1ZScsIHsgcmVtb3ZlSW5FZGdlOiB0cnVlIH0pO1xuXG4gICAgbmV3IERpc3RyaWJ1dGlvbihzdGFjaywgJ015RGlzdCcsIHtcbiAgICAgIGRlZmF1bHRCZWhhdmlvcjoge1xuICAgICAgICBvcmlnaW4sXG4gICAgICAgIGVkZ2VMYW1iZGFzOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgZnVuY3Rpb25WZXJzaW9uOiBlbnZMYW1iZGFGdW5jdGlvbi5jdXJyZW50VmVyc2lvbixcbiAgICAgICAgICAgIGV2ZW50VHlwZTogTGFtYmRhRWRnZUV2ZW50VHlwZS5PUklHSU5fUkVRVUVTVCxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6TGFtYmRhOjpGdW5jdGlvbicsIHtcbiAgICAgIEVudmlyb25tZW50OiBBQlNFTlQsXG4gICAgICBDb2RlOiB7XG4gICAgICAgIFppcEZpbGU6ICd3aGF0ZXZlcndpdGhlbnYnLFxuICAgICAgfSxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnd2l0aCBpbmNvbXBhdGlibGUgZW52IHZhcnMnLCAoKSA9PiB7XG4gICAgY29uc3QgZW52TGFtYmRhRnVuY3Rpb24gPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnRW52RnVuY3Rpb24nLCB7XG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlMsXG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tSW5saW5lKCd3aGF0ZXZlcndpdGhlbnYnKSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJyxcbiAgICAgIGVudmlyb25tZW50OiB7XG4gICAgICAgIEtFWTogJ3ZhbHVlJyxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBuZXcgRGlzdHJpYnV0aW9uKHN0YWNrLCAnTXlEaXN0Jywge1xuICAgICAgZGVmYXVsdEJlaGF2aW9yOiB7XG4gICAgICAgIG9yaWdpbixcbiAgICAgICAgZWRnZUxhbWJkYXM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBmdW5jdGlvblZlcnNpb246IGVudkxhbWJkYUZ1bmN0aW9uLmN1cnJlbnRWZXJzaW9uLFxuICAgICAgICAgICAgZXZlbnRUeXBlOiBMYW1iZGFFZGdlRXZlbnRUeXBlLk9SSUdJTl9SRVFVRVNULFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgZXhwZWN0KCgpID0+IGFwcC5zeW50aCgpKS50b1Rocm93KC9LRVkvKTtcbiAgfSk7XG5cbiAgdGVzdCgnd2l0aCBzaW5nbGV0b24gZnVuY3Rpb24nLCAoKSA9PiB7XG4gICAgY29uc3Qgc2luZ2xldG9uID0gbmV3IGxhbWJkYS5TaW5nbGV0b25GdW5jdGlvbihzdGFjaywgJ1NpbmdsZXRvbicsIHtcbiAgICAgIHV1aWQ6ICdzaW5nbGV0b24tZm9yLWNsb3VkZnJvbnQnLFxuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzEyX1gsXG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tSW5saW5lKCdjb2RlJyksXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgfSk7XG5cbiAgICBuZXcgRGlzdHJpYnV0aW9uKHN0YWNrLCAnTXlEaXN0Jywge1xuICAgICAgZGVmYXVsdEJlaGF2aW9yOiB7XG4gICAgICAgIG9yaWdpbixcbiAgICAgICAgZWRnZUxhbWJkYXM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBmdW5jdGlvblZlcnNpb246IHNpbmdsZXRvbi5jdXJyZW50VmVyc2lvbixcbiAgICAgICAgICAgIGV2ZW50VHlwZTogTGFtYmRhRWRnZUV2ZW50VHlwZS5PUklHSU5fUkVRVUVTVCxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNsb3VkRnJvbnQ6OkRpc3RyaWJ1dGlvbicsIHtcbiAgICAgIERpc3RyaWJ1dGlvbkNvbmZpZzoge1xuICAgICAgICBEZWZhdWx0Q2FjaGVCZWhhdmlvcjoge1xuICAgICAgICAgIExhbWJkYUZ1bmN0aW9uQXNzb2NpYXRpb25zOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIEV2ZW50VHlwZTogJ29yaWdpbi1yZXF1ZXN0JyxcbiAgICAgICAgICAgICAgTGFtYmRhRnVuY3Rpb25BUk46IHtcbiAgICAgICAgICAgICAgICBSZWY6ICdTaW5nbGV0b25MYW1iZGFzaW5nbGV0b25mb3JjbG91ZGZyb250Q3VycmVudFZlcnNpb24wMDc4NDA2MzQ4YTA5NjJhNTI0NDhhMjAwY2QwZGJjMGUyMmVkYjJhJyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH0pO1xufSk7XG5cbnRlc3QoJ3ByaWNlIGNsYXNzIGlzIGluY2x1ZGVkIGlmIHByb3ZpZGVkJywgKCkgPT4ge1xuICBjb25zdCBvcmlnaW4gPSBkZWZhdWx0T3JpZ2luKCk7XG4gIG5ldyBEaXN0cmlidXRpb24oc3RhY2ssICdEaXN0Jywge1xuICAgIGRlZmF1bHRCZWhhdmlvcjogeyBvcmlnaW4gfSxcbiAgICBwcmljZUNsYXNzOiBQcmljZUNsYXNzLlBSSUNFX0NMQVNTXzIwMCxcbiAgfSk7XG5cbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6Q2xvdWRGcm9udDo6RGlzdHJpYnV0aW9uJywge1xuICAgIERpc3RyaWJ1dGlvbkNvbmZpZzoge1xuICAgICAgUHJpY2VDbGFzczogJ1ByaWNlQ2xhc3NfMjAwJyxcbiAgICB9LFxuICB9KTtcbn0pO1xuXG50ZXN0KCdlc2NhcGUgaGF0Y2hlcyBhcmUgc3VwcG9ydGVkJywgKCkgPT4ge1xuICBjb25zdCBkaXN0ID0gbmV3IERpc3RyaWJ1dGlvbihzdGFjaywgJ0Rpc3QnLCB7XG4gICAgZGVmYXVsdEJlaGF2aW9yOiB7IG9yaWdpbjogZGVmYXVsdE9yaWdpbiB9LFxuICB9KTtcbiAgY29uc3QgY2ZuRGlzdCA9IGRpc3Qubm9kZS5kZWZhdWx0Q2hpbGQgYXMgQ2ZuRGlzdHJpYnV0aW9uO1xuICBjZm5EaXN0LmFkZFByb3BlcnR5T3ZlcnJpZGUoJ0Rpc3RyaWJ1dGlvbkNvbmZpZy5EZWZhdWx0Q2FjaGVCZWhhdmlvci5Gb3J3YXJkZWRWYWx1ZXMuSGVhZGVycycsIFsnKiddKTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpDbG91ZEZyb250OjpEaXN0cmlidXRpb24nLCB7XG4gICAgRGlzdHJpYnV0aW9uQ29uZmlnOiB7XG4gICAgICBEZWZhdWx0Q2FjaGVCZWhhdmlvcjoge1xuICAgICAgICBGb3J3YXJkZWRWYWx1ZXM6IHtcbiAgICAgICAgICBIZWFkZXJzOiBbJyonXSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSxcbiAgfSk7XG59KTtcbiJdfQ==