from rest_framework import serializers

from huscy.projects.models import Membership, Project, ResearchUnit
from huscy.projects.services import create_membership, create_project, update_membership


class ResearchUnitSerializer(serializers.ModelSerializer):
    class Meta:
        model = ResearchUnit
        fields = (
            'id',
            'code',
            'name',
            'principal_investigator',
        )


class ProjectSerializer(serializers.ModelSerializer):
    creator = serializers.HiddenField(default=serializers.CurrentUserDefault())
    participating = serializers.SerializerMethodField()
    principal_investigator_name = serializers.SerializerMethodField()
    research_unit_name = serializers.SerializerMethodField()

    # default=None is used as required=False does not work together with the unique_together
    # constraint, however this defaut=None is a workaround, see:
    # https://github.com/encode/django-rest-framework/issues/4456
    local_id = serializers.IntegerField(min_value=1, default=None)

    class Meta:
        model = Project
        fields = (
            'id',
            'creator',
            'description',
            'local_id',
            'local_id_name',
            'participating',
            'principal_investigator',
            'principal_investigator_name',
            'research_unit',
            'research_unit_name',
            'title',
        )

    def get_participating(self, project):
        user = self.context['request'].user
        return (project.principal_investigator == user or
                user.pk in project.membership_set.values_list('user', flat=True))

    def get_principal_investigator_name(self, project):
        return project.principal_investigator.get_full_name()

    def get_research_unit_name(self, project):
        return project.research_unit.name

    def create(self, validated_data):
        return create_project(**validated_data)


class MembershipSerializer(serializers.ModelSerializer):
    has_write_permission = serializers.BooleanField()
    username = serializers.SerializerMethodField()

    class Meta:
        model = Membership
        fields = (
            'id',
            'has_write_permission',
            'is_coordinator',
            'project',
            'user',
            'username',
        )
        read_only_fields = 'project',

    def get_username(self, membership):
        return membership.user.get_full_name()

    def create(self, validated_data):
        return create_membership(**validated_data)

    def update(self, membership, validated_data):
        return update_membership(membership, validated_data.get('is_coordinator'),
                                 validated_data.get('has_write_permission'))
