# Convert SHACL Model to code bindings

[![OpenSSF Best Practices](https://www.bestpractices.dev/projects/9999/badge)](https://www.bestpractices.dev/projects/9999)
[![OpenSSF Scorecard](https://api.scorecard.dev/projects/github.com/JPEWdev/shacl2code/badge)](https://scorecard.dev/viewer/?uri=github.com/JPEWdev/shacl2code)
[![Coverage Report](https://raw.githubusercontent.com/JPEWdev/shacl2code/python-coverage-comment-action-data/badge.svg)](https://htmlpreview.github.io/?https://github.com/JPEWdev/shacl2code/blob/python-coverage-comment-action-data/htmlcov/index.html)

This tool can be used to convert a SHACL model into various code bindings

## Installation

`shacl2code` can be installed using pip:

```shell
python3 -m pip install shacl2code
```

## Usage

`shacl2code` can generate bindings from either a local file:

```shell
shacl2code generate -i model.jsonld python -o out.py
```

Or from a URL:

```shell
shacl2code generate -i https://example.com/rdf/model.jsonld python -o out.py
```

Or from stdin:

```shell
cat model.jsonld | shacl2code generate -i - python -o - > out.py
```

For more information, run:

```shell
shacl2code --help
```

The available language bindings can be viewed by running:

```shell
shacl2code list
```

## Developing

Developing on `shacl2code` is best done using a virtual environment. You can
configure one and install shacl2code in editable mode with all necessary
development dependencies by running:

```shell
python3 -m venv .venv
. .venv/bin/activate
pip install -e ".[dev]"
```

## Testing

`shacl2code` has a test suite written in [pytest][pytest]. To run it, setup a
virtual environment as shown above, then run:

```shell
pytest
```

In addition to the test results, a test coverage report will also be generated
using [pytest-cov][pytest-cov]

## Custom Annotations

`shacl2code` supports a number of custom annotations that can be specified in a
SHACL model to give hints about the generated code. All of these annotations
live in the `https://jpewdev.github.io/shacl2code/schema#` namespace, and
commonly are given the `sh-to-code` prefix to make it easier to reference them.
For example, in Turtle one would add the prefix mapping:

```ttl
@prefix sh-to-code: <https://jpewdev.github.io/shacl2code/schema#> .
```

### ID Property Name

The `idPropertyName` annotation allows a class to specify what the name of the
"property" that specifies the RDF subject of an object is for serializations
that support it. For example, in JSON-LD, the `@id` property indicates the
subject in RDF. If you wanted to alias the `@id` property to another name, the
`idPropertyName` annotation will let you do this. For example, the following
turtle will use `MyId` instead of `@id` when writing JSON-LD bindings:

```ttl
<MyClass> a owl:Class, sh:NodeShape ;
    sh-to-code:idPropertyName "MyId"
    .
```

When doing this, the class would then look like this in JSON-LD:

```json
{
    "@type": "MyClass",
    "MyId": "http://example.com/id"
}
```

The `idPropertyName` annotation is inherited by derived classes, so for example
any class that derived from `MyClass` would also use `MyId` as the subject
property.

**Note:** This only specifies what the name of the field should be in generated
bindings and has no bearing on how an RDF parser would interpret the property.
In order to still be parsed by RDF, you would also need context file that maps
`MyId` to `@id`, for example:

```json
{
    "@context": {
        "MyId": "@id"
    }
}
```

`shacl2code` doesn't do this for you, nor does it validate that you have done
it.

### Extensible Classes

Most bindings generated from `shacl2code` are "closed" in that they do not
allow extra properties to be added to object outside of what is specified in
model. This ensures that field name typos and other unintended properties are
not added to an object. However, in some cases a class may be specifically
intended to be extended such that arbitrary fields can be added to it, which
can be done using the `isExtensible` property. This is a boolean property that
indicates if a class can be extended, and defaults to `false`. For example, the
following turtle will declare a class as extensible:

```ttl
<MyClass> a owl:Class, sh:NodeShape ;
    sh-to-code:isExtensible true
    .
```

The `isExtensible` property is _not_ inherited by derived classes, meaning it
is possible to have a class derived from `MyClass` which is itself not
extensible.

The mechanism for dealing with extensible classes will vary between the
different bindings, but in general it means that they will not be very picky
about object types and properties in any location where an extensible class is
allowed.

**Note**: You may want to be careful about where and how many extensible
classes are allowed in your model. If there are too many and they are allowed
anywhere, it may mean that typos in object types (e.g. `@type` in JSON-LD) are
not caught by validation as they will have to be assumed to be a derived class
from an extensible type.

### Abstract Classes

By default, classes generated by `shacl2code` are all instantiable (i.e. they
can be created). In some instances, it may be desirable to declare a class as
abstract (meaning that it cannot be instantiated, but non-abstract derived
classes can). There are several ways of marking a class as abstract listed
below, in order of preference (with the most preferred being first).

#### SHACL Validated Abstract Classes

A class can be prevented from being directly instantiated using SHACL by adding
a constraint on the shape that it cannot be of its own type. This can be done
with the following turtle:

```ttl
<MyClass> a owl:Class, sh:NodeShape ;
    # SHACL to prevent a class from being instantiated as this exact type
    sh:property [
        sh:path rdf:type ;
        sh:not [ sh:hasValue <MyClass> ]
    ] .
```

`shacl2code` will detect this pattern and generate abstract bindings for
`MyClass`.

This method is most preferred, since it is enforced by SHACL and not just
`shacl2code` bindings

#### shacl2code Annotation

`shacl2code` has a custom annotation that can be used to mark a class as
abstract. This can be done with the boolean `isAbstract` property. For
example, the following turtle will declare a class as abstract:

```ttl
<MyClass> a owl:Class, sh:NodeShape ;
    sh-to-code:isAbstract true
    .
```

The `isAbstract` property is _not_ inherited by derived classes, so any derived
classes are automatically concrete unless they indicate otherwise.

#### SPDX Abstract Class Parent

It is also possible to define a class as abstract by declaring it to be of
type: `http://spdx.invalid./AbstractClass`, but this is not preferred.

[pytest]: https://www.pytest.org
[pytest-cov]: https://pytest-cov.readthedocs.io/en/latest/
