"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Route = exports.RouteType = void 0;
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
/**
 * (experimental) Type of route.
 *
 * @experimental
 */
var RouteType;
(function (RouteType) {
    RouteType["HTTP"] = "http";
    RouteType["TCP"] = "tcp";
})(RouteType = exports.RouteType || (exports.RouteType = {}));
/**
 * (experimental) Route represents a new or existing route attached to a VirtualRouter and Mesh.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/routes.html
 * @experimental
 */
class Route extends cdk.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.routeName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        this.weightedTargets = new Array();
        this.virtualRouter = props.virtualRouter;
        const routeType = props.routeType !== undefined ? props.routeType :
            props.prefix !== undefined ? RouteType.HTTP :
                RouteType.TCP;
        if (routeType === RouteType.HTTP) {
            this.httpRoute = this.renderHttpRoute(props);
        }
        else {
            this.tcpRoute = this.renderTcpRoute(props);
        }
        const route = new appmesh_generated_1.CfnRoute(this, 'Resource', {
            routeName: this.physicalName,
            meshName: this.virtualRouter.mesh.meshName,
            virtualRouterName: this.virtualRouter.virtualRouterName,
            spec: {
                tcpRoute: this.tcpRoute,
                httpRoute: this.httpRoute,
            },
        });
        this.routeName = this.getResourceNameAttribute(route.attrRouteName);
        this.routeArn = this.getResourceArnAttribute(route.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualRouter/${props.virtualRouter.virtualRouterName}/route`,
            resourceName: this.physicalName,
        });
    }
    /**
     * (experimental) Import an existing route given an ARN.
     *
     * @experimental
     */
    static fromRouteArn(scope, id, routeArn) {
        return new ImportedRoute(scope, id, { routeArn });
    }
    /**
     * (experimental) Import an existing route given its name.
     *
     * @experimental
     */
    static fromRouteName(scope, id, meshName, virtualRouterName, routeName) {
        return new ImportedRoute(scope, id, { meshName, virtualRouterName, routeName });
    }
    /**
     * Utility method to add weighted route targets to an existing route
     */
    renderWeightedTargets(props) {
        for (const t of props) {
            this.weightedTargets.push({
                virtualNode: t.virtualNode.virtualNodeName,
                weight: t.weight || 1,
            });
        }
        return this.weightedTargets;
    }
    renderHttpRoute(props) {
        return {
            match: {
                prefix: props.prefix || '/',
            },
            action: {
                weightedTargets: this.renderWeightedTargets(props.routeTargets),
            },
        };
    }
    renderTcpRoute(props) {
        return {
            action: {
                weightedTargets: this.renderWeightedTargets(props.routeTargets),
            },
        };
    }
}
exports.Route = Route;
/**
 * Represents and imported IRoute
 */
class ImportedRoute extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.routeArn) {
            this.routeArn = props.routeArn;
            this.routeName = cdk.Fn.select(4, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.routeArn).resourceName));
        }
        else if (props.routeName && props.meshName && props.virtualRouterName) {
            this.routeName = props.routeName;
            this.routeArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualRouter/${props.virtualRouterName}/route`,
                resourceName: this.routeName,
            });
        }
        else {
            throw new Error('Need either arn or three names');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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