import re

from bs4 import BeautifulSoup, NavigableString

from .base_xblock import BaseXblock
from ..utils import jinja, remove_autogenerated_tags, get_back_jumps


class Discussion(BaseXblock):
    def __init__(
        self, xblock_json, relative_path, root_url, xblock_id, descendants, scraper
    ):
        super().__init__(
            xblock_json, relative_path, root_url, xblock_id, descendants, scraper
        )

        # extra vars
        self.data = []
        self.category_title = ""
        self.is_video = False
        self.pre_discussion_content = []
        self.post_discussion_content = []
        self.discussion_header = []

    def get_extra_content(self, soup):
        discussion_encountered = False
        for child_node in soup.find("div", attrs={"class": "xblock"}).contents:
            if isinstance(child_node, NavigableString):
                continue
            if "discussion-module" in child_node.attrs.get("class", ""):
                discussion_encountered = True
                for element in child_node.contents:
                    if isinstance(element, NavigableString):
                        continue
                    if (
                        "discussion-show" in element.attrs.get("class", "")
                        and element.name == "button"
                    ):
                        continue
                    if "inline-discussion" not in element.attrs.get("class", ""):
                        self.discussion_header.append(
                            remove_autogenerated_tags(
                                self.scraper.html_processor.dl_dependencies_and_fix_links(
                                    content=str(element),
                                    output_path=self.scraper.instance_assets_dir,
                                    path_from_html=get_back_jumps(5)
                                    + "instance_assets",
                                    root_from_html=get_back_jumps(5),
                                )
                            )
                        )
            elif not discussion_encountered:
                self.pre_discussion_content.append(
                    remove_autogenerated_tags(
                        self.scraper.html_processor.dl_dependencies_and_fix_links(
                            content=str(child_node),
                            output_path=self.scraper.instance_assets_dir,
                            path_from_html=get_back_jumps(5) + "instance_assets",
                            root_from_html=get_back_jumps(5),
                        )
                    )
                )
            else:
                self.post_discussion_content.append(
                    remove_autogenerated_tags(
                        self.scraper.html_processor.dl_dependencies_and_fix_links(
                            content=str(child_node),
                            output_path=self.scraper.instance_assets_dir,
                            path_from_html=get_back_jumps(5) + "instance_assets",
                            root_from_html=get_back_jumps(5),
                        )
                    )
                )

    def download(self, instance_connection):
        if self.scraper.forum:
            content = instance_connection.get_page(self.xblock_json["student_view_url"])
            if not content:
                return
            soup = BeautifulSoup(content, "lxml")

            self.get_extra_content(soup)

            discussion_block = soup.find(
                re.compile(r".*"), {"data-discussion-id": re.compile(r".*")}
            )
            if discussion_block:
                discussion_id = discussion_block["data-discussion-id"]
                for thread in self.scraper.forum.threads:
                    if thread["commentable_id"] == discussion_id:
                        self.data.append(thread)
                if len(self.data) != 0:
                    self.category_title = self.scraper.forum.categories[discussion_id]

    def render(self):
        if self.category_title:
            # render the discussion
            return jinja(
                None,
                "discussion.html",
                False,
                category_title=self.category_title,
                threads=self.data,
                discussion=self,
                staff_user=self.scraper.forum.staff_user,
                rooturl=get_back_jumps(5),
                pre_discussion_content=self.pre_discussion_content,
                post_discussion_content=self.post_discussion_content,
                discussion_header=self.discussion_header,
            )
        if not self.scraper.forum:
            # render nothing in no forum mode
            return ""
        # render an error message if forum mode and current discussion not supported
        return "The discussion here is not supported in offline mode"
