import pydantic
import pytest

from canonical import ResourceName


class Model(pydantic.BaseModel):
    resource: ResourceName


def test_init_valid_resource_id():
    rn = ResourceName('//mock.unimatrixapis.com/books/1')
    assert str(rn.service) == 'mock.unimatrixapis.com'
    assert rn.relname == 'books/1'


@pytest.mark.parametrize("v", [
    "//mock.unimatrixapis.com/",
    "//mock.unimatrixapis.com",
])
def test_relname_is_required(v: str):
    with pytest.raises((ValueError, pydantic.ValidationError)):
        adapter = pydantic.TypeAdapter(ResourceName)
        adapter.validate_python(v)
        ResourceName(v)


@pytest.mark.parametrize("v", [
    "/mock.unimatrixapis.com/books/1",
    "mock.unimatrixapis.com/books/1",
])
def test_preceding_slashes_required(v: str):
    with pytest.raises(ValueError):
        ResourceName(v)


def test_model_valid_resource_id():
    obj = Model.model_validate({'resource': '//mock.unimatrixapis.com/books/1'})
    assert isinstance(obj.resource, ResourceName)


@pytest.mark.parametrize("v", [
    "/mock.unimatrixapis.com/books/1",
    "mock.unimatrixapis.com/books/1",
    "//mock.unimatrixapis.com/",
    "//mock.unimatrixapis.com",
])
def test_model_invalid(v: str):
    with pytest.raises(ValueError):
        Model.model_validate({'resource': v})


def test_id_int():
    name = ResourceName('//foo.bar.baz/resource/1')
    assert name.id == '1'